package com.xdja.hr.service.compute.judgeTool;

import com.xdja.hr.domain.integration.DayComputeRecord;
import com.xdja.hr.domain.integration.DayStateRecord;
import com.xdja.hr.domain.original.Employee;
import com.xdja.hr.pojo.ValidFingerprintData;
import com.xdja.hr.service.compute.IJudge;
import com.xdja.hr.utils.DayState;
import org.joda.time.DateTime;
import org.joda.time.LocalTime;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * User: landy
 * Date: 15/8/19
 * Time: 下午11:41
 */
public class SummerRestDayJudge implements IJudge {
    // 8:45~10:30
    final static int MORNING_BEGIN_MILLIS = new LocalTime(8, 45).getMillisOfDay();
    final static int MORNING_END_MILLIS = new LocalTime(12, 0).getMillisOfDay();

    final static int AFTERNOON_BEGIN_MILLIS = new LocalTime(13, 30).getMillisOfDay();
    final static int AFTERNOON_END_MILLIS = LocalTime.MIDNIGHT.millisOfDay().getMaximumValue();

    private static final LocalTime EAT_MORNING_TIME = new LocalTime(9, 0);
    private static final LocalTime EAT_AFTERNOON_TIME = new LocalTime(19, 00);
    private static final LocalTime EAT_MIDDAY_BEGIN = new LocalTime(12, 00);
    private static final LocalTime EAT_MIDDAY_OVER = new LocalTime(13, 30);

    @Override
    public DayComputeRecord judge(Employee employee, ValidFingerprintData validData) {
        //非工作日
        boolean isWorkDay = false;
        double workOvertime = 0.0;
        //休息日没有地铁补助
        double assistSubway = 0.0;
        int assistMorning = 0;
        int assistMidday = 0;
        int assistAfternoon = 0;
        List<DayState> dayStates = new ArrayList<DayState>();

        if (hasTwoTime(validData)) {
            dayStates.add(DayState.Work);
            /**
             * 9~12点，13:30~，0.5小时一个档
             * 拿实际工作时间，与这两个时间段取交集分钟数。然后将总的分钟数按照半小时一档算出小时数
             */
            LocalTime begin = getWorkBeginTime(validData);
            LocalTime end = getWorkEndTime(validData);
            int amMinutes = getMorningWorkMinutes(begin.getMillisOfDay(), end.getMillisOfDay());
            int pmMinutes = getAfternoonWorkMinutes(begin.getMillisOfDay(), end.getMillisOfDay());
            workOvertime = ((amMinutes + pmMinutes) / 30) * 0.5;
            assistMorning = JudgeUtils.assistMorning(validData, EAT_MORNING_TIME);
            assistMidday = JudgeUtils.assistMidday(validData, EAT_MIDDAY_BEGIN, EAT_MIDDAY_OVER);
            assistAfternoon = JudgeUtils.assistAfternoon(validData, EAT_AFTERNOON_TIME);
        } else {
            dayStates.add(DayState.Rest);
        }

        DayComputeRecord rst = new DayComputeRecord();
        rst.setIsWorkDay(isWorkDay);
        rst.setAssistSubway(assistSubway);
        rst.setAssistEatMorning(assistMorning);
        rst.setAssistEatMidday(assistMidday);
        rst.setAssistEatAfternoon(assistAfternoon);
        rst.setCreateDate(validData.getCreateDate());
        rst.setWorkOvertime(workOvertime);
        rst.setEmployee(employee);
        for (DayState one : dayStates) {
            DayStateRecord state = new DayStateRecord();
            state.setDayComputeRecord(rst);
            state.setDayState(one);
            rst.getDayStateRecords().add(state);
        }
        return rst;
    }

    private boolean hasTwoTime(ValidFingerprintData record) {
        int flag = 0;
        if (record.getMorningTime() != null) {
            flag++;
        }
        if (record.getAfternoonTime() != null) {
            flag++;
        }
        if (record.getMiddayTime1() != null) {
            flag++;
        }
        if (record.getMiddayTime2() != null) {
            flag++;
        }
        return flag >= 2;
    }


    private int getAfternoonWorkMinutes(int begin, int end) {
        if (AFTERNOON_BEGIN_MILLIS >= end || begin >= AFTERNOON_END_MILLIS) {
            return 0;
        } else {
            int left = Math.max(begin, AFTERNOON_BEGIN_MILLIS);
            int right = Math.min(end, AFTERNOON_END_MILLIS);
            return (right - left) / 60000;
        }
    }


    private int getMorningWorkMinutes(int begin, int end) {
        if (MORNING_BEGIN_MILLIS >= end || begin >= MORNING_END_MILLIS) {
            return 0;
        } else {
            int left = Math.max(begin, MORNING_BEGIN_MILLIS);
            int right = Math.min(end, MORNING_END_MILLIS);
            return (right - left) / 60000;
        }
    }

    private LocalTime getWorkEndTime(ValidFingerprintData record) {
        Date time = null;
        if (record.getAfternoonTime() != null) {
            time = record.getAfternoonTime();
        } else if (record.getMiddayTime2() != null) {
            time = record.getMiddayTime2();
        } else if (record.getMiddayTime1() != null) {
            time = record.getMiddayTime1();
        } else {
            throw new IllegalStateException("全天只有一条上班指纹记录！");
        }
        return new DateTime(time).toLocalTime();
    }

    private LocalTime getWorkBeginTime(ValidFingerprintData record) {
        Date time = null;
        if (record.getMorningTime() != null) {
            time = record.getMorningTime();
        } else if (record.getMiddayTime1() != null) {
            time = record.getMiddayTime1();
        } else if (record.getMiddayTime2() != null) {
            time = record.getMiddayTime2();
        } else {
            throw new IllegalStateException("全天只有一条下班指纹记录！");
        }
        return new DateTime(time).toLocalTime();
    }
}
