package com.xdja.hr.service.compute.judgeTool;

import com.xdja.hr.domain.integration.DayComputeRecord;
import com.xdja.hr.domain.integration.DayStateRecord;
import com.xdja.hr.domain.original.Employee;
import com.xdja.hr.pojo.ValidFingerprintData;
import com.xdja.hr.service.compute.IJudge;
import com.xdja.hr.utils.DayState;
import org.joda.time.DateTime;
import org.joda.time.LocalTime;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 * User: landy
 * Date: 15/8/19
 * Time: 下午11:41
 */
public class SummerWorkDayJudge implements IJudge {
    // 8:45~10:30
    /**
     * 早上正常上班时间
     */
    private static final LocalTime WORK_MORNING_TIME = new LocalTime(8, 45);
    private static final LocalTime EAT_MORNING_TIME = new LocalTime(9, 0);
    /**
     * 下午正常下班时间
     */
    private static final LocalTime WORK_AFTERNOON_TIME = new LocalTime(17, 45);
    private static final LocalTime EAT_AFTERNOON_TIME = new LocalTime(19, 00);
    /**
     * 在此之后才算加班，记2小时
     */
    private static final LocalTime WORK_OVERTIME_TIME_BEGIN = new LocalTime(19, 30);
    /**
     * 在此之前，记2小时，之后记3小时
     */
    private static final LocalTime WORK_OVERTIME_TIME_3 = new LocalTime(20, 00);
    /**
     * 在此之前记3小时，之后每半小时记0.5
     */
    private static final LocalTime WORK_OVERTIME_TIME_AFTER = new LocalTime(20, 30);
    private static final LocalTime EAT_MIDDAY_BEGIN = new LocalTime(12, 00);
    private static final LocalTime EAT_MIDDAY_OVER = new LocalTime(13, 30);

    @Override
    public DayComputeRecord judge(Employee employee, ValidFingerprintData validData) {
        //非工作日
        boolean isWorkDay = true;
        double workOvertime = 0.0;
        double assistSubway = 0.0;
        int assistMorning = 0;
        int assistMidday = 0;
        int assistAfternoon = 0;
        List<DayState> dayStates = new ArrayList<DayState>();

        if (hasLeastTwoTime(validData)) {
            //至少有两条时间记录
            DayState morningState = getMoringState(validData);
            if (morningState != null) {
                dayStates.add(morningState);
            }
            DayState afternoonState = getAfternoonState(validData);
            if (afternoonState != null) {
                dayStates.add(afternoonState);
            }
        } else {
            dayStates.add(DayState.Absent);
        }

        if (dayStates.isEmpty()) {
            dayStates.add(DayState.Work);
        } else if (Collections.frequency(dayStates, DayState.HalfAbsent) == 2) {
            //同时存在两个半旷，变成一个全旷
            dayStates.clear();
            dayStates.add(DayState.Absent);
        }

        if (!dayStates.contains(DayState.Absent)) {
            //非全旷，才计算补贴和加班
            assistMorning = JudgeUtils.assistMorning(validData, EAT_MORNING_TIME);
            assistMidday = JudgeUtils.assistMidday(validData, EAT_MIDDAY_BEGIN, EAT_MIDDAY_OVER);
            assistAfternoon = JudgeUtils.assistAfternoon(validData, EAT_AFTERNOON_TIME);
            assistSubway = JudgeUtils.assistSubway(validData);

            workOvertime = calcWorkOvertime(validData.getAfternoonTime());
        }

        DayComputeRecord rst = new DayComputeRecord();
        rst.setIsWorkDay(isWorkDay);
        rst.setAssistSubway(assistSubway);
        rst.setAssistEatMorning(assistMorning);
        rst.setAssistEatMidday(assistMidday);
        rst.setAssistEatAfternoon(assistAfternoon);
        rst.setCreateDate(validData.getCreateDate());
        rst.setWorkOvertime(workOvertime);
        rst.setEmployee(employee);
        for (DayState one : dayStates) {
            DayStateRecord state = new DayStateRecord();
            state.setDayComputeRecord(rst);
            state.setDayState(one);
            rst.getDayStateRecords().add(state);
        }
        return rst;
    }

    private boolean hasLeastTwoTime(ValidFingerprintData validData) {
        return countOfNotNull(validData.getMorningTime(), validData.getAfternoonTime(), validData.getMiddayTime()) >= 2;
    }

    private int countOfNotNull(Date... times) {
        int flag = 0;
        for (Date one : times) {
            if (one != null) {
                flag++;
            }
        }
        return flag;
    }

    private DayState getAfternoonState(ValidFingerprintData validData) {
        Date time = validData.getAfternoonTime();
        if (time == null) {
            return DayState.HalfAbsent;
        }
        LocalTime afternoonTime = new DateTime(time).toLocalTime();
        if (afternoonTime.isBefore(WORK_AFTERNOON_TIME)) {
            //旷工
            if (afternoonTime.plusHours(1).isBefore(WORK_AFTERNOON_TIME)) {
                //早退超过1小时算旷工
                return (DayState.HalfAbsent);
            } else {
                return (DayState.LeaveEarly);
            }
        }
        return null;
    }

    private DayState getMoringState(ValidFingerprintData validData) {
        Date time = validData.getMorningTime();
        if (time == null) {
            return DayState.HalfAbsent;
        }
        LocalTime morningTime = new DateTime(time).toLocalTime();
        if (morningTime.isAfter(WORK_MORNING_TIME)) {
            //迟到
            if (morningTime.minusHours(2).isAfter(WORK_MORNING_TIME)) {
                //迟到超过两小时算旷工
                return DayState.HalfAbsent;
            } else {
                return DayState.ArriveLate;
            }
        }
        return null;
    }

    private double calcWorkOvertime(Date time) {
        if (time == null) {
            return 0.0;
        }
        LocalTime afternoonTime = new DateTime(time).toLocalTime();
        if (afternoonTime.isBefore(WORK_OVERTIME_TIME_BEGIN)) {
            return 0.0;
        } else if (afternoonTime.isBefore(WORK_OVERTIME_TIME_3)) {
            return 2.0;
        } else if (afternoonTime.isBefore(WORK_OVERTIME_TIME_AFTER)) {
            return 3.0;
        } else {
            int offsetMillis = afternoonTime.getMillisOfDay() - WORK_OVERTIME_TIME_AFTER.getMillisOfDay();
            int offsetMinute = offsetMillis / 60000;
            double hours = (offsetMinute / 30) * 0.5;
            return 3.0 + hours;
        }
    }
}
