package com.xdja.im.lib.album.ui.activity;

import android.Manifest;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.MotionEvent;

import com.xdja.im.common.imageloader.CustomImageLoader;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.lib.album.ImageDataSource;
import com.xdja.im.lib.album.ImagePickerManager;
import com.xdja.im.lib.album.bean.ImageItem;
import com.xdja.im.lib.album.bean.ImageThumbBean;
import com.xdja.im.lib.album.listener.OnItemClickListener;
import com.xdja.im.lib.album.listener.OnItemDataChangedListener;
import com.xdja.im.lib.album.ui.adapter.ImagePickerAdapter;
import com.xdja.im.lib.album.ui.presenter.ImagePickerPresenter;
import com.xdja.im.lib.album.ui.view.ViewImagePicker;
import com.xdja.im.lib.album.ui.view.vu.ImagePickerVu;
import com.xdja.im.lib.album.utils.DataTools;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package:com.xdja.im.lib.album
 * @Author:Lei
 * @Creation:2017-04-23 12:21
 * @Version 1.0
 * @Description:相册图片预览
 */
public class ImagePickerActivity extends ImBaseActivity<ImagePickerPresenter, ImagePickerVu>
        implements ImagePickerPresenter {

    private List<ImageItem> mImageItemList = new ArrayList<>();

    private ImagePickerAdapter mAdapter;

    /**
     * 标志位，如果正在发送图片中，则为true
     */
    private boolean bSendingImage = false;

    @NonNull
    @Override
    protected Class<? extends ImagePickerVu> getVuClass() {
        return ViewImagePicker.class;
    }

    @NonNull
    @Override
    protected ImagePickerPresenter getPresenter() {
        return this;
    }

    @Override
    protected void preBindView(@Nullable Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);
        //请求读取本地数据权限
        requestPermission();
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        syncGalley();
    }

    @Override
    protected void onResume() {
        super.onResume();
        //更新界面相关控件
        getVu().refreshSelectPictureIndicator();
        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }

    /**
     * 同步数据库信息
     */
    private void syncGalley() {
        Intent scanIntent = new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE);
        scanIntent.setData(Uri.fromFile(new File(Environment.getExternalStorageDirectory()
                .getAbsolutePath())));
        sendBroadcast(scanIntent);
    }

    /**
     * 请求权限
     */
    private void requestPermission() {
        if (PermissionUtils.hasSelfPermissions(this,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            loadData();
        } else {
            //请求权限
            PermissionUtils.requestPermissions(this,
                    new String[]{Manifest.permission.READ_EXTERNAL_STORAGE,
                                 Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_STORAGE);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode,
                                           @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        if (requestCode == Constant.REQUEST_PERMISSION_STORAGE) {
            if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                loadData();
            } else {
                showToast(R.string.im_uikit_storage_deny);
            }
        }
    }

    /**
     * 加载本地媒体数据数据
     */
    private void loadData() {
        Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<List<ImageItem>>>() {
                    @Override
                    public Observable<List<ImageItem>> call(String s) {
                        List<ImageItem> data = DataTools.loadImageData(ImagePickerActivity.this);
                        return Observable.just(data);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<ImageItem>>() {
                    @Override
                    public void onCompleted() {
                        getVu().setProgressBarVisible(false);
                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("load image failed! " + e.getMessage());
                        getVu().setEmptyLayoutVisible();
                        getVu().setProgressBarVisible(false);
                    }

                    @Override
                    public void onNext(List<ImageItem> imageItems) {

                        if (imageItems == null || imageItems.isEmpty()) {
                            getVu().setEmptyLayoutVisible();
                            return;
                        }

                        //数据缓存区
                        ImageDataSource.getInstance().addAll(imageItems);

                        mImageItemList.clear();
                        mImageItemList.addAll(imageItems);
                        mAdapter = buildAdapter(mImageItemList);
                        getVu().setAdapter(mAdapter);
                    }
                });
    }

    public ImagePickerAdapter buildAdapter(List<ImageItem> imageItemList) {
        return new ImagePickerAdapter(this, imageItemList) {
            @Override
            public OnItemClickListener getOnItemClickListener() {
                return onItemClickListener;
            }

            @Override
            public OnItemDataChangedListener getOnItemDataChangedListener() {
                return onItemDataChangedListener;
            }
        };
    }

    private OnItemClickListener onItemClickListener = new OnItemClickListener() {

        @Override
        public void onItemClick(int position, ImageItem item) {
            //启动预览界面
            Intent intent = new Intent();
            intent.setClass(ImagePickerActivity.this, ImagePreviewActivity.class);
            intent.putExtra(ImagePreviewActivity.TAG_TYPE, ImagePreviewActivity.TYPE_ALL);
            intent.putExtra(ImagePreviewActivity.TAG_POSITION, position);
            startActivityForResult(intent, 0);
        }
    };


    private OnItemDataChangedListener onItemDataChangedListener = new OnItemDataChangedListener() {
        @Override
        public void onDataChanged(int position, boolean isChecked) {
            //获取数据，添加进已选图片集合
            ImageItem imageItem = mAdapter.getItemData(position);
            if (imageItem == null) {
                return;
            }
            LogUtil.d("onDataChanged, isChecked:" + isChecked + ",imageItem:" + imageItem.toString());

            if (isChecked) {
                ImageDataSource.getInstance().add(imageItem);
            } else {
                ImageDataSource.getInstance().remove(imageItem);
            }

            //更新界面相关控件
            getVu().refreshSelectPictureIndicator();
        }
    };

    @Override
    protected void onDestroy() {
        super.onDestroy();

        bSendingImage = false;

        ImageDataSource.getInstance().clearCache();

        if (ImagePickerManager.getInstance().getImageLoader() != null) {
            ImagePickerManager.getInstance().getImageLoader().clearMemoryCache(this);
        }
        CustomImageLoader.getInstance().clearCache();
    }

    @Override
    public void sendImage(List<ImageItem> dataList) {
        if (dataList == null || dataList.isEmpty()) {
            return;
        }

        bSendingImage = true;
        getVu().setProgressBarVisible(true);

        Observable.from(dataList)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<ImageItem, Observable<ImageThumbBean>>() {
                    @Override
                    public Observable<ImageThumbBean> call(ImageItem imageItem) {
                        return DataTools.createThumbNails(imageItem);
                    }
                })
                .toList()
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<ImageThumbBean>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("Create thumbnail error, " + e.getMessage());

                        bSendingImage = false;
                        getVu().setProgressBarVisible(false);
                        CommonTool.showToast(R.string.im_uikit_send_image_failed);
                    }

                    @Override
                    public void onNext(List<ImageThumbBean> imageThumbBeen) {
                        setResult(imageThumbBeen);
                    }
                });
    }

    @Override
    public void preView() {
        //启动预览界面
        Intent intent = new Intent();
        intent.setClass(ImagePickerActivity.this, ImagePreviewActivity.class);
        intent.putExtra(ImagePreviewActivity.TAG_TYPE, ImagePreviewActivity.TYPE_SELECTED);
        startActivityForResult(intent, 0);
    }


    /**
     * 返回结果
     *
     * @param thumbBean 缩略图信息
     */
    private void setResult(List<ImageThumbBean> thumbBean) {
        Intent intent = new Intent();

        //数据绑定
        Bundle bundle = new Bundle();
        ArrayList<ImageThumbBean> bundleList = new ArrayList<>();
        bundleList.addAll(thumbBean);
        //发送图片数据
        bundle.putParcelableArrayList(Constant.TAG_PIC_SELECT, bundleList);
        //添加数据到Intent
        intent.putExtras(bundle);
        //返回结果
        setResult(Constant.RESULT_CODE_ALBUM, intent);
        finish();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (data != null) {
            setResult(Constant.RESULT_CODE_ALBUM, data);
            finish();
        }
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent ev) {
        if (bSendingImage) {
            return true;
        }
        return super.dispatchTouchEvent(ev);
    }
}
