package com.xdja.im.lib.audio.player;

import com.xdja.im.lib.audio.manager.VoicePlayState;

import java.util.ArrayList;
import java.util.List;

public class AudioListenerManager {

    private static AudioListenerManager INSTANCE;
    /**
     * 所有需要播放语音页面的实现的监听
     */
    private List<AudioListener> listeners;

    public static AudioListenerManager getInstance() {
        if (INSTANCE == null) {
            synchronized (AudioListenerManager.class) {
                if (INSTANCE == null) {
                    INSTANCE = new AudioListenerManager();
                }
            }
        }

        return INSTANCE;
    }

    private AudioListenerManager() {
        listeners = new ArrayList<>();
    }

    /**
     * 添加监听
     *
     * @param listener
     */
    public void addListener(AudioListener listener) {
        if (listeners != null) {
            listeners.add(listener);
        }
    }

    /**
     * 删除监听
     *
     * @param listener
     */
    public void removeListener(AudioListener listener) {
        if (listeners != null) {
            listeners.remove(listener);
        }
    }

    /**
     * 移除所有监听器
     */
    public void clearAll() {
        if (listeners != null) {
            listeners.clear();
        }
    }

    /**
     * 播放状态改变的回调
     *
     * @param filePath 正在播放的文件
     * @param state    更改的状态
     */
    public void playStateChange(final String filePath, final VoicePlayState state, final Object... args) {
        if (listeners != null && listeners.size() > 0) {
            for (final AudioListener listener : listeners) {
                dispatchPlayingState(listener, filePath, state, args);
            }
        }
    }

    /**
     * 根据不同的变化在不同的监听上做不同的操作
     *
     * @param listener 监听的listener
     * @param filePath 状态发生变化的路径
     * @param state    相应的状态
     * @param args
     */
    private void dispatchPlayingState(AudioListener listener, String filePath,
                                      VoicePlayState state, Object... args) {
        if (listener == null) {
            return;
        }

        switch (state) {
            case PLAYING:
                listener.playingCallback(filePath, args);
                break;
            case ERROR:
                listener.playingErrorCallback(filePath, args);
                break;
            case COMPLETION:
                listener.playingCompletionCallback(filePath, args);
                break;
            case STOP:
                listener.playingStopCallback(filePath, args);
                break;
        }
    }
}
