package com.xdja.im.lib.audio.player;

import android.media.AudioManager;
import android.media.MediaPlayer;
import android.text.TextUtils;

import com.xdja.im.lib.audio.manager.VoicePlayState;
import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

/**
 * @Package: com.xdja.im.lib.audio.player
 * @Author: xdjaxa
 * @Creation: 2017-04-24 11:58
 * @Version V1.0
 * @Description: 语音播放管理类
 */
public class AudioPlayerManager {

    private MediaPlayer player;
    private String playPath;
    private Object[] args;

    private static class SingletonInstance{
        private static final AudioPlayerManager INSTANCE = new AudioPlayerManager();
    }

    public static AudioPlayerManager getInstance() {
        return SingletonInstance.INSTANCE;
    }

    private AudioPlayerManager() {
    }

    /**
     * 播放对应的音频文件
     *
     * @param filePath 文件路径
     */
    public synchronized void startToPlay(String filePath, Object... args) {
        if (!checkNullForPath(filePath)) {
            return;
        }

        if (player != null) {
            if (player.isPlaying()) {
                stopToPlay(VoicePlayState.STOP, args);
            } else {
                player.release();
            }
            player = null;
        }

        FileInputStream fis = null;
        player = new MediaPlayer();
        player.setAudioStreamType(AudioManager.STREAM_MUSIC);
        player.setOnErrorListener(errorListener);
        player.setOnCompletionListener(completionListener);

        if (TextUtils.equals(playPath, filePath)) {
            stopToPlay(VoicePlayState.STOP, args);
            AudioFunctions.muteAudioFocus(false);
            return;
        }
        AudioFunctions.muteAudioFocus(true);
        try {
            File file = new File(filePath);
            fis = new FileInputStream(file);
            player.setDataSource(fis.getFD());
            player.prepare();
            player.start();
            playPath = filePath;
            this.args = args;
            AudioListenerManager.getInstance().playStateChange(
                    playPath, VoicePlayState.PLAYING, args);
        } catch (Exception e) {
            LogUtil.e("ERROR:Play voice failed. " + e.getMessage());
            stopToPlay(VoicePlayState.ERROR, args);
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }

    }

    /**
     * 对文件信息进行必要的校验
     *
     * @param filePath
     * @return
     */
    private boolean checkNullForPath(String filePath) {
        if (filePath == null || TextUtils.isEmpty(filePath)) {
            return false;
        }

        File file = new File(filePath);
        return file.exists();
    }

    /**
     * 停止播放
     *
     * @param state 播放状态,因为某种原因停止播放
     */
    public synchronized void stopToPlay(VoicePlayState state, Object... args) {
        if (player != null) {
            try {
                if (player.isPlaying()) {
                    player.stop();
                }
                player.release();
            } catch (IllegalStateException e) {
                e.printStackTrace();
            }
            player = null;
            AudioFunctions.muteAudioFocus(false);
        }

        playPath = null;
        this.args = null;
        AudioListenerManager.getInstance().playStateChange(
                playPath,
                state,
                args);
    }

    public void stopPlay() {
        stopToPlay(VoicePlayState.STOP, args);
    }

    /**
     * 释放MediaPlayer,调用方式在每个页面销毁的时候
     * 尽量释放掉不用的MediaPlayer
     */
    public void releasePlayer() {
        stopToPlay(VoicePlayState.STOP);
        if (player != null) {
            player.release();
            player = null;
        }
    }

    public String getCurrentPlayPath() {
        return playPath;
    }

    public Object[] getArgs() {
        return args;
    }

    /**
     * 判断当前是否在播放
     *
     * @return
     */
    public boolean isPlaying() {
        return player != null
                && player.isPlaying()
                && !TextUtils.isEmpty(playPath)
                && args != null && args.length > 0;
    }

    /**
     * 播放出错的监听事件
     */
    private MediaPlayer.OnErrorListener errorListener = new MediaPlayer.OnErrorListener() {
        @Override
        public boolean onError(MediaPlayer mp, int what, int extra) {
            stopToPlay(VoicePlayState.ERROR, args);
            return false;
        }
    };

    /**
     * 播放完成的监听
     */
    private MediaPlayer.OnCompletionListener completionListener = new MediaPlayer.OnCompletionListener() {
        @Override
        public void onCompletion(MediaPlayer mp) {
            stopToPlay(VoicePlayState.COMPLETION, args);
        }
    };
}
