package com.xdja.im.lib.audio.widget;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Message;
import android.os.SystemClock;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Button;
import android.widget.Chronometer;
import android.widget.Chronometer.OnChronometerTickListener;

import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.core.utils.FilePathUtils;
import com.xdja.im.lib.audio.manager.AudioManager;
import com.xdja.im.lib.audio.manager.DialogManager;
import com.xdja.im.lib.audio.manager.TelphoneState;
import com.xdja.im.lib.audio.player.AudioPlayerManager;
import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.constans.Constant;

import java.lang.ref.WeakReference;

public class AudioRecorderButton extends Button {

    /**
     * 默认的状态
     */
    private static final int STATE_NORMAL = 1;
    /**
     * 正在录音
     */
    private static final int STATE_RECORDING = 2;
    /**
     * 希望取消
     */
    private static final int STATE_WANT_TO_CANCEL = 3;
    /**
     * 最后十秒取消
     */
    private static final int LAST_TEN_SECOND_TO_CANCEL = 4;


    private int mCurrentState = STATE_NORMAL; // 当前的状态

    private final DialogManager mDialogManager;//录音弹出框
    private final AudioManager mAudioManager;//录音业务操作类
    private boolean isRecording = false; // 已经开始录音
    private float mTime;

    private boolean mReady;// 是否触发longClick

    private static final int DISTANCE_Y_CANCEL = 50;//滑动Y轴最大高度
    private static final int MSG_AUDIO_PREPARED = 0x110;
    private static final int MSG_DIALOG_DIMISS = 0x112;

    private final String dirPath;//录音保存位置

    private static String lastTenSecordstr;//最后十秒提示字符串

    private final Context mContext;

    private TimerHandler timeHandler = new TimerHandler(new WeakReference<>(this));
    private MyHandler mHandler = new MyHandler(new WeakReference<>(this));

    //无参构造函数
    public AudioRecorderButton(Context context) {
        this(context, null);
    }

    /**
     * 以下2个方法是构造方法
     */
    public AudioRecorderButton(final Context context, AttributeSet attrs) {
        super(context, attrs);

        mContext = context;

        mDialogManager = new DialogManager(context);

        //录音文件存储位置  初始化录音操作类使用
        dirPath = FilePathUtils.getVoicePath();

        mAudioManager = new AudioManager(context, dirPath);
        //录音准备状态回调
        mAudioManager.setOnAudioStateListener(new AudioManager.AudioStateListener() {

            public void wellPrepared() {
                mHandler.sendEmptyMessage(MSG_AUDIO_PREPARED);
            }

            @Override
            public void checkCurrentFocusView() {
                changeState(STATE_NORMAL);
                mDialogManager.tooShortToShow(getContext());
            }

            @Override
            public void prepareFailed(int exceptionType) {
                changeState(STATE_NORMAL);
                if (exceptionType == mAudioManager.EXCEPTION) {
                    mDialogManager.getAudioPermissionErrToShow(getContext());
                } else if (exceptionType == mAudioManager.ILLEGALSTATE_EXCEPTION) {
                    mDialogManager.getAudioDeviceErrToShow(getContext());
                }
            }
        });

        // 由于这个类是button所以在构造方法中添加监听事件
        setOnLongClickListener(new OnLongClickListener() {

            public boolean onLongClick(View v) {
                if (!TelphoneState.getPhotoStateIsIdle(mContext)) {
                    CommonTool.showToast(R.string.im_uikit_phone_is_inCall);
                    return true;
                }

                //停止语音消息播放
                AudioPlayerManager.getInstance().stopPlay();

                requestPermission();
                return true;
            }
        });

        setOnClickListener(new OnClickListener() {

            @Override
            public void onClick(View v) {
                CommonTool.showToast(R.string.im_uikit_record_too_short);
            }
        });
    }

    /**
     * 权限申请
     */
    private void requestPermission() {
        if (PermissionUtils.hasSelfPermissions(mContext,
                Manifest.permission.RECORD_AUDIO,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            new AudioAsyncTask().execute();
        } else {

            //请求权限
            PermissionUtils.requestPermissions((Activity) mContext,
                    new String[]{Manifest.permission.RECORD_AUDIO,
                            Manifest.permission.READ_EXTERNAL_STORAGE,
                            Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_RECORD_AUDIO);

            //提示
            if (!PermissionUtils.shouldShowRequestPermissionRationale((Activity) mContext,
                    Manifest.permission.RECORD_AUDIO,
                    Manifest.permission.READ_EXTERNAL_STORAGE,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
                CommonTool.showToast(R.string.im_uikit_audio_deny);
            }

        }
    }

    public void setActivity(Activity activity) {
        mAudioManager.setActivity(activity);
    }

    private static class TimerHandler extends Handler {
        private final WeakReference<AudioRecorderButton> mActivity;

        public TimerHandler(WeakReference<AudioRecorderButton> mActivity) {
            this.mActivity = mActivity;
        }

        @Override
        public void handleMessage(Message msg) {
            if (mActivity.get() != null) {
                lastTenSecordstr = String.format(
                        CommonTool.getString(R.string.im_uikit_also_have), msg.what);
                switch (msg.what) {
                    case 0:
                        MotionEvent motionEvent = MotionEvent.obtain(SystemClock.uptimeMillis(),
                                SystemClock.uptimeMillis(),
                                MotionEvent.ACTION_UP,
                                10,
                                10,
                                0);
                        mActivity.get().dispatchTouchEvent(motionEvent);
                        break;
                    default:
                        mActivity.get().mDialogManager.mLabel.setText(lastTenSecordstr);
                        break;
                }
            }
        }
    }

    /*
     * 获取音量大小的线程
     */
    private final Runnable mGetVoiceLevelRunnable = new Runnable() {

        public void run() {
            long beginTime = System.currentTimeMillis();
            while (isRecording) {
                try {
                    Thread.sleep(100);
                    mTime = (System.currentTimeMillis() - beginTime) / 1000.0f;
                    // 修改录音音量图片
                    setImage(mAudioManager.getAmplitude());
                    if (isRecording) {
                        if (mTime >= 50) {
                            timeHandler.sendEmptyMessage(60 - (int) mTime);
                        }
                    }
                } catch (InterruptedException e) {
                    e.printStackTrace();
                    // 重置数据
                    reset();
                }
            }
        }
    };

    private static class MyHandler extends Handler {
        private final WeakReference<AudioRecorderButton> mActivity;

        public MyHandler(WeakReference<AudioRecorderButton> mActivity) {
            this.mActivity = mActivity;
        }

        @Override
        public void handleMessage(Message msg) {
            if (mActivity.get() != null) {
                switch (msg.what) {
                    case MSG_AUDIO_PREPARED:
                        //获取音频焦点
                        AudioFunctions.muteAudioFocus(true);
                        mActivity.get().startToRecording();
                        break;

                    case MSG_DIALOG_DIMISS:
                        mActivity.get().mDialogManager.dismissDialog();
                        break;

                    //录音振幅设置图片资源
                    default:
                        mActivity.get().mDialogManager.updateVoiceLevel(msg.what);
                        break;
                }
                super.handleMessage(msg);
            }
        }
    }


    /**
     * 开始录音
     */
    private void startToRecording() {
        // 显示對話框在开始录音以后
        isRecording = true;


        mDialogManager.showRecordingDialog();
        mDialogManager.mVoiceChronometer.setBase(SystemClock
                .elapsedRealtime());
        mDialogManager.mVoiceChronometer.start();
        mDialogManager.mVoiceChronometer.setOnChronometerTickListener(new OnChronometerTickListener() {

            @Override
            public void onChronometerTick(Chronometer chronometer) {
                // 时间改变后就会触发  1s回调一次
                // 时间达到60s就自动完成录音并断掉录音  button长按事件取消  在这里可以记录时间的变化值
            }
        });

        // 开启一个线程
        Thread voiceThread = new Thread(mGetVoiceLevelRunnable);
        voiceThread.start();
    }

    /**
     * 屏幕的触摸事件
     */
    @Override
    public boolean onTouchEvent(MotionEvent event) {

        int action = event.getAction();
        int x = (int) event.getX();// 获得x轴坐标
        int y = (int) event.getY();// 获得y轴坐标

        switch (action) {
            case MotionEvent.ACTION_DOWN:
                changeState(STATE_RECORDING);
                break;
            case MotionEvent.ACTION_MOVE:
                if (isRecording) {
                    // 如果想要取消，根据x,y的坐标看是否需要取消
                    if (wantToCancle(x, y)) {
                        changeState(STATE_WANT_TO_CANCEL);
                    } else {
                        changeState(STATE_RECORDING);
                    }

                }

                break;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                //释放音频焦点
                AudioFunctions.muteAudioFocus(false);

                if (!mReady) {
                    reset();
                    if (mDialogManager.isShowing()) {
                        mDialogManager.dismissDialog();
                        mAudioManager.cancel();
                    }
                    return super.onTouchEvent(event);
                }
                if (!isRecording || mTime < 0.6f) {
                    // Toast 提示录音时间过短
                    mDialogManager.tooShort(getContext());
                    mAudioManager.cancel();
                    // 延迟显示对话框
                    mHandler.sendEmptyMessageDelayed(MSG_DIALOG_DIMISS, 10);
                } else if (mCurrentState == STATE_RECORDING || mTime >= 60) {
                    // 正在录音的时候，结束
                    mDialogManager.dismissDialog();
                    mAudioManager.release();
                    if (audioFinishRecorderListener != null) {
                        audioFinishRecorderListener.onFinish(mTime,
                                mAudioManager.getCurrentFilePath());
                    }

                } else if (mCurrentState == STATE_WANT_TO_CANCEL) { // 想要取消
                    mDialogManager.dismissDialog();
                    mAudioManager.cancel();
                }
                //重置配置数据
                reset();
                break;
            default:
                break;

        }
        return super.onTouchEvent(event);
    }

    /**
     * 恢复状态及标志位
     */
    private void reset() {
        isRecording = false;
        mTime = 0;
        mReady = false;
        changeState(STATE_NORMAL);
    }

    //根据x、y坐标计算想要取消的状态值
    private boolean wantToCancle(int x, int y) {
        if (x < 0 || x > getWidth()) { // 超过按钮的宽度
            return true;
        }
        // 超过按钮的高度
        return y < -DISTANCE_Y_CANCEL || y > getHeight() + DISTANCE_Y_CANCEL;
    }

    /**
     * 改变
     */
    private void changeState(int state) {
        if (mCurrentState != state) {
            mCurrentState = state;
            switch (state) {
                case STATE_NORMAL:
                    setText(R.string.im_uikit_start_record);
                    break;

                case STATE_RECORDING:
                    setText(R.string.im_uikit_stop_record);
                    if (isRecording) {
                        mDialogManager.recording();
                    }
                    break;

                case STATE_WANT_TO_CANCEL:
                    setText(R.string.im_uikit_cancel_send);

                    mDialogManager.wantToCancel();
                    break;
                case LAST_TEN_SECOND_TO_CANCEL:

                    mDialogManager.lastTenSeconds(lastTenSecordstr);
                    break;
                default:
                    break;
            }
        }
    }


    private void setImage(double voiceValue) {
        int voiceImageID;
        if (voiceValue < 600.0) {
            voiceImageID = 1;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 600.0 && voiceValue < 1000.0) {
            voiceImageID = 1;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 1000.0 && voiceValue < 1200.0) {
            voiceImageID = 1;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 1200.0 && voiceValue < 1400.0) {
            voiceImageID = 2;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 1400.0 && voiceValue < 1600.0) {
            voiceImageID = 2;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 1600.0 && voiceValue < 1800.0) {
            voiceImageID = 2;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 1800.0 && voiceValue < 2000.0) {
            voiceImageID = 3;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 2000.0 && voiceValue < 3000.0) {
            voiceImageID = 3;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 3000.0 && voiceValue < 4000.0) {
            voiceImageID = 4;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 4000.0 && voiceValue < 6000.0) {
            voiceImageID = 5;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 6000.0 && voiceValue < 8000.0) {
            voiceImageID = 6;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 8000.0 && voiceValue < 10000.0) {
            voiceImageID = 6;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 10000.0 && voiceValue < 12000.0) {
            voiceImageID = 6;
            mHandler.sendEmptyMessage(voiceImageID);
        } else if (voiceValue > 12000.0) {
            voiceImageID = 7;
            mHandler.sendEmptyMessage(voiceImageID);
        }
    }

    /**
     * 录音完成后的回调
     */
    public interface AudioFinishRecorderListener {
        void onFinish(float seconds, String filePath);
    }

    private AudioFinishRecorderListener audioFinishRecorderListener;

    public void setAudioFinishRecorderListener(
            AudioFinishRecorderListener listener) {
        audioFinishRecorderListener = listener;
    }

    /**
     * OnLongClick对应的异步操作
     */
    private class AudioAsyncTask extends AsyncTask<Void, Void, Boolean> {

        @Override
        protected Boolean doInBackground(Void... params) {
            //获取音频焦点
            AudioFunctions.muteAudioFocus(true);

            mReady = true;
            //开始录音前的准备操作

            mAudioManager.prepareAudio();
            return true;
        }
    }
}
