package com.xdja.im.lib.file.ui.activity;

import android.Manifest;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.View;

import com.bumptech.glide.Glide;
import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.FileSizeUtils;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.lib.file.FileExplorerConfig;
import com.xdja.im.lib.file.FileExplorerManager;
import com.xdja.im.lib.file.bean.FileCollections;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.file.bean.FileListType;
import com.xdja.im.lib.file.ui.adapter.FileExplorerFragmentAdapter;
import com.xdja.im.lib.file.ui.fragment.FileListFragment;
import com.xdja.im.lib.file.ui.presenter.FileExplorerPresenter;
import com.xdja.im.lib.file.ui.view.ViewFileExplorer;
import com.xdja.im.lib.file.ui.view.vu.FileExplorerVu;
import com.xdja.im.lib.file.utils.FileConstant;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;

import java.util.ArrayList;
import java.util.List;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.lib.file
 * @Author: xdjaxa
 * @Creation: 2017-04-24 9:12
 * @Version V1.0
 * @Description:
 */
public class FileExplorerActivity extends ImBaseActivity<FileExplorerPresenter, FileExplorerVu>
        implements FileExplorerPresenter {

    //一级tab
    private Integer[] parentTabTitles = {
            R.string.im_uikit_lately_file,
            R.string.im_uikit_local_file
    };

    private FileExplorerConfig config;

    private FileExplorerFragmentAdapter adapter;

    @NonNull
    @Override
    protected Class<? extends FileExplorerVu> getVuClass() {
        return ViewFileExplorer.class;
    }

    @NonNull
    @Override
    protected FileExplorerPresenter getPresenter() {
        return this;
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (mComponent == null) {
            return;
        }
        mComponent.inject(this);
        BusProvider.getInstance().register(this);

        config = FileExplorerManager.getInstance().getConfig();
        //清空cache中的缓存条目
        FileCollections.getInstance().clearCache();

        List<Fragment> fragments = new ArrayList<>();

        Bundle bundleRec = new Bundle();
        bundleRec.putInt(Constant.TAB_TYPE, FileListType.RECENT.getType());
        fragments.add(FileListFragment.newInstance(bundleRec));

        Bundle bundleLocal = new Bundle();
        bundleLocal.putInt(Constant.TAB_TYPE, FileListType.LOCAL.getType());
        fragments.add(FileListFragment.newInstance(bundleLocal));

        adapter = buildAdapter(parentTabTitles, fragments);
        getVu().setAdapter(adapter);
        getVu().setCurrentSelectedFileSize(this, 0);
    }

    @Override
    protected void onResume() {
        super.onResume();

        requestPermission();
    }

    /**
     * 请求权限
     */
    private void requestPermission() {
        if (!PermissionUtils.hasSelfPermissions(this,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            //请求权限
            PermissionUtils.requestPermissions(this,
                    new String[]{Manifest.permission.READ_EXTERNAL_STORAGE,
                            Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_STORAGE);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode,
                                           @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        if (requestCode == Constant.REQUEST_PERMISSION_STORAGE) {
            if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                adapter.notifyDataSetChanged();
            } else {
                showToast(R.string.im_uikit_storage_deny);
            }
        }
    }

    private FileExplorerFragmentAdapter buildAdapter(final Integer[] titles, List<Fragment> fragments) {
        return new FileExplorerFragmentAdapter<Integer>(getSupportFragmentManager(), this,
                titles, fragments) {
            @Override
            public CharSequence getPageTitle(int position) {
                return context.getString(titles[position]);
            }
        };
    }

    @Override
    public void onSendBtnClick(View view) {
        getVu().setProgressBarVisibility(View.VISIBLE);

        Observable.from(FileCollections.getInstance().getAllSelectFiles())
                .subscribeOn(Schedulers.immediate())
                .flatMap(new Func1<FileItem, Observable<FileItem>>() {
                    @Override
                    public Observable<FileItem> call(FileItem localFileInfo) {
                        long fileSize = localFileInfo.getFileSize();

                        if (fileSize < 0) {

                            CommonTool.showToast(R.string.im_uikit_file_error);
                            return Observable.error(new IllegalArgumentException("file error! fileSize " + fileSize));

                        } else if (fileSize > FileConstant.MAX_FILE_SIZE) {

                            CommonTool.showToast(String.format(getString(R.string.im_uikit_selec_file_too_large),
                                    FileSizeUtils.formatFileSize(FileConstant.MAX_FILE_SIZE, FileSizeUtils.SIZETYPE_MB)));
                            return Observable.error(new Exception("File is too larger!"));

                        }
                        return Observable.just(localFileInfo);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<FileItem>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getVu().setProgressBarVisibility(View.GONE);
                    }

                    @Override
                    public void onNext(FileItem localFileInfo) {
                        getVu().setProgressBarVisibility(View.GONE);
                        setResult();
                    }
                });

    }

    private void setResult() {
        //发送文件
        Intent intent = new Intent();
        //数据绑定
        Bundle bundle = new Bundle();
        ArrayList<FileItem> bundleList = new ArrayList<>();
        bundleList.addAll(FileCollections.getInstance().getAllSelectFiles());
        bundle.putParcelableArrayList(Constant.TAG_FILE_SELECT, bundleList);
        //添加数据到Intent
        intent.putExtras(bundle);
        setResult(Constant.RESULT_CODE_FILE, intent);
        finish();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }

        FileCollections.getInstance().clearCache();
        Glide.get(this).clearMemory();
    }

    @Subscribe
    public void onRefreashFileSelectStatus(IMProxyEvent.FileSelectedEvent event) {
        //更新已选文件大小
        getVu().setCurrentSelectedFileSize(this, FileCollections.getInstance().getSelectedFileSize());
        //更新已选文件个数
        getVu().setCurrentSelectedFileCount(this, FileCollections.getInstance().getSelectedFileCount());
    }
}
