package com.xdja.im.lib.file.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.os.Build;
import android.support.v4.view.PagerAdapter;
import android.support.v4.view.ViewPager;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.HorizontalScrollView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.xdja.im.uikit.R;

public class ViewPagerIndicator extends HorizontalScrollView {

    //需要从系统属性读取数值。
    private static final int[] ATTRS = new int[]{
            android.R.attr.textSize,
            android.R.attr.textColor
    };

    private PagerListener pagerListener = new PagerListener();
    private ViewPager.OnPageChangeListener delegateListener;

    private LinearLayout.LayoutParams expandLayoutParams;

    private Paint indicatorPaint;
    private LinearLayout tabLayout;//标签页父布局
    private ViewPager viewPager;//联动的ViewPager

    private int tabCount;//标签页个数

    private int currentPosition;//当前页数
    private float currentPositionOffset;//当前页偏移量

    private int indicatorColor = 0xFF666666;//指示线颜色
    private int underlineColor = 0x1A000000;//下标线颜色

    private int scrollOffset = 52;
    private int indicatorHeight = 8;
    private int underlineHeight = 2;
    private int tabPadding = 12;

    private int tabTextSize = 12;
    private int tabTextColor = 0xFF666666;
    private int tabTextSelectColor = 0xF3000000;

    private int lastScrollX = 0;

    private int tabBackgroundResId = R.drawable.im_uikit_bg_viewapager_bg_tab;

    public ViewPagerIndicator(Context context) {
        this(context, null);
    }

    public ViewPagerIndicator(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public ViewPagerIndicator(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        setFillViewport(true);
        setWillNotDraw(false);

        tabLayout = new LinearLayout(context);
        tabLayout.setOrientation(LinearLayout.HORIZONTAL);
        tabLayout.setLayoutParams(new LinearLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
        addView(tabLayout);

        DisplayMetrics dm = getResources().getDisplayMetrics();

        scrollOffset = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, scrollOffset, dm);//dp->px
        indicatorHeight = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, indicatorHeight, dm);
        underlineHeight = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, underlineHeight, dm);
        tabPadding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, tabPadding, dm);
        tabTextSize = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, tabTextSize, dm);

        TypedArray ta = context.obtainStyledAttributes(attrs, ATTRS);
        tabTextSize = ta.getDimensionPixelSize(0, tabTextSize);
        tabTextColor = ta.getColor(1, tabTextColor);
        ta.recycle();

        ta = context.obtainStyledAttributes(attrs, R.styleable.VpIndicator);
        indicatorColor = ta.getColor(R.styleable.VpIndicator_vpIndicatorColor, indicatorColor);
        underlineColor = ta.getColor(R.styleable.VpIndicator_vpUnderLineColor, underlineColor);
        indicatorHeight = ta.getDimensionPixelOffset(R.styleable.VpIndicator_vpIndicatorHeight, indicatorHeight);
        underlineHeight = ta.getDimensionPixelOffset(R.styleable.VpIndicator_vpUnderlineHeight, underlineHeight);
        tabPadding = ta.getDimensionPixelOffset(R.styleable.VpIndicator_vpTabPaddingLeftRight, tabPadding);
        tabBackgroundResId = ta.getResourceId(R.styleable.VpIndicator_vpTabBackground, tabBackgroundResId);
        tabTextSelectColor = ta.getColor(R.styleable.VpIndicator_vpTabTextSelectColor, tabTextSelectColor);
        ta.recycle();

        initPaintAndLayoutParams();
    }

    private void initPaintAndLayoutParams() {
        indicatorPaint = new Paint();
        indicatorPaint.setAntiAlias(true);
        indicatorPaint.setStyle(Paint.Style.FILL);

        expandLayoutParams = new LinearLayout.LayoutParams(0, LayoutParams.MATCH_PARENT, 1.0f);
    }

    public void setViewPager(ViewPager vPager) {
        viewPager = vPager;
        viewPager.setOnPageChangeListener(pagerListener);

        notifyDataSetChanged();
    }

    public void setDelegateListener(ViewPager.OnPageChangeListener listener) {
        delegateListener = listener;
    }

    public void notifyDataSetChanged() {
        tabLayout.removeAllViews();

        PagerAdapter pAdapter = viewPager.getAdapter();
        tabCount = pAdapter.getCount();

        for (int i = 0; i < tabCount; i++) {
            addTextTab(i, pAdapter.getPageTitle(i));
        }
        updateTabStyle();

        getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
                    getViewTreeObserver().removeGlobalOnLayoutListener(this);
                } else {
                    getViewTreeObserver().removeOnGlobalLayoutListener(this);
                }
                currentPosition = viewPager.getCurrentItem();
                scrollToChild(currentPosition, 0);
            }
        });
    }

    private void scrollToChild(int position, int offset) {
        if (tabCount == 0) {
            return;
        }

        int newScrollX = tabLayout.getChildAt(position).getLeft() + offset;

        if (position > 0 || offset > 0) {
            newScrollX -= scrollOffset;
        }

        if (newScrollX != lastScrollX) {
            lastScrollX = newScrollX;
            scrollTo(newScrollX, 0);
        }

    }

    private void addTextTab(int position, CharSequence title) {
        TextView tv = new TextView(getContext());
        tv.setGravity(Gravity.CENTER);
        tv.setText(title);
        tv.setSingleLine();

        addTab(position, tv);
    }

    private void addTab(final int position, View tab) {
        tab.setFocusable(true);
        tab.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                viewPager.setCurrentItem(position, false);
            }
        });

        tab.setPadding(tabPadding, 0, tabPadding, 0);
        tabLayout.addView(tab, position, expandLayoutParams);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        if (isInEditMode() || tabCount == 0) {
            return;
        }

        final int height = getHeight();

        //indicator line
        indicatorPaint.setColor(indicatorColor);

        View currentTab = tabLayout.getChildAt(currentPosition);
        int lineLeft = currentTab.getLeft();
        int lineRight = currentTab.getRight();

        if (currentPositionOffset > 0 && currentPosition < tabCount - 1) {
            View nextTab = tabLayout.getChildAt(currentPosition + 1);

            int nextLeft = nextTab.getLeft();
            int nextRight = nextTab.getRight();

            lineLeft += (nextLeft - lineLeft) * currentPositionOffset;
            lineRight += (nextRight - lineRight) * currentPositionOffset;

        }
        canvas.drawRect(lineLeft, height - indicatorHeight, lineRight, height, indicatorPaint);

        indicatorPaint.setColor(underlineColor);
        canvas.drawRect(0, height - underlineHeight, tabLayout.getWidth(), height, indicatorPaint);

    }

    private void updateTabStyle() {
        //背景和字体颜色的更新
        for (int i = 0; i < tabCount; i++) {
            View v = tabLayout.getChildAt(i);
            v.setBackgroundResource(tabBackgroundResId);
            if (v instanceof TextView) {
                TextView tab = (TextView) v;
                tab.setTextSize(TypedValue.COMPLEX_UNIT_PX, tabTextSize);
                tab.setTextColor(tabTextColor);
                if (i == currentPosition) {
                    tab.setTextColor(tabTextSelectColor);
                }
            }
        }

    }

    private final class PagerListener implements ViewPager.OnPageChangeListener {

        @Override
        public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            currentPosition = position;
            currentPositionOffset = positionOffset;

            scrollToChild(currentPosition, (int) (positionOffset * tabLayout.getChildAt(position).getWidth()));

            invalidate();

            if (delegateListener != null) {
                delegateListener.onPageScrolled(position, positionOffset, positionOffsetPixels);
            }
        }

        @Override
        public void onPageSelected(int position) {
            currentPosition = position;
            updateTabStyle();

            if (delegateListener != null) {
                delegateListener.onPageSelected(position);
            }
        }

        @Override
        public void onPageScrollStateChanged(int state) {
            if (state == ViewPager.SCROLL_STATE_IDLE) {//停止
                scrollToChild(currentPosition, 0);
            }

            if (delegateListener != null) {
                delegateListener.onPageScrollStateChanged(state);
            }
        }
    }
}
