package com.xdja.im.lib.video.palyer;

import android.media.AudioManager;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnBufferingUpdateListener;
import android.media.MediaPlayer.OnCompletionListener;
import android.media.MediaPlayer.OnErrorListener;
import android.media.MediaPlayer.OnPreparedListener;
import android.media.MediaPlayer.OnSeekCompleteListener;
import android.text.TextUtils;
import android.view.SurfaceHolder;

import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.utils.log.LogUtil;

/**
 * @Package: com.xdja.im.lib.video.palyer
 * @Author: xdjaxa
 * @Creation: 2017-05-16 9:29
 * @Version V1.0
 * @Description:视频播放器帮助类
 */
public class VideoPlayerHelper implements VideoPlayer,
        OnPreparedListener, OnErrorListener, OnBufferingUpdateListener,
        OnCompletionListener, SurfaceHolder.Callback, OnSeekCompleteListener {

    private SurfaceHolder mHolder;
    /**
     * 播放器
     */
    private MediaPlayer mMediaPlayer;
    /**
     * 播放状态
     */
    private VideoPlayState mPlayState = VideoPlayState.DEFAULT;

    /**
     * 状态变化监听器
     */
    private onStateChangedListener mStateChangedListener;

    private boolean isRepeat;
    private String filePath;
    /**
     * 添加监听器
     *
     * @param listener 监听器
     */
    public void setOnStateChangedListener(onStateChangedListener listener) {
        this.mStateChangedListener = listener;
    }


    public VideoPlayerHelper(SurfaceHolder holder) {
        this.mHolder = holder;
        mHolder.addCallback(this);
        createMediaPlayer();
    }


    /**
     * 创建媒体播放器
     */
    private void createMediaPlayer() {
        mMediaPlayer = new MediaPlayer();
        mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
        mMediaPlayer.setOnBufferingUpdateListener(this);
        mMediaPlayer.setOnCompletionListener(this);
        mMediaPlayer.setOnErrorListener(this);
        mMediaPlayer.setOnPreparedListener(this);
        mMediaPlayer.setOnSeekCompleteListener(this);
    }

    @Override
    public void play(String filePath, boolean isRepeat) {
        this.isRepeat = isRepeat;
        this.filePath = filePath;

        if (mHolder == null || mHolder.getSurface() == null ||
                !mHolder.getSurface().isValid()) {
            LogUtil.e("Start to play, but surface is not valid.");
            return;
        }

        if (TextUtils.isEmpty(filePath)) {
            LogUtil.e("Error:start play video, but filePath is null.");
            return;
        }

        if (mMediaPlayer == null) {
            createMediaPlayer();
        }

        try {
            stop();
            reset();

            AudioFunctions.muteAudioFocus(true);

            mMediaPlayer.setDisplay(mHolder);
            mMediaPlayer.setDataSource(filePath);
            mMediaPlayer.prepareAsync();
            mPlayState = VideoPlayState.OPEN;
        } catch (Exception e) {
            e.printStackTrace();
            reset();
        }
    }

    @Override
    public void pause() {
        if (mMediaPlayer != null) {
            mMediaPlayer.pause();
            mPlayState = VideoPlayState.PAUSE;
            AudioFunctions.muteAudioFocus(false);
        }
    }

    @Override
    public void resume() {
        if (mMediaPlayer != null) {
            mMediaPlayer.start();
            mPlayState = VideoPlayState.PLAY;
            AudioFunctions.muteAudioFocus(true);
        }
    }

    @Override
    public void stop() {
        if (mMediaPlayer != null) {
            mMediaPlayer.stop();
            mPlayState = VideoPlayState.STOP;
        }
    }

    @Override
    public void destroy() {
        if (mMediaPlayer != null) {
            mMediaPlayer.release();
            mMediaPlayer = null;
            mPlayState = VideoPlayState.DEFAULT;
        }
    }

    @Override
    public void reset() {
        if (mMediaPlayer != null) {
            mMediaPlayer.reset();
            mPlayState = VideoPlayState.DEFAULT;
        }
    }

    @Override
    public boolean isPlaying() {
        if (mMediaPlayer != null) {
            return mMediaPlayer.isPlaying();
        }
        return false;
    }

    @Override
    public VideoPlayState getPlayState() {
        return mPlayState;
    }

    @Override
    public int getCurrentPosition() {
        if (mMediaPlayer != null && mMediaPlayer.isPlaying()) {
            return mMediaPlayer.getCurrentPosition();
        }
        return 0;
    }

    @Override
    public void onBufferingUpdate(MediaPlayer mp, int percent) {

    }

    @Override
    public void onCompletion(MediaPlayer mp) {
        if (isRepeat) {
            play(filePath, isRepeat);
        } else {
            mPlayState = VideoPlayState.COMPLETION;
            if (mStateChangedListener != null) {
                mStateChangedListener.onStateChanged(mPlayState);
            }
            AudioFunctions.muteAudioFocus(false);
        }
    }

    @Override
    public boolean onError(MediaPlayer mp, int what, int extra) {
        mPlayState = VideoPlayState.ERROR;
        if (mStateChangedListener != null) {
            mStateChangedListener.onStateChanged(mPlayState);
        }
        AudioFunctions.muteAudioFocus(false);
        return false;
    }

    @Override
    public void onPrepared(MediaPlayer mp) {
        mp.start();
        mPlayState = VideoPlayState.PLAY;
    }

    @Override
    public void onSeekComplete(MediaPlayer mp) {

    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {

    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
        if (mMediaPlayer != null) {
            mMediaPlayer.setDisplay(holder);
        }
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {

    }
}
