package com.xdja.im.uikit.notification;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.media.MediaPlayer;
import android.media.RingtoneManager;
import android.os.Bundle;
import android.os.SystemClock;
import android.os.Vibrator;
import android.support.annotation.NonNull;
import android.support.v4.app.NotificationCompat;
import android.text.TextUtils;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.animation.GlideAnimation;
import com.bumptech.glide.request.target.SimpleTarget;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.param.DisturbParam;
import com.xdja.im.core.model.param.RemindParam;
import com.xdja.im.core.model.param.SettingParam;
import com.xdja.im.core.repository.interf.datasource.DiskDataStore;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.IntentParam;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.image.ImageUtils;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import rx.Subscriber;

public class NotificationUtil {
    private static final int DISTANCE_TIME = 3000;
    private static final String NOT_NOTIFY_ACTIVITY = "com.xdja.im.uikit.ui.activity.ChatListActivity";
    private Context mContext;
    private long addPushTime;
    private boolean isCurrentTalker = false;

    /**
     * 存储单人提醒消息列表
     */
    private final ArrayList<NotificationBean> singleNotifyList = new ArrayList<>();
    /**
     * 存储群组提醒消息列表
     */
    private final ArrayList<NotificationBean> groupNotifyList = new ArrayList<>();

    private Set<String> accounts = new HashSet<>();
    private MediaPlayer player = null;
    private NotificationManager notifyManager;

    private DiskDataStore diskDataStore;

    /**
     * 当前登录用户账号
     */
    private String loginAccount;
    /**
     * 当前聊天用户账号
     */
    private String mCurTalkAccount;

    private static class SingletonInstance {
        private static final NotificationUtil mInstance = new NotificationUtil();
    }

    public static NotificationUtil getInstance() {
        return SingletonInstance.mInstance;
    }

    private NotificationUtil() {
        this.mContext = ImUiKit.getInstance().getContext();
        notifyManager = (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);
        if (ImUiKit.getInstance().getComponent() != null) {
            diskDataStore = ImUiKit.getInstance().getComponent().diskDataStore();
        }
    }

    public void setLoginAccount(String account) {
        loginAccount = account;
    }

    public void setCurTalkAccount(String account) {
        mCurTalkAccount = account;
    }

    /**
     * 消息提醒
     *
     * @param account   群组为群号，单人为聊天对象
     * @param imgUrl    这个账号对应的头像的地址
     * @param userName  这个账号对应的名称
     * @param talkId    群组为群号_消息类型，单人为聊天对象_消息类型
     * @param msgSize   传递过来的新消息的数量
     * @param msgTime   消息时间
     * @param isDisturb 是否免打扰
     */
    public void remindMessage(String account, String imgUrl,
                              String userName, String talkId,
                              int talkType, int msgSize, long msgTime,
                              boolean isDisturb) {
        NotifyConstant.getAction();
        NotifyContactInfo info = new NotifyContactInfo(account, imgUrl, userName, msgTime);

        NewMessageInfo msgInfo = new NewMessageInfo(talkId, msgSize, talkType, isDisturb);
        accounts.add(account);
        initNotifications(info, msgInfo);
    }

    private void initNotifications(final NotifyContactInfo contactInfo, final NewMessageInfo newMsgInfo) {
        List<NotificationBean> notifyBeans = new ArrayList<>();

        if (!newMsgInfo.isDisturb && accounts.contains(contactInfo.account)) {
            notifyBeans = addNotificationBean(contactInfo, newMsgInfo);
        }

        if (TextUtils.isEmpty(loginAccount)) {
            LogUtil.e("ERROR: Current login account is null.");
            return;
        }
        if (notifyBeans == null || notifyBeans.isEmpty()) {
            LogUtil.e("ERROR: Notify bean is null.");
            return;
        }

        final List<NotificationBean> finalNotifyBeans = notifyBeans;
        diskDataStore.loadSettingParams(new Subscriber<SettingParam>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(SettingParam settingParam) {

                if (settingParam == null) {
                    LogUtil.w("Load setting param is null.");
                    return;
                }

                RemindParam remind = settingParam.getRemind();
                DisturbParam disturb = settingParam.getDisturb();
                //免打扰时间段内
                if (disturb != null && disturb.isOpen() &&
                        isInNoDisturb(disturb.getStartTime(), disturb.getEndTime())) {
                    LogUtil.d("Current is within disturb.");
                    return;
                }

                //消息通知关闭
                if (!remind.isOpen()) {
                    LogUtil.d("Remind is close.");
                    return;
                }
                sendNotify(finalNotifyBeans, newMsgInfo, contactInfo);
            }
        });
    }

    /**
     * 发送通知消息
     *
     * @param finalNotifyBeans
     * @param newMsgInfo
     * @param contactInfo
     */
    private void sendNotify(List<NotificationBean> finalNotifyBeans, NewMessageInfo newMsgInfo,
                            NotifyContactInfo contactInfo) {

        final String[] showMessages = new String[2];

        int newCount = 0;
        int size = finalNotifyBeans.size();
        StringBuilder sbName = new StringBuilder();
        for (int i = 0; i < size; i++) {
            NotificationBean notificationBean = finalNotifyBeans.get(i);
            String name;
            if (TextUtils.isEmpty(notificationBean.accountName)) {
                name = CommonTool.getString(R.string.im_uikit_group_name_default);
            } else {
                name = notificationBean.accountName;
            }
            sbName.append(name);
            newCount += notificationBean.msgCount;
            if (i != finalNotifyBeans.size() - 1) {
                if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G) {
                    //多个群组名称之间用“；”隔开
                    sbName.append("；");
                } else {
                    //多个单人名称之间用“、”隔开
                    sbName.append("、");
                }
            }
        }

        showMessages[0] = sbName.toString();
        showMessages[1] = CommonTool.getString(R.string.im_uikit_notify_prestr)
                + newCount + CommonTool.getString(R.string.im_uikit_notify_endstr);
        showNotificationHeader(contactInfo, newMsgInfo, showMessages);
    }

    private void showNotificationHeader(final NotifyContactInfo contactInfo,
                                        final NewMessageInfo newMsgInfo, final String[] messages) {
        String imageUrl = null;
        if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G) {
            if (groupNotifyList.size() > 0) {
                NotificationBean lastBean = groupNotifyList.get(0);
                imageUrl = lastBean.imageUrl;
            }
        } else {
            if (singleNotifyList.size() > 0) {
                NotificationBean lastBean = singleNotifyList.get(0);
                imageUrl = lastBean.imageUrl;
            }
        }

        Glide.with(mContext).load(imageUrl).asBitmap().into(new SimpleTarget<Bitmap>() {

            @Override
            public void onResourceReady(Bitmap resource, GlideAnimation<? super Bitmap> glideAnimation) {
                Bitmap bitmap = ImageUtils.getRoundedCornerBitmap(resource);
                notifyMessage(bitmap, messages, contactInfo, newMsgInfo);
            }

            @Override
            public void onLoadFailed(Exception e, Drawable errorDrawable) {
                super.onLoadFailed(e, errorDrawable);
                Drawable drawable = CommonTool.getDrawable(R.mipmap.im_uikit_user_header_40dp);//默认为单人头像
                if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G) {
                    drawable = CommonTool.getDrawable(R.mipmap.im_uikit_user_header_40dp);
                }
                Bitmap largeBitmap = ImageUtils.drawableToBitmap(drawable);
                notifyMessage(largeBitmap, messages, contactInfo, newMsgInfo);
            }
        });
    }

    private void notifyMessage(Bitmap header, String[] messages,
                               NotifyContactInfo contactInfo,
                               NewMessageInfo newMsgInfo) {
        isCurrentTalker = false;
        if (!newMsgInfo.isDisturb) {
            if (!NotifyUtils.isAppOnForeground(mContext)) {
                addMediaNotification(mContext);
                showNotification(contactInfo, newMsgInfo, header, messages);
            } else {
                String currentActivityName = NotifyUtils.getCurrentActivityName(mContext);
                if (currentActivityName == null) {
                    addMediaNotification(mContext);
                    showNotification(contactInfo, newMsgInfo, header, messages);
                } else if (NotifyUtils.isScreenOffOrLock(mContext)) {
                    addMediaNotification(mContext);
                    showNotification(contactInfo, newMsgInfo, header, messages);
                } else if (currentActivityName.equals(NOT_NOTIFY_ACTIVITY)
                        && !TextUtils.isEmpty(mCurTalkAccount) && contactInfo != null && mCurTalkAccount.equals(contactInfo.account)) {
                    isCurrentTalker = true;
                    addMediaNotification(mContext);
                    // 屏幕在前台，屏幕亮，且解锁,正在聊天界面与当前人员聊天
                } else if (currentActivityName.equals(NOT_NOTIFY_ACTIVITY)
                        && !TextUtils.isEmpty(mCurTalkAccount) && contactInfo != null && !mCurTalkAccount.equals(contactInfo.account)) {
                    // 在前台，并且与当前人员聊天界面，实现：只震动。
                    addMediaNotification(mContext);
                    //msgIdList.clear();
                } else {
                    // 只做提醒，不显示通知消息
                    addMediaNotification(mContext);
                }
            }
        }
    }

    private void showNotification(NotifyContactInfo contactInfo, NewMessageInfo newMsgInfo,
                                  Bitmap header, String[] messages) {
        String notifyMsg = messages[1];
        String notifyTitleMsg = messages[0];

        if (notifyMsg == null || TextUtils.isEmpty(notifyTitleMsg)) {
            return;
        }

        NotificationCompat.Builder notificationBuilder = new NotificationCompat.Builder(mContext);
        NotificationIntent intent = new NotificationIntent(NotifyConstant.PARAMTARGET);
        Bundle bundle = new Bundle();
        bundle.putInt("pageIndex", 0);

        if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2P && singleNotifyList.size() > 0) {
            if (singleNotifyList.size() == 1) {
                bundle.putString(NotifyConstant.NOTIFYTARGET, NotifyConstant.NOT_NOTIFY_ACTIVITY);
            } else {
                bundle.putString(NotifyConstant.NOTIFYTARGET, NotifyConstant.NOT_NOTIFY_FRAGMENT);
            }
        }

        if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G && groupNotifyList.size() > 0) {
            if (groupNotifyList.size() == 1) {
                bundle.putString(NotifyConstant.NOTIFYTARGET, NotifyConstant.NOT_NOTIFY_ACTIVITY);
            } else {
                bundle.putString(NotifyConstant.NOTIFYTARGET, NotifyConstant.NOT_NOTIFY_FRAGMENT);
            }
        }

        String notificationTag;
        if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G) {
            notificationTag = NotifyConstant.NOTIFICATION_GROUP;
        } else {
            notificationTag = NotifyConstant.NOTIFICATION_SINGLE;
        }

        bundle.putString(IntentParam.PARAM_SESSION_ID, contactInfo.account);

        bundle.putInt(IntentParam.PARAM_SESSION_TYPE, newMsgInfo.talkType);
        intent.setPackage(mContext.getPackageName());
        intent.putExtras(bundle);

        int requestId = 0;
        PendingIntent pendingIntent = intent.buildPendingIntent(
                mContext, ((int) System.currentTimeMillis()));
        notificationBuilder.setContentIntent(pendingIntent);

        // TODO: 2017/5/26 获取应用图标
        int res = R.mipmap.ic_launcher;
        // step1
        notificationBuilder
                .setContentText(notifyMsg)
                .setContentTitle(notifyTitleMsg)
                .setSmallIcon(res)
                .setTicker(notifyMsg)
                .setLargeIcon(header)
                .setContentIntent(pendingIntent);

        Notification notification = notificationBuilder.build();
        notification.ledARGB = 0xff00ff00;
        notification.ledOnMS = 1;
        notification.ledOffMS = 0;
        notification.when = contactInfo.time;
        notification.flags |= Notification.FLAG_SHOW_LIGHTS;
        notification.flags |= Notification.FLAG_AUTO_CANCEL;
        notifyManager.notify(notificationTag, requestId, notification);
    }

    private void addMediaNotification(final Context context) {
        long bootTime = SystemClock.elapsedRealtime();
        long time = bootTime - addPushTime;
        if (time < DISTANCE_TIME) {
            LogUtil.d("Last remind time is " + time);
            return;
        }
        addPushTime = SystemClock.elapsedRealtime();
        diskDataStore.loadSettingParams(new Subscriber<SettingParam>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(SettingParam settingParam) {
                if (settingParam == null) {
                    LogUtil.w("Load setting param is null.");
                    return;
                }

                RemindParam remind = settingParam.getRemind();
                if (!remind.isOpen()) {
                    LogUtil.d("Remind is close.");
                }

                if (remind.isVoice()) {  //声音提醒
                    remindByVoice(context);
                }

                if (remind.isVibrate()) { //震动提醒
                    remindByVibrate(context);
                }
            }
        });
    }

    /**
     * 声音提醒
     */
    private void remindByVoice(Context context) {
        try {
            //没有判空是因为更改系统提示音，消息提示音应同步修改
            player = MediaPlayer.create(context,
                    RingtoneManager.getActualDefaultRingtoneUri(context, RingtoneManager.TYPE_NOTIFICATION));
            player.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
                @Override
                public void onCompletion(MediaPlayer mediaPlayer) {
                    mediaPlayer.stop();
                    mediaPlayer.release();
                    mediaPlayer = null;
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (player != null && !player.isPlaying() && !isCurrentTalker) {
                player.start();
            }
        }
    }

    /**
     * 震动提醒
     */
    private void remindByVibrate(Context context) {
        Vibrator vibrator = (Vibrator) context.getSystemService(Context.VIBRATOR_SERVICE);
        if (vibrator != null) {
            vibrator.vibrate(300);
        }
    }

    private List<NotificationBean> addNotificationBean(NotifyContactInfo contactInfo,
                                                       NewMessageInfo newMsgInfo) {
        String account = contactInfo.account;
        String imgUrl = contactInfo.imgUrl;
        String accountName = contactInfo.name;
        String talkId = newMsgInfo.talkId;

        int containIndex = -1;

        List<NotificationBean> tempList;
        if (newMsgInfo.talkType == ConstDef.CHAT_TYPE_P2G) {
            tempList = groupNotifyList;
        } else {
            tempList = singleNotifyList;
        }

        int size = tempList.size();
        for (int i = 0; i < size; i++) {
            if (tempList.get(i).equals(account)) {
                containIndex = i;
                break;
            }
        }

        if (containIndex > -1) {
            NotificationBean notificationBean = tempList.get(containIndex);
            if (notificationBean != null) {
                int preMsgCount = notificationBean.msgCount < 0 ? 0 : notificationBean.msgCount;
                notificationBean.accountName = accountName;
                notificationBean.msgCount = preMsgCount + newMsgInfo.newMsgCount;
            } else {
                notificationBean = new NotificationBean(account, accountName, newMsgInfo.newMsgCount, imgUrl);
            }
            tempList.remove(containIndex);
            notificationBean.imageUrl = imgUrl;
            tempList.add(0, notificationBean);
        } else {
            NotificationBean notificationBean = new NotificationBean(account, accountName, newMsgInfo.newMsgCount, imgUrl);
            tempList.add(0, notificationBean);
        }

        return tempList;
    }

    /**
     * 取消推送消息通知
     */
    public void clearPNNotification() {
        notifyManager.cancel(NotifyConstant.NOTIFICATION_SINGLE, 0);
        notifyManager.cancel(NotifyConstant.NOTIFICATION_GROUP, 0);
        singleNotifyList.clear();
        groupNotifyList.clear();
    }

    /**
     * 是否在应用免打扰时间内
     */
    public boolean isInNoDisturb(long startTime, long endTime) {
        Calendar calendar = Calendar.getInstance();
        int nowHour = calendar.get(Calendar.HOUR_OF_DAY);
        int nowMin = calendar.get(Calendar.MINUTE);
        int startHour = (int) (startTime / 3600);
        int startMin = (int) ((startTime - startHour * 3600) / 60);
        int endHour = (int) (endTime / 3600);
        int endMin = (int) ((endTime - endHour * 3600) / 60);
        int tempHour = 0;
        boolean flag = false;
        if (startHour > endHour) {
            tempHour = startHour;
            startHour = endHour;
            endHour = tempHour;
            flag = true;
        }
        if (nowHour < startHour || nowHour > endHour) {
            return flag;
        }
        if (nowHour == startHour && nowHour < endHour) {
            if (flag) {
                return !(nowMin > startMin);
            }
            return nowMin > startMin;
        }

        if (nowHour == endHour && nowHour > startHour) {
            if (flag) {
                return !(nowMin < endMin);
            }
            return nowMin < endMin;
        }

        if (nowHour > startHour && nowHour < endHour) {
            return !flag;
        }

        if (nowHour == startHour && nowHour == endHour) {
            return startMin < nowMin && nowMin < endMin;
        }
        return false;
    }

    public class NotifyContactInfo {
        public String account;
        public String imgUrl;
        public String name;
        public long time;

        public NotifyContactInfo(String account, String imgUrl, String name, long time) {
            this.account = account;
            this.imgUrl = imgUrl;
            this.name = name;
            this.time = time;
        }
    }

    public class NewMessageInfo {
        /**
         * 会话id
         */
        public String talkId;
        /**
         * 新消息数量
         */
        public int newMsgCount;
        /**
         * 是否免打扰
         */
        public boolean isDisturb;

        /**
         * 会话类型
         */
        public int talkType;

        public NewMessageInfo(String talkId, int newMsgCount, int talkType, boolean isDisturb) {
            this.talkId = talkId;
            this.talkType = talkType;
            this.newMsgCount = newMsgCount;
            this.isDisturb = isDisturb;
        }
    }

    public class NotificationBean {
        /**
         * 账号
         */
        public String account;
        /**
         * 账号对应的名称
         */
        public String accountName;
        /**
         * 消息数量
         */
        public int msgCount;
        /**
         * 头像
         */
        public String imageUrl;

        public NotificationBean(String account, String accountName, int msgCount, String imageUrl) {
            this.account = account;
            this.accountName = accountName;
            this.msgCount = msgCount;
            this.imageUrl = imageUrl;
        }

        @Override
        public boolean equals(Object o) {
            if (account == null || o == null) {
                return false;
            }
            if (o instanceof NotificationBean) {
                NotificationBean notificationBean = (NotificationBean) o;
                if (notificationBean.account.equals(account)) {
                    return true;
                }
            } else {
                if (o instanceof String) {
                    String tempId = (String) o;
                    if (tempId.equals(account)) {
                        return true;
                    }
                }
                return false;
            }
            return super.equals(o);
        }
    }

    public class NotificationIntent extends Intent {

        public NotificationIntent(@NonNull String targetName) {
            super(NotifyConstant.NOTIFYACTION);
            this.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            this.putExtra(NotifyConstant.NOTIFYTARGET, targetName);
        }

        public PendingIntent buildPendingIntent(@NonNull Context context, int reqCode) {
            return PendingIntent.getBroadcast(context, reqCode, this, PendingIntent.FLAG_UPDATE_CURRENT);
        }
    }
}
