package com.xdja.im.common.exception;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.Process;

import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * @Package: com.xdja.im.common.utils
 * @Author: xdjaxa
 * @Creation: 2017-06-05 10:31
 * @Version V1.0
 * @Description:
 */
public class CrashHandler implements Thread.UncaughtExceptionHandler {

    private static final boolean DEBUG = true;

    private static final String FILE_NAME = "crash";

    private static final String FILE_NAME_SUFFIX = ".txt";

    private Context mContext;

    private static String mCrashPath;

    private Thread.UncaughtExceptionHandler mDefaultCrashHandler;

    private static class CrashHandlerInstance {
        private static final CrashHandler mInstance = new CrashHandler();
    }

    private CrashHandler() {
    }

    public static CrashHandler getInstance() {
        return CrashHandlerInstance.mInstance;
    }

    public void init(Context context) {
        mDefaultCrashHandler = Thread.getDefaultUncaughtExceptionHandler();
        Thread.setDefaultUncaughtExceptionHandler(this);
        mContext = context.getApplicationContext();
        mCrashPath = Environment.getExternalStorageDirectory() + File.separator +
                "CrashLog" + File.separator + "log";
    }

    /**
     * 当程序中有未被捕获的异常，系统会自动调用 uncaughtException 方法
     *
     * @param thread    出现未捕获异常的线程
     * @param throwable 未捕获的异常
     */
    @Override
    public void uncaughtException(Thread thread, Throwable throwable) {
        try {
            dumpExceptionToSDCard(throwable);
            uploadExceptionToServer();
        } catch (IOException e) {
            e.printStackTrace();
        }

        throwable.printStackTrace();
        //如果系统提供了默认异常处理器，则交给系统去结束程序，否则就自己结束程序
        if (mDefaultCrashHandler != null) {
            mDefaultCrashHandler.uncaughtException(thread, throwable);
        } else {
            Process.killProcess(Process.myPid());
        }
    }

    /**
     * 将异常信息写入SD卡中
     *
     * @param throwable
     * @throws IOException
     */
    private void dumpExceptionToSDCard(Throwable throwable) throws IOException {
        //如果 SD 卡不存在或无法使用，则无法把异常信息写入SD卡中
        if (!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            if (DEBUG) {
                LogUtil.w("sdcard unmounted,skip dump exception");
                return;
            }
        }

        File dir = new File(mCrashPath);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        long current = System.currentTimeMillis();
        String time = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(current));
        String fileName = FILE_NAME + time + FILE_NAME_SUFFIX;
        File file = new File(mCrashPath, fileName);
        try {
            PrintWriter pw = new PrintWriter(new BufferedWriter(new FileWriter(file)));
            pw.println(time);
            dumpPhoneInfo(pw);
            pw.println();
            throwable.printStackTrace(pw);
            pw.close();
        } catch (IOException e) {
            LogUtil.e("dump crash info failed");
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * 将手机信息写入文件中
     *
     * @param pw
     * @throws PackageManager.NameNotFoundException
     */
    private void dumpPhoneInfo(PrintWriter pw) throws PackageManager.NameNotFoundException {

        PackageManager pm = mContext.getPackageManager();
        PackageInfo pi = pm.getPackageInfo(mContext.getPackageName(), PackageManager.GET_ACTIVITIES);
        pw.write("App Version：");
        pw.print(pi.versionName);
        pw.print('_');
        pw.println(pi.versionCode);

        //Android 版本号
        pw.print("OS Version： ");
        pw.print(Build.VERSION.RELEASE);
        pw.print('_');
        pw.println(Build.VERSION.SDK_INT);

        //手机制造商
        pw.print("Model： ");
        pw.println(Build.MANUFACTURER);

        //手机型号
        pw.print("Model： ");
        pw.println(Build.MODEL);

        //CPU 架构
        pw.print("CPU ABI：");
        pw.println(Build.CPU_ABI);

    }

    private void uploadExceptionToServer() {
        //把文件上传到服务器的操作


    }
}
