package com.xdja.im.uikit.ui.activity;

import android.Manifest;
import android.app.Activity;
import android.content.Intent;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.support.annotation.NonNull;
import android.support.annotation.RequiresApi;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.widget.Toast;

import com.bumptech.glide.Glide;
import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.FileSizeUtils;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.event.IMContactEvent;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.core.model.event.IMProxyFileEvent;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.NormalFileInfo;
import com.xdja.im.core.model.message.file.VoiceFileInfo;
import com.xdja.im.core.model.param.SessionParam;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.core.repository.interf.datasource.DiskDataStore;
import com.xdja.im.lib.action.bean.OptionBean;
import com.xdja.im.lib.action.uitls.OptionParser;
import com.xdja.im.lib.album.ImagePickerManager;
import com.xdja.im.lib.album.bean.ImageThumbBean;
import com.xdja.im.lib.audio.player.AudioListener;
import com.xdja.im.lib.audio.player.AudioListenerManager;
import com.xdja.im.lib.audio.player.AudioPlayerManager;
import com.xdja.im.lib.camera.CameraManager;
import com.xdja.im.lib.camera.bean.PhotoThumbBean;
import com.xdja.im.lib.file.FileExplorerManager;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.video.VideoRecorderManager;
import com.xdja.im.lib.video.bean.VideoItem;
import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.IntentParam;
import com.xdja.im.uikit.action.media.MediaManger;
import com.xdja.im.uikit.action.media.ReceiverModeManger;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.notification.NotificationUtil;
import com.xdja.im.uikit.ui.adapter.ChatListAdapter;
import com.xdja.im.uikit.ui.presenter.ChatListPresenter;
import com.xdja.im.uikit.ui.view.ViewChatList;
import com.xdja.im.uikit.ui.view.vu.ChatListVu;
import com.xdja.im.uikit.utils.DataConvertUtils;
import com.xdja.im.uikit.utils.ImageCache;
import com.xdja.im.uikit.utils.audio.AudioPlayerManagerProxy;
import com.xdja.im.uikit.utils.file.TranslateInfoUtil;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.uikit.ui.activity
 * @Author: xdjaxa
 * @Creation: 2017-05-22 11:37
 * @Version V1.0
 * @Description: 聊天界面
 */
public class ChatListActivity extends ImBaseActivity<ChatListPresenter, ChatListVu>
        implements ChatListPresenter, AudioListener {

    /**
     * 每页消息展示数量
     */
    private static final int PAGE_SIZE = 15;

    //IMSDK回调
    @Inject
    Lazy<ProxyRepository> proxyRepository;

    //数据存储
    @Inject
    Lazy<DiskDataStore> diskDataStore;

    /**
     * 聊天对象账号id
     */
    private String mTalkId;
    /**
     * 会话类型
     */
    private int mTalkType;

    /**
     * 是否选中闪信
     */
    private boolean isLimit = false;

    /**
     * 进入聊天界面时，是否存在草稿信息
     */
    private boolean isHasDraft;

    /**
     * 消息列表
     */
    private List<TalkMessageBean> mMessageList = new ArrayList<>();

    /**
     * 适配器
     */
    private ChatListAdapter mAdapter;

    @Override
    protected Class<? extends ChatListVu> getVuClass() {
        return ViewChatList.class;
    }

    @NonNull
    @Override
    protected ChatListPresenter getPresenter() {
        return this;
    }

    @Override
    public Activity getActivity() {
        return this;
    }

    @Override
    public String getTalkFlag() {
        return mTalkId + "_" + mTalkType;
    }

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        obtainTalker();
        if (mComponent == null || TextUtils.isEmpty(mTalkId)) {
            finish();
            return;
        }

        //注册回调
        mComponent.inject(this);
        BusProvider.getInstance().register(this);
        NotificationUtil.getInstance().setCurTalkAccount(mTalkId);
        AudioListenerManager.getInstance().addListener(this);
        //听筒相关
        new ReceiverModeManger().init(getActivity());
        //初始化
        initView();
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        setIntent(intent);
        recreate();
    }

    @Override
    protected void onStart() {
        super.onStart();

        requestPermissions();
    }

    /**
     * 获取聊天账号信息
     */
    private void obtainTalker() {
        Bundle arguments = getIntent().getExtras();
        if (arguments != null) {
            mTalkId = arguments.getString(Constant.TAG_TALKERID);
            mTalkType = arguments.getInt(Constant.TAG_TALKTYPE, ConstDef.CHAT_TYPE_P2P);
        }
    }

    /**
     * 初始化
     */
    private void initView() {
        loadMessageList(0);
        //清除未读消息数
        clearUnReadMsgCount();
        //获取草稿信息
        getDraftContent();
        //加载消息Option数据
        startLoadOptions();
        setTitleName();
    }



    private void setTitleName() {
        getVu().setToolBarAddVisibility(View.VISIBLE);
        getVu().setToolBarVisibility(View.VISIBLE);
        if (mTalkType == ConstDef.CHAT_TYPE_P2P) {
            IMUiKitProxyImpl.getInstance().getAccount(mTalkId, ConstDef.CHAT_TYPE_P2P, new Subscriber<AccountBean>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {

                }

                @Override
                public void onNext(AccountBean accountBean) {
                    getVu().setTitle(accountBean.getDisplayName());
                    getVu().setToolBarAddVisibility(View.VISIBLE);
                }
            });
        } else {
            IMUiKitProxyImpl.getInstance().getAccount(mTalkId, ConstDef.CHAT_TYPE_P2G, new Subscriber<AccountBean>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {

                }

                @Override
                public void onNext(AccountBean accountBean) {
                    getVu().setTitle(accountBean.getDisplayName());

                    if (IMUiKitProxyImpl.getInstance().isShowGroupSetting(mTalkId)) {
                        getVu().setToolBarAddVisibility(View.VISIBLE);
                    } else {
                        getVu().setToolBarAddVisibility(View.INVISIBLE);
                    }
                }
            });

        }
    }

    /**
     * 请求读写sd卡权限
     */
    private void requestPermissions() {
        if (!PermissionUtils.hasSelfPermissions(getActivity(),
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {

            //请求权限
            PermissionUtils.requestPermissions(getActivity(),
                    new String[]{Manifest.permission.READ_EXTERNAL_STORAGE,
                            Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_STORAGE);
        }
    }

    /**
     * 构建Adapter
     *
     * @return
     */
    private ChatListAdapter buildAdapter() {
        if (mAdapter == null) {
            mAdapter = new ChatListAdapter(this, mMessageList,
                    getVu().getListView(), proxyRepository.get());
            getVu().getListView().setAdapter(mAdapter);
        }
        return mAdapter;
    }

    /**
     * 加载当前会话中的所有聊天消息
     *
     * @param msgId 消息id，如果为0，则表示拉取最新消息
     */
    private void loadMessageList(final long msgId) {

        LogUtil.d("Start load message msgId " + msgId);
        proxyRepository.get().getMessageList(mTalkId, mTalkType, msgId, PAGE_SIZE, new Subscriber<List<TalkMessageBean>>() {
            @Override
            public void onCompleted() {
                if (getVu().isRefreshing()) {
                    getVu().stopRefresh();
                }
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("load message error:" + e.getMessage());
                if (getVu().isRefreshing()) {
                    getVu().stopRefresh();
                }
            }

            @Override
            public void onNext(List<TalkMessageBean> talkMessageList) {
                LogUtil.d(talkMessageList.toString());

                if (talkMessageList == null /*|| talkMessageList.isEmpty()*/) {
                    return;
                }
                if (msgId == 0) {
                    //传入0，拉取最新消息
                    mMessageList.clear();
                }

                if (talkMessageList.size() > 0) {
                    mMessageList.addAll(0, talkMessageList);
                }
                buildAdapter().notifyDataSetChanged();

                LogUtil.d("Load message is refreshing " + getVu().isRefreshing());
                if (getVu().isRefreshing()) {
                    if (talkMessageList.size() > 0) {
                        getVu().setListSelection(talkMessageList.size() - 1);
                    }
                } else {
                    if (mMessageList.size() > 0) {
                        getVu().setListSelection(mMessageList.size() - 1);
                    }
                    getVu().restoreActionState();
                }
            }
        });
    }

    /**
     * 加载Options数据
     */
    private void startLoadOptions() {
        Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<OptionBean>>() {
                    @Override
                    public Observable<OptionBean> call(String s) {
                        try {
                            List<OptionBean> options = OptionParser.getOptions(getActivity());
                            return Observable.from(options);
                        } catch (Exception e) {
                            e.printStackTrace();
                            return Observable.error(e);
                        }
                    }
                })
                .filter(new Func1<OptionBean, Boolean>() {
                    @Override
                    public Boolean call(OptionBean optionBean) {
                        if (mTalkType == ConstDef.CHAT_TYPE_P2G) {
                            return optionBean.getType() != 1;
                        } else if (mTalkType == ConstDef.CHAT_TYPE_P2P) {
                            return optionBean.getType() != 2;
                        }
                        return true;
                    }
                })
                .toList()
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<OptionBean>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Load options data error." + e.getMessage());
                    }

                    @Override
                    public void onNext(List<OptionBean> optionBeen) {
                        LogUtil.d("ChatOptions:" + optionBeen.toString());
                        getVu().setOptions(optionBeen);
                    }
                });
    }


    @Override
    public void downRefreshList() {

        long msgId = 0;
        if (!mMessageList.isEmpty()) {
            msgId = mMessageList.get(0).getId();
        }
        LogUtil.d("do refresh, msgId :" + msgId);

        loadMessageList(msgId);
    }

    /**
     * 退出聊天界面，通知更新会话内容
     */
    private void notifyUpdateLastSession() {

        String draftContent = getVu().getInputContent();
        LogUtil.d("Draft input content:" + draftContent);
        if (draftContent != null && !TextUtils.isEmpty(draftContent.trim())) {
            //退出时有草稿信息，更新草稿信息
            updateDraftContent(draftContent);
        } else {
            //通知更新消息
            if (!mMessageList.isEmpty()) {
                TalkSessionBean updateSessionBean = new TalkSessionBean();
                TalkMessageBean lastMessageBean = mMessageList.get(mMessageList.size() - 1);
                updateSessionBean.setContent(lastMessageBean.getContent());
                updateSessionBean.setLastTime(lastMessageBean.getShowTime());
                updateSessionBean.setLastMsg(lastMessageBean);
                sendNotifyUpdateEvent(updateSessionBean);
            }

            //进入时有草稿，退出时清除草稿
            LogUtil.d("Has draft:" + isHasDraft);
            if (isHasDraft) {
                updateDraftContent(draftContent);
            }
        }
    }

    /**
     * 发送通知事件
     *
     * @param bean
     */
    private void sendNotifyUpdateEvent(TalkSessionBean bean) {
        bean.setTalkFlag(getTalkFlag());
        IMProxySessionEvent.RefreshSessionEvent event = new IMProxySessionEvent.RefreshSessionEvent();
        event.setSessionBean(bean);
        event.setUpdateData(true);
        BusProvider.getInstance().post(event);
    }

    /**
     * 获取草稿信息
     */
    private void getDraftContent() {
        diskDataStore.get().getDraftContent(getTalkFlag(), new Subscriber<String>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Get draft content failed.");
            }

            @Override
            public void onNext(String s) {
                if (!TextUtils.isEmpty(s)) {
                    isHasDraft = true;
                    getVu().setInputContent(s);
                }
            }
        });
    }

    /**
     * 更新草稿信息
     *
     * @param draftContent
     */
    private void updateDraftContent(final String draftContent) {
        //先同步时间差
        proxyRepository.get().getProxyConfig(Constant.KEY_TIME_DIFFERENT)
                .subscribe(new Subscriber<String>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Sync time difference failed." + e.getMessage());
                        long draftTime = System.currentTimeMillis();
                        if (draftContent != null && !TextUtils.isEmpty(draftContent.trim())) {
                            saveDraft(draftContent, draftTime);
                        } else {
                            clearDraft(draftTime);
                        }
                    }

                    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR1)
                    @Override
                    public void onNext(String s) {
                        long draftTime = 0;
                        if (!TextUtils.isEmpty(s)) {
                            try {
                                draftTime = (SystemClock.elapsedRealtimeNanos() + Long.parseLong(s))
                                        / 1000000;
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                        draftTime = draftTime > 0 ? draftTime : System.currentTimeMillis();
                        LogUtil.d("Draft: content:" + draftContent + ", time:" + draftTime);
                        if (draftContent != null && !TextUtils.isEmpty(draftContent.trim())) {
                            saveDraft(draftContent, draftTime);
                        } else {
                            clearDraft(draftTime);
                        }
                    }
                });
    }

    /**
     * 保存草稿信息
     *
     * @param draftContent
     * @param draftTime
     */
    private void saveDraft(final String draftContent, final long draftTime) {

        diskDataStore.get().saveDraftContent(getTalkFlag(), draftContent, draftTime, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: save draft failed.");
            }

            @Override
            public void onNext(Boolean aBoolean) {
                LogUtil.d("Save draft successfully.");
            }
        });

        //保存草稿失败与否，都退出时，都需要更新会话内容和时间
        TalkSessionBean updateSessionBean = new TalkSessionBean();
        SessionParam param = new SessionParam();
        param.setDraftContent(draftContent);
        param.setDraftTime(draftTime);
        updateSessionBean.setParam(param);
        sendNotifyUpdateEvent(updateSessionBean);
    }

    /**
     * 清除草稿信息
     */
    private void clearDraft(final long draftTime) {
        diskDataStore.get().clearDraftContent(getTalkFlag(), draftTime, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Clear draft failed. " + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                LogUtil.d("Clear draft successfully.");
            }
        });

        //如果消息列表不为空，则显示最后一条消息
        TalkSessionBean updateSessionBean = new TalkSessionBean();
        if (!mMessageList.isEmpty()) {
            //清除草稿后，消息不为空
            TalkMessageBean lastMessageBean = mMessageList.get(mMessageList.size() - 1);
            updateSessionBean.setContent(lastMessageBean.getContent());
            updateSessionBean.setLastTime(lastMessageBean.getShowTime());
            updateSessionBean.setLastMsg(lastMessageBean);
        } else {
            //消息为空，则显示草稿时间
            SessionParam param = new SessionParam();
            param.setDraftContent("");
            param.setDraftTime(draftTime);
            updateSessionBean.setParam(param);
        }
        sendNotifyUpdateEvent(updateSessionBean);
    }

    /**
     * 清除未读消息标识
     */
    private void clearUnReadMsgCount() {

        //清除默认的通知栏消息数
        NotificationUtil.getInstance().clearPNNotification();

        //2017-10-30 wangzhen start modify: Fix the bug 22118 of NullPointException
        if(proxyRepository != null) {
            proxyRepository.get().clearUnReadMsgCount(getTalkFlag(), new Subscriber<Integer>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.d("Error:clear unRead msg flag failed.");
                }

                @Override
                public void onNext(Integer integer) {
                    LogUtil.d("clear unRead msg flag successfully.");
                }
            });
        }
    }

    @Override
    public int getMessageSize() {
        return mMessageList.size();
    }

    @Override
    public boolean sendTextMessage(String message) {
        LogUtil.d("send text message:" + message);
        proxyRepository.get().sendTextMessage(message, mTalkId, isLimit,
                mTalkType == ConstDef.CHAT_TYPE_P2G,
                new Subscriber<TalkMessageBean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("send text error, ERROR:" + e.getMessage());
                    }

                    @Override
                    public void onNext(TalkMessageBean talkMessageBean) {
                        int index = mMessageList.indexOf(talkMessageBean);
                        if (index >= 0) {
                            return;
                        }
                        mMessageList.add(talkMessageBean);
                        mAdapter.notifyDataSetChanged();
                        //跳转
                        getVu().setListSelection(mMessageList.size() - 1);
                    }
                });
        return true;
    }

    @Override
    public void sendFileMessage(List<FileInfo> fileList) {

        LogUtil.d("send file message:" + fileList.toString());
        proxyRepository.get().sendFileMessage(mTalkId, isLimit,
                mTalkType == ConstDef.CHAT_TYPE_P2G, fileList,
                new Subscriber<TalkMessageBean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("send file ERROR:" + e.getMessage());
                    }

                    @Override
                    public void onNext(TalkMessageBean bean) {
                        int index = mMessageList.indexOf(bean);
                        if (index >= 0) {
                            return;
                        }
                        //更新状态
                        mMessageList.add(bean);
                        mAdapter.notifyDataSetChanged();
                        //跳转
                        getVu().setListSelection(mMessageList.size() - 1);
                    }
                });
    }

    @Override
    public void sendVoiceMessage(String path, int seconds) {
        VoiceFileInfo voiceFileInfo = new VoiceFileInfo();
        voiceFileInfo.setAmountOfTime(seconds);
        voiceFileInfo.setFilePath(path);
        voiceFileInfo.setFileName(path.substring(path.lastIndexOf("/") + 1, path.length()));
        voiceFileInfo.setFileSize(FileSizeUtils.getFileSize(path));
        voiceFileInfo.setSuffix("amr");
        voiceFileInfo.setFileType(ConstDef.FILE_TYPE_VOICE);

        //发送文件
        List<FileInfo> fileInfoList = new ArrayList<>();
        fileInfoList.add(voiceFileInfo);
        sendFileMessage(fileInfoList);
    }

    //***********************************************************************
    //*            以下为chatAction启动第三方界面，以及返回结果
    //***********************************************************************
    private Intent buildIntent(OptionBean optionBean) {
        Intent intent = new Intent();
        if (!TextUtils.isEmpty(optionBean.getPkgName())) {
            intent.setClassName(optionBean.getPkgName(), optionBean.getClsName());
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        } else {
            intent.setClassName(getPackageName(), optionBean.getClsName());
        }

        //params
        Map<String, String> params = optionBean.getParam();
        if (params != null && !params.isEmpty()) {
            for (Map.Entry<String, String> entity : params.entrySet()) {
                intent.putExtra(entity.getKey(), entity.getValue());
            }
        }
        intent.putExtra(IntentParam.PARAM_SESSION_ID, mTalkId);
        intent.putExtra(IntentParam.PARAM_SESSION_TYPE, mTalkType);
        return intent;
    }

    @Override
    public void startToOptionView(int position, OptionBean optionBean) {

        LogUtil.d("startToOptionView:" + optionBean.toString());
        Intent intent = buildIntent(optionBean);
        try {
            //捕获异常，防止
            if (optionBean.getRequestCode() > 0) {
                startActivityForResult(intent, optionBean.getRequestCode());
            } else {
                startActivity(intent);
            }
            //恢复面板状态
            getVu().restoreActionState();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void startToAlbum() {
        getVu().restoreActionState();
        //打开相册
        ImagePickerManager.getInstance().openAlbum(this, Constant.RESULT_CODE_ALBUM);
    }

    @Override
    public void startToCamera() {
        getVu().restoreActionState();
        CameraManager.getInstance().startTakePhoto(this, Constant.RESULT_CODE_CAMERA);
    }

    @Override
    public void startToFileExplorer() {
        getVu().restoreActionState();
        FileExplorerManager.getInstance().openFileExplorer(this, Constant.RESULT_CODE_FILE);
    }

    @Override
    public void startToVideo() {
        getVu().restoreActionState();
        VideoRecorderManager.getInstance().startVideoRecorder(this, Constant.RESULT_CODE_VIDEO);
    }

    @Override
    public void startSetting() {
        getVu().restoreActionState();
        try {
            boolean result = IMUiKitProxyImpl.getInstance().openChatSettingCallback(mTalkId, mTalkType);
            if (!result) {
                IntentParam.startSettingActivity(getActivity(), mTalkId, mTalkType);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void startLocation(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startLocation(mTalkId);
    }

    @Override
    public void startCallback(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startCallback(mTalkId);
    }

    @Override
    public void startVoice(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startVoice(mTalkId, mTalkType);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        LogUtil.d("onActivityResult requestCode:" + requestCode);
        if (data == null) {
            LogUtil.w("ERROR: on bad activity result, data is null.");
            return;
        }

        switch (resultCode) {
            case Constant.RESULT_CODE_EMOTION:
                break;
            case Constant.RESULT_CODE_ALBUM:
                handleAlbumResult(data);
                break;
            case Constant.RESULT_CODE_CAMERA:
                handleCameraResult(data);
                break;
            case Constant.RESULT_CODE_AUDIO:
                break;
            case Constant.RESULT_CODE_VIDEO:
                handleVideoRecorderResult(data);
                break;
            case Constant.RESULT_CODE_FILE:
                handleFileResult(data);
                break;
            case Constant.RESULT_CODE_LOCATION:
                break;
            case Constant.RESULT_CODE_WEB:
                break;
            case Constant.RESULT_CODE_FILE_DOWNLOAD:
                handleFileLoadResult(data);
                break;
        }
    }

    /**
     * 相册返回值处理
     *
     * @param data
     */
    private void handleAlbumResult(Intent data) {
        List<ImageThumbBean> thumbBeen = data.getParcelableArrayListExtra(Constant.TAG_PIC_SELECT);
        if (thumbBeen == null || thumbBeen.isEmpty()) {
            LogUtil.e("ERROR: thumb list is null or empty.");
            return;
        }
        List<FileInfo> imageFileInfoList = new ArrayList<>();
        for (ImageThumbBean bean : thumbBeen) {
            FileInfo fileInfo = DataConvertUtils.mapImageFileInfo(bean);
            if (fileInfo != null) {
                imageFileInfoList.add(fileInfo);
            }
        }
        sendFileMessage(imageFileInfoList);
    }

    /**
     * 拍照返回值处理
     *
     * @param data
     */
    private void handleCameraResult(Intent data) {
        PhotoThumbBean thumbBean = data.getParcelableExtra(Constant.TAG_PHOTO);
        if (thumbBean == null) {
            LogUtil.e("ERROR: thumb list is null or empty.");
            return;
        }
        List<FileInfo> photoFileInfos = new ArrayList<>();
        FileInfo photoFileInfo = DataConvertUtils.mapPhotoFileInfo(thumbBean);
        if (photoFileInfo != null) {
            photoFileInfos.add(photoFileInfo);
        }
        sendFileMessage(photoFileInfos);
    }

    /**
     * 视频录制返回值处理
     *
     * @param data
     */
    private void handleVideoRecorderResult(Intent data) {
        VideoItem videoItem = data.getParcelableExtra(Constant.TAG_VIDEO_RECORDER);
        if (videoItem == null) {
            LogUtil.e("ERROR: videoItem is null.");
            return;
        }
        FileInfo videoFileInfo = DataConvertUtils.mapVideoFileInfo(videoItem);
        List<FileInfo> videoFileInfoList = new ArrayList<>();
        videoFileInfoList.add(videoFileInfo);
        sendFileMessage(videoFileInfoList);
    }

    /**
     * 文件选择返回值处理
     *
     * @param data
     */
    private void handleFileResult(Intent data) {
        List<FileItem> fileItemList = data.getParcelableArrayListExtra(Constant.TAG_FILE_SELECT);
        if (fileItemList == null || fileItemList.isEmpty()) {
            LogUtil.e("ERROR: file list is null or empty.");
            return;
        }
        List<FileInfo> fileInfoList = new ArrayList<>();
        for (FileItem bean : fileItemList) {
            FileInfo fileInfo = DataConvertUtils.mapFileInfo(bean);
            if (fileInfo != null) {
                fileInfoList.add(fileInfo);
            }
        }
        sendFileMessage(fileInfoList);
    }

    /**
     * 文件返回值结果处理
     *
     * @param data
     */
    private void handleFileLoadResult(Intent data) {
        Bundle bundle = data.getExtras();
        if (bundle == null) {
            LogUtil.e("ERROR: hand file load result, bundle is null.");
            return;
        }

        long msgId = bundle.getLong(Constant.TAG_MSGID, -1);
        int state = bundle.getInt(Constant.TAG_FILE_STATE, ConstDef.INACTIVE);
        long translateSize = bundle.getLong(Constant.TAG_FILE_TRANS_SIZE, 0);

        int position = mathMessage(msgId);
        if (position < 0) {
            LogUtil.w("Result file downloadFile code, position:" + position);
            return;
        }

        TalkMessageBean bean = mMessageList.get(position);
        NormalFileInfo fileInfo = (NormalFileInfo) bean.getFileInfo();
        if (fileInfo != null) {
            fileInfo.setTranslateSize(translateSize);
            fileInfo.setFileState(state);
            mAdapter.updateItem(position, bean);
        }
    }
    //***********************************************************************
    //*            启动第三方界面结束
    //***********************************************************************

    /**
     * 更新文件传输进度
     */
    private void updateFileTranslateSize() {
        FileInfo fileInfo;
        for (TalkMessageBean bean : mMessageList) {
            fileInfo = bean.getFileInfo();
            if (fileInfo != null
                    && (bean.getMessageType() == ConstDef.MSG_TYPE_FILE
                    || bean.getMessageType() == ConstDef.MSG_TYPE_VIDEO)
                    && bean.getFileInfo() != null
                    && bean.getFileInfo().getTranslateSize() != 0
                    && bean.getFileInfo().getTranslateSize() != bean.getFileInfo().getFileSize()) {
                TranslateInfoUtil.putInfo(bean.getId(), fileInfo.getTranslateSize());
            }
        }
    }

    @Override
    public void onActivityFocusChanged(boolean hasFocus) {
        super.onActivityFocusChanged(hasFocus);
        LogUtil.d("Window focus changed " + hasFocus);
        if (!hasFocus) {
            AudioPlayerManagerProxy.stopPlay();
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        IMProxySessionEvent.ClearUnReadMsgEvent event =
                new IMProxySessionEvent.ClearUnReadMsgEvent();
        event.setTalkId(getTalkFlag());
        BusProvider.getInstance().post(event);
		MediaManger.getInstance().restoreAudioMode();
        LogUtil.d("onPause.");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        LogUtil.d("onDestroy.");

        //清除未读消息
        clearUnReadMsgCount();

        //通知更新最有一条会话消息
        notifyUpdateLastSession();

        //更新文件传输进度
        updateFileTranslateSize();

        AudioPlayerManagerProxy.stopPlay();
        AudioListenerManager.getInstance().clearAll();

        //TranslateInfoUtil.clear();

        //清除缓存信息
        ImageCache.getInstance().clearAllCache();
        Glide.get(this).clearMemory();

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 更改消息状态
     *
     * @param bean 消息体
     */
    private void changeMsgState(TalkMessageBean bean) {

        int position = mathMessage(bean.getId());
        if (position < 0) {
            LogUtil.e("ERROR: Change Msg State, position:" + position);
            return;
        }

        TalkMessageBean messageBean = mMessageList.get(position);
        if (messageBean != null) {
            messageBean.setMessageState(bean.getMessageState());
            messageBean.setFailCode(bean.getFailCode());
            messageBean.setContent(bean.getContent());
            mAdapter.updateItem(position, messageBean);
        }
    }

    /**
     * 收到新消息，更新列表
     *
     * @param talkMessageList
     */
    private void addNewMessageToList(List<TalkMessageBean> talkMessageList) {

        int scrolledX = getVu().getListView().getScrollX();
        int scrolledY = getVu().getListView().getScrollY();

        mMessageList.addAll(talkMessageList);
        mAdapter.notifyDataSetChanged();

        int lastVisiblePosition = getVu().getListView().getLastVisiblePosition();
        if (mMessageList.size() > PAGE_SIZE) {

            if (lastVisiblePosition <= mMessageList.size() - 1
                    && lastVisiblePosition > mMessageList.size() - 1 - PAGE_SIZE) {
                //15条消息之内
                getVu().setListSelection(mMessageList.size() - 1);
            } else {
                //15条消息之外
                //getVu().setListSelection(firstVisiblePosition + 1);
                getVu().scrollTo(scrolledX, scrolledY);
            }
        } else {
            getVu().setListSelection(mMessageList.size() - 1);
        }
    }

    //*****************************************************************
    //*     以下为OTTO事件处理模块，该模块所有的回调均在主线程中
    //*****************************************************************
    @Subscribe
    public void onReceiveNewMessage(IMProxyMessageEvent.ReceiveNewMessageEvent event) {

        LogUtil.d("EVENT: Receive a new message.");

        //消息内容校验
        List<TalkMessageBean> talkMessageList = event.getTalkMessageList();
        if (talkMessageList == null || talkMessageList.isEmpty()) {
            LogUtil.w("receive new message event, but messageBean is null or empty.");
            return;
        }
        if (mTalkType == ConstDef.CHAT_TYPE_P2G) {
            for (TalkMessageBean bean : talkMessageList) {
                Log.d("ChatListActivity", "bean.MsgType:" + bean.getMessageType());
                if (bean.getMessageType() == ConstDef.MSG_TYPE_PRESENTATION) {
                    if (IMUiKitProxyImpl.getInstance().isShowGroupSetting(mTalkId)) {
                        getVu().setToolBarAddVisibility(View.VISIBLE);
                    } else {
                        getVu().setToolBarAddVisibility(View.INVISIBLE);
                    }
                }
            }
        }
        //发送方账号校验
        String from = event.getMsgAccount();
        if (TextUtils.isEmpty(from) || !TextUtils.equals(from, mTalkId)) {
            LogUtil.w("receive new message event, but account is empty.");
            return;
        }

        addNewMessageToList(talkMessageList);
    }

    @Subscribe
    public void onDeleteMessage(IMProxyMessageEvent.DeleteMessageEvent event) {

        LogUtil.d("EVENT: Delete a new message.");

        TalkMessageBean talkMessageBean = event.getTalkMessageBean();

        //如果当时语音正在播放，则停止当前播放语音
        FileInfo fileInfo = talkMessageBean.getFileInfo();
        if (fileInfo != null && fileInfo.getFileType() == ConstDef.FILE_TYPE_VOICE) {
            AudioPlayerManagerProxy.stopPlay();
        }

        mMessageList.remove(talkMessageBean);
        mAdapter.notifyDataSetChanged();
    }

    @Subscribe
    public void onRefreshMessage(IMProxyMessageEvent.RefreshMessageEvent event) {

        LogUtil.d("EVENT: refresh single message.");

        TalkMessageBean bean = event.getTalkMessageBean();
        if (bean == null) {
            LogUtil.e("refresh single message, but messageBean is null.");
            return;
        }
        changeMsgState(bean);
    }

    @Subscribe
    public void onRefreshMessageList(IMProxyMessageEvent.RefreshMessageListEvent event) {
        LogUtil.d("EVENT: refresh message list.");

        if (event.isNeedClearInput()) {
            getVu().setInputContent("");
        }
        loadMessageList(0);
    }

    @Subscribe
    public void onRecallMessage(final IMProxyMessageEvent.RecallMessageEvent event) {
        TalkMessageBean bean = event.getTalkMessageBean();
        if (bean == null) {
            LogUtil.e("refresh single message, but messageBean is null.");
            return;
        }

        int position = mathMessage(bean.getId());
        if (position < 0) {
            LogUtil.w("math message failed." + bean.getId());
            return;
        }

        mMessageList.set(position, bean);
        mAdapter.updateItem(position, bean);
    }

    @Subscribe
    public void onSendFileComplete(IMProxyFileEvent.SendFileFinishedEvent event) {
        LogUtil.d("EVENT: Send file complete.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Send file completed. position:" + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file completed, but fileInfo is null.");
            return;
        }

        fileInfo.setFileState(ConstDef.DONE);
        updateFileInfo(position, fileInfo);


        if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VOICE) {
            LogUtil.d("Voice send finished.");
            return;
        }
        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onSendFilePause(IMProxyFileEvent.SendFilePausedEvent event) {

        LogUtil.d("EVENT: Send file pause.");
    }

    @Subscribe
    public void onSendFileProgressUpdate(IMProxyFileEvent.SendFileProgressUpdateEvent event) {

        LogUtil.d("EVENT: Send file progress update.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Send file progress update, position: " + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file progress update, but post fileInfo is null.");
            return;
        }

        //视频和图片，只有在上传原图时，才更新进度
        if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO ||
                fileInfo.getFileType() == ConstDef.FILE_TYPE_VIDEO) {

            int fileProperty = event.getFileProperty();
            if (fileProperty == ConstDef.IS_RAW) {
                int percent = event.getPercent();
                fileInfo.setTranslatePercent(percent);
            }
        }
        FileInfo eventFileInfo = event.getFileInfo();
        if (eventFileInfo != null) {
            fileInfo.setTranslateSize(eventFileInfo.getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.STATE_SENDING);
        updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onSendFileFailed(IMProxyFileEvent.SendFileFailedEvent event) {
        LogUtil.d("EVENT: Send file failed.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: send file failed, position:" + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file failed , but fileInfo is null.");
            return;
        }
        LogUtil.w("Send failed file info:" + fileInfo.toString());

        fileInfo.setFileState(ConstDef.FAIL);
        updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onReceiveFileComplete(IMProxyFileEvent.ReceiveFileFinishedEvent event) {
        LogUtil.d("EVENT: Receive file complete.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Receive file completed, position:" + position);
            return;
        }


        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file completed, but fileInfo is null.");
            return;
        }

        //更新状态
        fileInfo.setFileState(ConstDef.DONE);
        updateFileInfo(position, fileInfo);

        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onReceiveFilePause(IMProxyFileEvent.ReceiveFilePausedEvent event) {
        LogUtil.d("EVENT: Receive file pause.");
        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Receive file PAUSE, position: " + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file PAUSE, but post fileInfo is null.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.PAUSE);
        updateFileInfo(position, fileInfo);

    }

    @Subscribe
    public void onReceiveFileProgressUpdate(IMProxyFileEvent.ReceiveFileProgressUpdateEvent event) {
        LogUtil.d("EVENT: Receive file progress update.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Send file progress update, position: " + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file progress update, but post fileInfo is null.");
            return;
        }

        if (fileInfo.getFileType() != ConstDef.FILE_TYPE_NORMAL) {
            LogUtil.d("No need update file progress.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.LOADING);
        updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onReceiveFileFailed(IMProxyFileEvent.ReceiveFileFailedEvent event) {
        LogUtil.d("EVENT: Receive file failed.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Receive file Failed, position: " + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file Failed, but post fileInfo is null.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.FAIL);
        updateFileInfo(position, fileInfo);
        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onRefreshGroupEvent(IMContactEvent.RefreshGroupEvent event) {
        LogUtil.d("EVENT: Receive group refresh event.");

        String groupId = event.getAccount();
        //刷新对应的会话列表
        //刷新对应的聊天对象，刷新对应导航栏标题
        mAdapter.notifyDataSetChanged();
        if (mTalkId.equals(groupId)) {
            //获取群信息
            IMUiKitProxyImpl.getInstance().getAccount(groupId, ConstDef.CHAT_TYPE_P2G, new Subscriber<AccountBean>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.d("获取群信息失败");
                }

                @Override
                public void onNext(AccountBean accountBean) {
                    getVu().setTitle(accountBean.getDisplayName());
                }
            });

        }
    }

    @Subscribe
    public void onDissolutionGroupEvent(IMContactEvent.DissolutionGroupEvent event) {
        LogUtil.d("EVENT: dissolution group event.");

        getActivity().finish();
    }

    @Subscribe
    public void onDataTimeChanged(IMProxyEvent.DataTimeChangedEvent event) {

        LogUtil.d("EVENT: Receive date time changed event.");

        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }

    /**
     * 事件和消息匹配
     *
     * @param messageId 事件ID
     * @return
     */
    private int mathMessage(long messageId) {
        int size = mMessageList.size();
        for (int i = size - 1; i >= 0; i--) {
            if (messageId == mMessageList.get(i).getId()) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 更新消息文件信息
     *
     * @param position
     * @param fileInfo
     */
    private void updateFileInfo(int position, FileInfo fileInfo) {
        if (position < 0 || position > mMessageList.size()) {
            return;
        }
        mMessageList.get(position).setFileInfo(fileInfo);
        mAdapter.updateItem(position, mMessageList.get(position));
    }

    /**
     * ------------------------------------语音播放回调-----------------start
     */
    @Override
    public void playingCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }
        long msgId = (long) args[0];
        mAdapter.startAnimation(filePath, msgId);
    }

    @Override
    public void playingStopCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        mAdapter.stopAnimation(filePath, msgId);
    }

    @Override
    public void playingErrorCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        mAdapter.stopAnimation(filePath, msgId);
    }

    @Override
    public void playingCompletionCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        boolean isNext = (boolean) args[1];
        mAdapter.playNextAudio(filePath, msgId, isNext);
    }
    /**
     * ------------------------------------语音播放回调-----------------end
     */
}
