package com.xdja.im.lib.action;

import android.content.Context;
import android.support.annotation.Nullable;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.xdja.im.lib.action.listener.IChatActionCallback;
import com.xdja.im.lib.audio.widget.AudioRecorderButton;
import com.xdja.im.lib.emoji.utils.EmotionUtils;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.view.PastEditText;

/**
 * @Package: com.xdja.im.uikit.view
 * @Author: xdjaxa
 * @Creation: 2017-04-12 11:21
 * @Version V1.0
 * @Description:消息输入操作栏
 */
public class ChatInputView extends LinearLayout {

    /**
     * 输入框最大输入长度
     */
    private final int MAX_INPUT_SIZE = 2048;

    /**
     * 更多面板
     */
    private ImageView mMoreImg;
    /**
     * 选中闪信/表情
     */
    private ImageView mShanImg;
    /**
     * 语音文字输入切换
     */
    private CheckBox mInputSwitchChx;
    /**
     * 文字输入框
     */
    private PastEditText mInputEdt;
    /**
     * 录音按钮
     */
    private AudioRecorderButton mRecorderBtn;
    /**
     * 发送按钮
     */
    private Button mSendBtn;

    public ChatInputView(Context context) {
        super(context);
        initView();
    }

    public ChatInputView(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    public ChatInputView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView();
    }

    private void initView() {
        View.inflate(getContext(), R.layout.im_uikit_layout_chat_operate, this);

        mMoreImg = (ImageView) findViewById(R.id.img_more_action);
        mShanImg = (ImageView) findViewById(R.id.img_shan_action);
        mInputSwitchChx = (CheckBox) findViewById(R.id.chx_input_switch);
        mInputEdt = (PastEditText) findViewById(R.id.edt_input_text);
        mRecorderBtn = (AudioRecorderButton) findViewById(R.id.btn_input_voice);
        mSendBtn = (Button) findViewById(R.id.btn_send);
    }

    boolean isTouchEditText;

    public void registerActionViewCallBack(final IChatActionCallback callback) {
        //文本输入监听事件
        mInputEdt.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                int length = s.toString().length();
                if (length >= MAX_INPUT_SIZE) {
                    s = s.delete(MAX_INPUT_SIZE, length);
                }

                // 文本长度为0时最右边显示为语音切换
                if (s.length() == 0) {
                    mSendBtn.setVisibility(GONE);
                    mMoreImg.setVisibility(View.VISIBLE);
                } else {
                    mSendBtn.setVisibility(View.VISIBLE);
                    mMoreImg.setVisibility(View.GONE);
                }
                /*if (s.length() == 0) {
                    mSendBtn.setEnabled(false);
                } else {
                    mSendBtn.setEnabled(true);
                }*/
            }
        });
        //文本编辑框点击监听事件
        mInputEdt.setOnTouchListener(new OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                mInputEdt.setFocusable(true);
                mInputEdt.setFocusableInTouchMode(true);
                mInputEdt.requestFocus();
                if (!TextUtils.isEmpty(getInputText())) {
                    mSendBtn.setVisibility(View.VISIBLE);
                    mMoreImg.setVisibility(View.GONE);
                }
                if (event.getAction() == MotionEvent.ACTION_UP) {
                    isTouchEditText = true;
                    mMoreImg.setImageResource(R.mipmap.im_uikit_action_chat_panel_normal);
                    callback.onInputEareTouchCallBack();
                }
                return false;
            }
        });

        //文本语音输入切换监听
        mInputSwitchChx.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
                switchToInputEdt(!isChecked);
                callback.onInputCheckChanged(!isChecked);
            }
        });

        //闪信/文本切换
        mShanImg.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                setMoreState(false);
                switchToInputEdt(true);
                mInputSwitchChx.setChecked(false);
                callback.onShanCheckChanged();
            }
        });

        //更多选择切换监听
        mMoreImg.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                setShanState(false);
                switchToInputEdt(true);
                mInputSwitchChx.setChecked(false);
                callback.onMoreCheckChanged();
            }
        });

        //发送按钮点击事件监听
        mSendBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                callback.onSendTextCallBack();
            }
        });

        //录音完成回调事件监听
        mRecorderBtn.setAudioFinishRecorderListener(new AudioRecorderButton.AudioFinishRecorderListener() {
            @Override
            public void onFinish(float seconds, String filePath) {
                callback.onRecorderFinish((int) seconds, filePath);
            }
        });
    }

    /**
     * 切换至文本输入
     *
     * @param isSwitch
     */
    public void switchToInputEdt(boolean isSwitch) {
        if (isSwitch) {
            mInputEdt.setVisibility(VISIBLE);
            mRecorderBtn.setVisibility(GONE);
            mInputEdt.requestFocus();
            if (getInputEdt().length() == 0) {
                mSendBtn.setVisibility(GONE);
                mMoreImg.setVisibility(View.VISIBLE);
            } else {
                mSendBtn.setVisibility(View.VISIBLE);
                mMoreImg.setVisibility(View.GONE);
            }
        } else {
            mInputEdt.setVisibility(GONE);
            mRecorderBtn.setVisibility(VISIBLE);
            mSendBtn.setVisibility(View.GONE);
            mMoreImg.setVisibility(View.VISIBLE);
        }
    }

    /**
     * 获取输入的文字信息
     */
    public String getInputText() {
        return mInputEdt.getText().toString();
    }

    /**
     * 设置输入显示信息
     *
     * @param text 文本信息
     */
    public void setInputText(CharSequence text) {
        mInputEdt.setText(EmotionUtils.formatNormalSpanContent(text, getContext()));
        mInputEdt.setFocusable(true);
        mInputEdt.setFocusableInTouchMode(true);
        mInputEdt.requestFocus();
    }

    /**
     * 获取输入框，设置给表情View
     */
    public EditText getInputEdt() {
        return mInputEdt;
    }

    public void setMoreState(boolean isChecked) {
        if (!TextUtils.isEmpty(getInputText()) && mInputEdt.isShown()) {
            mMoreImg.setVisibility(View.GONE);
            mSendBtn.setVisibility(View.VISIBLE);
            mInputEdt.requestFocus();
        } else {
            mSendBtn.setVisibility(View.GONE);
            mMoreImg.setVisibility(View.VISIBLE);

            if (isChecked) {
                mMoreImg.setImageResource(R.mipmap.im_uikit_actioin_chat_panel_press);
            } else {
                mMoreImg.setImageResource(R.mipmap.im_uikit_action_chat_panel_normal);
            }
        }
    }

    public void setShanState(boolean isChecked) {
        if (isChecked) {
            mShanImg.setImageResource(R.mipmap.im_uikit_action_chat_keyinput_normal);
        } else {
            mShanImg.setImageResource(R.mipmap.im_uikit_action_chat_emotion_normal);
        }
    }
}
