package com.xdja.im.lib.file.ui.adapter;

import android.content.Context;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.Toast;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.lib.file.FileExplorerConfig;
import com.xdja.im.lib.file.FileExplorerManager;
import com.xdja.im.lib.file.bean.FileCollections;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.file.bean.FileItemNode;
import com.xdja.im.lib.file.utils.FileConstant;
import com.xdja.im.lib.file.utils.NodeUtils;
import com.xdja.im.lib.file.widget.recyclerview.BaseMultiItemTypeAdapter;
import com.xdja.im.lib.file.widget.recyclerview.ViewHolder;
import com.xdja.im.uikit.R;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class CommonRecyclerAdapter extends BaseMultiItemTypeAdapter<FileItemNode>
        implements BaseMultiItemTypeAdapter.OnItemClickListener {

    private long lastClickTime = 0;
    private RecyclerView recyclerView;
    private Map<String, List<FileItem>> dataMaps;
    private CommonSpanSizeLookup spanSizeLookup;

    public CommonRecyclerAdapter(Context context,
                                 List<FileItemNode> dataSources,
                                 Map<String, List<FileItem>> dataMaps) {
        super(context, dataSources);
        this.dataMaps = dataMaps;
        this.dataSources = NodeUtils.convertData(dataMaps);
        setOnItemClickListener(this);
    }

    public void updateAllItems() {
        dataSources = NodeUtils.convertData(dataMaps);
        notifyDataSetChanged();
    }

    @Override
    public void onItemClick(View view, ViewHolder holder, int position) {
        switch (getItemViewType(position)) {
            case FileConstant.PARENT_DIR:
                long currentTime = System.currentTimeMillis();
                if (currentTime - lastClickTime > FileConstant.MIN_CLICK_DELAY_TIME) {
                    lastClickTime = currentTime;
                    collapseOrExpand(position);
                }
                break;
            default:
                clickChildLeaf(holder, position);
                break;
        }
    }

    /**
     * 点击折叠和打开
     *
     * @param position
     */
    protected void collapseOrExpand(int position) {
        FileItemNode bean = dataSources.get(position);
        String title = (String) bean.getValue();
        List<FileItem> childLists = bean.getChildList();

        List<FileItem> originChilds = dataMaps.get(title);
        if (bean.isExpand()) {
            //折叠
            for (FileItem fInfo : childLists) {
                FileItemNode<FileItem> deleteBean = new FileItemNode();
                deleteBean.setValue(fInfo);
                dataSources.remove(deleteBean);
            }
            bean.setExpand(!bean.isExpand());
            notifyItemRangeRemoved(position + 1, childLists.size());
        } else {
            //显示
            List<FileItemNode> addChildItem = new ArrayList<>();
            for (FileItem fInfo : originChilds) {
                FileItemNode<FileItem> addBean = new FileItemNode();
                addBean.setValue(fInfo);
                addBean.setChildList(null);
                addChildItem.add(addBean);
            }
            dataSources.addAll(position + 1, addChildItem);
            bean.setExpand(!bean.isExpand());
            notifyItemRangeInserted(position + 1, addChildItem.size());
        }
        notifyItemChanged(position);
    }

    /**
     * 点击某个文件进行选择操作
     *
     * @param holder
     * @param position
     */
    protected void clickChildLeaf(ViewHolder holder, int position) {
        FileItem fileInfo = (FileItem) dataSources.get(position).getValue();

        if (fileInfo != null) {

            //文件选择个数限制
            if (!fileInfo.isSelected()) {
                if (FileCollections.getInstance().getSelectedFileCount() >= FileConstant.MAX_SELECT_COUNT) {
                    CommonTool.showToast(String.format(context.getString(R.string.im_uikit_select_file_hint),
                            FileConstant.MAX_SELECT_COUNT));
                    return;
                }
            }

            //通知刷新界面
            fileInfo.setSelected(!fileInfo.isSelected());
            if (fileInfo.isSelected()) {
                FileCollections.getInstance().putFileToSelectedCache(
                        fileInfo.getFilePath() + fileInfo.getModifiedDate(),
                        fileInfo);
            } else {
                FileCollections.getInstance().removeToSelectedCache(
                        fileInfo.getFilePath() + fileInfo.getModifiedDate());
            }
            IMProxyEvent.FileSelectedEvent event = new IMProxyEvent.FileSelectedEvent(fileInfo);
            BusProvider.getInstance().post(event);

            holder.setChecked(R.id.chx_file_fragment_select, fileInfo.isSelected());
        }
    }

    @Override
    public boolean onItemLongClick(View view, ViewHolder holder, int position) {
        switch (getItemViewType(position)) {
            case FileConstant.PARENT_DIR:
                break;
            default:
                Toast.makeText(context, "click long leaf position=" + position, Toast.LENGTH_SHORT).show();
                break;
        }
        return false;
    }

    @Override
    public int getItemViewType(int position) {
        if (dataSources.get(position).isLeaf()) {
            FileItem fileInfo = (FileItem) dataSources.get(position).getValue();

            //图片显示方式
            int photoStyle = FileExplorerManager.getInstance().getConfig().getPhotoStyle();
            if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO && photoStyle == FileExplorerConfig.GRID) {
                return FileConstant.CHILD_LEAF;
            }
            return FileConstant.CHILD_LEAF_FILE;
        }
        return FileConstant.PARENT_DIR;
    }

    public void setRecyclerView(RecyclerView recyclerView) {
        this.recyclerView = recyclerView;
    }

    public void setLayoutManager(int col) {
        GridLayoutManager layoutManager = new GridLayoutManager(context, col);
        layoutManager.setSpanSizeLookup(getSpanSizeLookup());
        recyclerView.setLayoutManager(layoutManager);
    }

    private CommonSpanSizeLookup getSpanSizeLookup() {
        if (spanSizeLookup == null) {
            spanSizeLookup = new CommonSpanSizeLookup();
        }
        return spanSizeLookup;
    }

    private class CommonSpanSizeLookup extends GridLayoutManager.SpanSizeLookup {

        @Override
        public int getSpanSize(int position) {
            int viewType = getItemViewType(position);
            if (viewType == FileConstant.PARENT_DIR
                    || viewType == FileConstant.CHILD_LEAF_FILE) {
                return FileConstant.PARENT_DIR_WIDTH;
            }
            return FileConstant.CHILD_LEAF_WIDTH;
        }
    }
}
