package com.xdja.im.lib.camera.utils;

import com.xdja.im.common.utils.FileSizeUtils;
import com.xdja.im.lib.album.ImagePickerConfig;
import com.xdja.im.lib.camera.CameraManager;
import com.xdja.im.lib.camera.bean.PhotoItem;
import com.xdja.im.lib.camera.bean.PhotoThumbBean;

import java.io.File;
import java.util.UUID;

import rx.Observable;
import rx.functions.Func1;
import rx.functions.Func3;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.lib.album.utils
 * @Author: xdjaxa
 * @Creation: 2017-04-24 15:55
 * @Version V1.0
 * @Description:数据处理工具类
 */
public class DataTools {

    /**
     * 生成图片缩略图信息
     *
     * @param photo 图片原信息
     * @return 图片信息列表：至少包含图片缩略图和高清缩略图，如果需要发送原图，则含有原图信息
     */
    public static Observable<PhotoThumbBean> createImageThumb(PhotoItem photo) {

        return Observable.just(photo)
                .flatMap(new Func1<PhotoItem, Observable<PhotoThumbBean>>() {
                    @Override
                    public Observable<PhotoThumbBean> call(PhotoItem photo) {
                        return createThumbNails(photo);
                    }
                });
    }

    /**
     * 单张图片生成缩略图
     *
     * @param photo
     * @return
     */
    private static Observable<PhotoThumbBean> createThumbNails(final PhotoItem photo) {

        //缩略图文件保存父路径
        String photoRootPath = CameraManager.getInstance().getConfig().getCachePath();
        //文件名称
        final String baseFileName = UUID.randomUUID().toString();
        final String thumbName = ImagePickerConfig.THUMBNAIL_FILE_PREFIX + baseFileName;
        final String hdThumbName = ImagePickerConfig.THUMBNAIL_FILE_PREFIX + baseFileName + "_hd";

        return Observable.zip(
                //高清缩略图
                createHDThumbnail(photo.getFilePath(), new File(photoRootPath, hdThumbName)),
                //缩略图
                createThumbnail(photo.getFilePath(), new File(photoRootPath, thumbName)),
                //原图
                createRawFilePath(photo.getFilePath(), new File(photoRootPath, baseFileName), false),
                //生成实体类
                new Func3<String, String, String, PhotoThumbBean>() {

                    @Override
                    public PhotoThumbBean call(String hdPath, String thumbPath, String rawPath) {
                        PhotoThumbBean photoBean = new PhotoThumbBean();
                        photoBean.setThumbName(thumbName);
                        photoBean.setThumbPath(thumbPath);
                        photoBean.setThumbSize(FileSizeUtils.getFileSize(thumbPath));

                        photoBean.setHdThumbName(hdThumbName);
                        photoBean.setHdThumbPath(hdPath);
                        photoBean.setHdThumbSize(FileSizeUtils.getFileSize(hdPath));
                        if(photo.isbOriginal()){
                            photoBean.setRawName(photo.getFileName());
                            photoBean.setRawPath(rawPath);
                            photoBean.setRawSize(FileSizeUtils.getFileSize(rawPath));
                        }
                        return photoBean;
                    }
                }
        );
    }

    /**
     * 高清缩略图处理
     *
     * @param originalPath
     * @param thumbFile
     * @return
     */
    private static Observable<String> createHDThumbnail(final String originalPath,
                                                        final File thumbFile) {

        return Observable.just(originalPath)
                .subscribeOn(Schedulers.newThread())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String path) {
                        return ThumbnailUtils.createImageThumbnail(path,
                                ThumbnailUtils.KIND_HD, thumbFile);
                    }
                });
    }

    /**
     * 缩略图处理
     *
     * @param originalPath
     * @param thumbFile
     */
    private static Observable<String> createThumbnail(final String originalPath,
                                                      final File thumbFile) {
        return Observable.just(originalPath)
                .subscribeOn(Schedulers.newThread())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String path) {
                        return ThumbnailUtils.createImageThumbnail(path,
                                ThumbnailUtils.KIND_TH, thumbFile);
                    }
                });
    }

    /**
     * 原图处理
     *
     * @param originalPath
     * @param filePath
     * @param isOriginalPic
     * @return
     */
    private static Observable<String> createRawFilePath(final String originalPath,
                                                        final File filePath,
                                                        final boolean isOriginalPic) {
        return Observable.just(originalPath)
                .subscribeOn(Schedulers.io())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String originalPath) {
                        //将原图拷贝至缓存目录下
                        //if (isOriginalPic) {
                        //    return ThumbUtils.copyFile(originalPath, filePath);
                        //} else {
                        return originalPath;
                        //}
                    }
                });
    }
}
