package com.xdja.im.uikit.ui.activity;

import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;

import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.SystemHelper;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyFileEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.NormalFileInfo;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.presenter.FileDownloadPresenter;
import com.xdja.im.uikit.ui.view.ViewFileDownload;
import com.xdja.im.uikit.ui.view.vu.FileDownloadVu;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.uikit.ui.activity
 * @Author: xdjaxa
 * @Creation: 2017-05-22 15:47
 * @Version V1.0
 * @Description:
 */
public class FileDownloadActivity extends ImBaseActivity<FileDownloadPresenter, FileDownloadVu>
        implements FileDownloadPresenter {

    @Inject
    Lazy<ProxyRepository> proxyRepository;

    /**
     * 当前消息信息
     */
    private TalkMessageBean mCurMessageBean;
    /**
     * 当前消息中的文件信息
     */
    private NormalFileInfo mCurFileInfo;

    @NonNull
    @Override
    protected Class<? extends FileDownloadVu> getVuClass() {
        return ViewFileDownload.class;
    }

    @NonNull
    @Override
    protected FileDownloadPresenter getPresenter() {
        return this;
    }


    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (mComponent == null) {
            return;
        }

        //初始化注入
        mComponent.inject(this);
        BusProvider.getInstance().register(this);

        Intent intent = getIntent();
        if (intent == null) {
            LogUtil.e("ERROR: Intent is null.");
            CommonTool.showToast(R.string.im_uikit_file_open_failed);
            return;
        }

        long msgId = intent.getLongExtra(Constant.TAG_MSGID, -1);
        if (msgId < 0) {
            LogUtil.e("ERROR: Message id ." + msgId);
            CommonTool.showToast(R.string.im_uikit_file_open_failed);
            return;
        }
        long translateSize = intent.getLongExtra(Constant.TAG_FILE_TRANS_SIZE, 0);

        loadTalkMessageInfo(msgId, translateSize);
    }

    /**
     * 获取消息信息
     */
    private void loadTalkMessageInfo(long msgId, final long translateSize) {

        proxyRepository.get().getTalkMessage(msgId, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Get message entity failed.");
                CommonTool.showToast(R.string.im_uikit_file_open_failed);
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {

                mCurMessageBean = talkMessageBean;
                mCurFileInfo = (NormalFileInfo) talkMessageBean.getFileInfo();
                if (mCurFileInfo == null) {
                    CommonTool.showToast(R.string.im_uikit_file_open_failed);
                    return;
                }


                //同步进度
                mCurFileInfo.setTranslateSize(Math.max(translateSize, mCurFileInfo.getTranslateSize()));

                getVu().setData();

                if (mCurFileInfo.getFileState() != ConstDef.PAUSE &&
                        mCurFileInfo.getFileState() != ConstDef.DONE) {

                    downLoadFile(true);
                }
            }
        });
    }

    /**
     * 下载文件之前，监测网络连接状态
     */
    private void checkNetStatue() {
        checkNetType(new OnNetCheckCallback() {
            @Override
            public void onResult(boolean isCanceled) {
                //true:取消下载 false：继续下载
                if (isCanceled) {
                    LogUtil.d("Canceled download file!");
                    finish();
                } else {
                    //开始下载文件
                    conformDownload();
                }
            }

            @Override
            protected String getContent() {
                return String.format(getString(R.string.im_uikit_mobile_data_hint_contnet),
                        getString(R.string.im_uikit_download));
            }
        });
    }

    /**
     * 确定下载文件
     */
    private void conformDownload() {
        if (mCurFileInfo.getFileState() == ConstDef.INACTIVE) {
            startDownloadFile();
        } else {
            resumeDownloadFile();
        }
    }

    @Override
    public void downLoadFile(final boolean flag) {
        if (mCurFileInfo.getFileState() == ConstDef.LOADING) {
            return;
        }
        if (!flag) {
            getVu().setCtrlBtnText(R.string.im_uikit_pause_download);
        }

        checkNetStatue();
    }

    @Override
    public void startDownloadFile() {
        if (mCurMessageBean == null) {
            return;
        }
        List<FileInfo> fileInfos = new ArrayList<>();
        fileInfos.add(mCurFileInfo);
        mCurFileInfo.setFileState(ConstDef.LOADING);
        proxyRepository.get().downloadFile(fileInfos, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
                LogUtil.d("downloadFile file finished.");
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("ERROR : downloadFile file error.");
            }

            @Override
            public void onNext(Integer integer) {
                LogUtil.d("downloadFile file start.");
            }
        });
    }

    @Override
    public void pauseDownloadFile() {
        if (mCurFileInfo == null) {
            LogUtil.e("ERROR: pause download file, but file info is null.");
            return;
        }
        mCurFileInfo.setFileState(ConstDef.PAUSE);
        proxyRepository.get().pauseFileReceiving(mCurFileInfo, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
                LogUtil.d("Pause downloadFile file finished.");
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("ERROR : pause downloadFile file error.");
            }

            @Override
            public void onNext(Integer integer) {
                LogUtil.d("Pause downloadFile file start.");
            }
        });
    }

    @Override
    public void resumeDownloadFile() {
        if (mCurFileInfo == null) {
            return;
        }
        mCurFileInfo.setFileState(ConstDef.LOADING);
        proxyRepository.get().resumeFileReceive(mCurFileInfo, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
                LogUtil.d("Resume downloadFile file finished.");
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("ERROR : resume downloadFile file error.");
            }

            @Override
            public void onNext(Integer integer) {
                LogUtil.d("Resume downloadFile file start.");
            }
        });
    }

    /**
     * 文件下载完成
     */
    @Subscribe
    public void onReceiveFileFinished(IMProxyFileEvent.ReceiveFileFinishedEvent event) {

        long attachedMsgId = event.getAttachedMsgId();
        if (attachedMsgId <= -1) {
            LogUtil.e("ERROR: Receive file finished event, msgId " + attachedMsgId);
            return;
        }

        if (mCurMessageBean.getId() == attachedMsgId) {
            mCurFileInfo.setFileState(ConstDef.DONE);
            getVu().downloadFileFinish();
        }
    }

    /**
     * 文件下载过程中进度更新回调
     */
    @Subscribe
    public void onReceiveFileUpdated(IMProxyFileEvent.ReceiveFileProgressUpdateEvent event) {

        long attachedMsgId = event.getAttachedMsgId();
        if (attachedMsgId <= -1) {
            LogUtil.e("ERROR: Receive file update event, msgId " + attachedMsgId);
            return;
        }

        if (mCurMessageBean.getId() == attachedMsgId) {
            mCurFileInfo.setFileState(ConstDef.LOADING);
            mCurFileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
            getVu().updateDownloadRate();

        }
    }

    /**
     * 文件下载过程中暂停回调
     */
    @Subscribe
    public void onReceiveFilePaused(IMProxyFileEvent.ReceiveFilePausedEvent event) {

        long attachedMsgId = event.getAttachedMsgId();
        if (attachedMsgId <= -1) {
            LogUtil.e("ERROR: Receive file pause event, msgId " + attachedMsgId);
            return;
        }

        if (mCurMessageBean.getId() == attachedMsgId) {
            mCurFileInfo.setFileState(ConstDef.PAUSE);

        }
    }

    /**
     * 下载文件失败
     */
    @Subscribe
    public void onReceiveFileFailed(IMProxyFileEvent.ReceiveFileFailedEvent event) {

        long attachedMsgId = event.getAttachedMsgId();
        if (attachedMsgId <= -1) {
            LogUtil.e("ERROR: Receive file pause event, msgId " + attachedMsgId);
            return;
        }

        if (mCurMessageBean.getId() == attachedMsgId) {
            mCurFileInfo.setFileState(ConstDef.FAIL);
        }
    }

    @Override
    public void openFile() {
        if (mCurMessageBean == null) {
            return;
        }
        FileInfo fileInfo = mCurMessageBean.getFileInfo();
        File file = new File(fileInfo.getFilePath());
        if (!file.exists()) {
            CommonTool.showToast(R.string.im_uikit_file_not_exist);
        } else {
            SystemHelper.openFile(this, fileInfo.getFilePath());
        }
    }

    @Override
    public NormalFileInfo getCurFileInfo() {
        return mCurFileInfo;
    }

    @Override
    public void onBackPressed() {
        //在下载界面返回时，需要将当前文件信息返回给会话详情界面
        Intent intent = new Intent();
        //数据绑定
        Bundle bundle = new Bundle();
        bundle.putLong(Constant.TAG_MSGID, mCurMessageBean.getId());
        bundle.putInt(Constant.TAG_FILE_STATE, mCurFileInfo.getFileState());
        bundle.putLong(Constant.TAG_FILE_TRANS_SIZE, mCurFileInfo.getTranslateSize());
        //添加数据到Intent
        intent.putExtras(bundle);
        //返回结果
        setResult(Constant.RESULT_CODE_FILE_DOWNLOAD, intent);
        finish();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
