package com.xdja.im.core.proxy;

import android.text.TextUtils;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.callback.AccountCallback;
import com.xdja.im.core.callback.CommonCallback;
import com.xdja.im.core.callback.NotificationCallback;
import com.xdja.im.core.callback.OnChatSettingCallback;
import com.xdja.im.core.callback.UnReadMessageCallback;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountAction;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.account.AccountType;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.notify.NotificationBean;
import com.xdja.im.core.utils.DataCache;
import com.xdja.im.core.utils.ToolUtils;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.log.LogUtil;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.core.proxy
 * @Author: xdjaxa
 * @Creation: 2017-06-26 8:49
 * @Version V1.0
 * @Description:
 */
public class IMUiKitProxyImpl implements IMUiKitProxy {

    private AccountCallback accountCallback;

    private NotificationCallback notificationCallback;

    private UnReadMessageCallback unReadMessageCallback;

    private OnChatSettingCallback onChatSettingCallback;

    private CommonCallback commonCallback;

    private IMUiKitProxyImpl() {
    }

    private static class SingletonInstance {
        private static final IMUiKitProxyImpl mInstance = new IMUiKitProxyImpl();
    }

    public static IMUiKitProxyImpl getInstance() {
        return SingletonInstance.mInstance;
    }

    public void releaseProxy() {
        accountCallback = null;
        notificationCallback = null;
        unReadMessageCallback = null;
        onChatSettingCallback = null;
        commonCallback = null;
    }

    @Override
    public void setAccountCallback(AccountCallback callback) {
        accountCallback = callback;
    }

    @Override
    public void setNotificationCallback(NotificationCallback callback) {
        notificationCallback = callback;
    }

    @Override
    public void setUnReadMessageCallback(UnReadMessageCallback callback) {
        unReadMessageCallback = callback;
    }

    @Override
    public void setOpenChatSettingCallback(OnChatSettingCallback callback) {
        onChatSettingCallback = callback;
    }

    @Override
    public void setCommonCallback(CommonCallback callback) {
        commonCallback = callback;
    }

    @Override
    public void getAccount(String account, final int type, Subscriber<AccountBean> subscriber) {
        Observable.just(account)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<AccountBean>>() {
                    @Override
                    public Observable<AccountBean> call(String account) {

                        //缓存中获取
                        String sessionFlag = ToolUtils.getSessionTag(account, type);
                        AccountBean accountBean = (AccountBean) DataCache.getInstance().get(sessionFlag);
                        if (accountBean != null) {
                            return Observable.just(accountBean);
                        }

                        //上层应用获取
                        if (accountCallback == null) {
                            return Observable.error(new IllegalArgumentException("Not register callback."));
                        }
                        AccountType accountType = type == ConstDef.CHAT_TYPE_P2P
                                ? AccountType.TYPE_P2P : AccountType.TYPE_P2G;
                        accountBean = accountCallback.getAccount(account, accountType);
                        if (accountBean == null) {
                            return Observable.error(new IllegalArgumentException("Get account failed."));
                        }
                        DataCache.getInstance().put(sessionFlag, accountBean);
                        return Observable.just(accountBean);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public AccountBean getAccount(String account, int type) {
        if (null == account) return null;
        //缓存中获取
        String sessionFlag = ToolUtils.getSessionTag(account, type);
        AccountBean accountBean = (AccountBean) DataCache.getInstance().get(sessionFlag);
        if (accountBean != null) {
            return accountBean;
        }

        //上层应用获取
        if (accountCallback == null) {
            LogUtil.e(this.getClass().getName(), "can't get account info, accountCallback is null");
            return null;
        }
        AccountType accountType = type == ConstDef.CHAT_TYPE_P2P
                ? AccountType.TYPE_P2P : AccountType.TYPE_P2G;
        accountBean = accountCallback.getAccount(account, accountType);
        if (accountBean == null) {
            LogUtil.e(this.getClass().getName(), "account info is null, from accountCallback");
            return null;
        }
        DataCache.getInstance().put(sessionFlag, accountBean);
        return accountBean;
    }

    @Override
    public void notifyAccountChanged(String account, AccountAction action) {
        if (TextUtils.isEmpty(account)) {
            LogUtil.e("ERROR: notify account changed, but account is empty.");
            return;
        }
        LogUtil.d("Notify account Changed account " + account + "(" + action + ")");

        String sFlag = ToolUtils.getSessionTag(account, ConstDef.CHAT_TYPE_P2P);
        switch (action) {
            case ACTION_UPDATE: //更新联系人信息
                DataCache.getInstance().remove(sFlag);
                BusProvider.getInstance().post(new IMProxySessionEvent.RefreshSessionListEvent());
                break;

            case ACTION_DELETE: //删除联系人信息
                DataCache.getInstance().remove(sFlag);
                IMProxySessionEvent.DeleteSessionEvent deleteSessionEvent = new IMProxySessionEvent.DeleteSessionEvent();
                TalkSessionBean talkSessionBean = new TalkSessionBean();
                talkSessionBean.setTalkerAccount(account);
                talkSessionBean.setTalkType(ConstDef.CHAT_TYPE_P2P);
                deleteSessionEvent.setSessionBean(talkSessionBean);
                BusProvider.getInstance().post(deleteSessionEvent);
                break;
        }
    }

    @Override
    public boolean notify(TalkSessionBean sessionBean, int newMsgCnt, boolean isDisturb) {

        if (notificationCallback == null) {
            LogUtil.d("Notify callback is null.");
            return false;
        }

        //对外接口数据实体类
        NotificationBean notificationBean = new NotificationBean();
        notificationBean.setAccount(sessionBean.getTalkerAccount());
        notificationBean.setTalkId(sessionBean.getTalkFlag());
        notificationBean.setTalkType(sessionBean.getTalkType());
        notificationBean.setMsgCount(newMsgCnt);

        TalkMessageBean talkMessageBean = sessionBean.getLastMsg();
        if (talkMessageBean == null || talkMessageBean.isMine()) {
            LogUtil.w("This notify come form myself, so return!!");
            return true;
        }
        String content = talkMessageBean.getContent();
        switch (talkMessageBean.getMessageType()) {
            case ConstDef.MSG_TYPE_VOICE:
                content = CommonTool.getString(R.string.im_uikit_voice);
                break;
            case ConstDef.MSG_TYPE_PHOTO:
                content = CommonTool.getString(R.string.im_uikit_image);
                break;
            case ConstDef.MSG_TYPE_VIDEO:
                content = CommonTool.getString(R.string.im_uikit_video);
                break;
            case ConstDef.MSG_TYPE_FILE:
                content = CommonTool.getString(R.string.im_uikit_file);
                break;
        }
        notificationBean.setContent(content);
        notificationBean.setDisturb(isDisturb);
        notificationBean.setShowTime(sessionBean.getLastTime());

        LogUtil.d("Start to notify..." + notificationBean.toString());
        notificationCallback.notify(notificationBean);
        return true;
    }

    @Override
    public void updateUnReadMessageCount(int count) {
        if (unReadMessageCallback == null) {
            LogUtil.d("UnReadMessage callback is null.");
            return;
        }
        LogUtil.d("Update unRead message " + count);
        unReadMessageCallback.onRefresh(count);
    }

    @Override
    public boolean openChatSettingCallback(String sessionId, int sessionType) {
        if (onChatSettingCallback == null) {
            LogUtil.d("open chat setting callback is null.");
            return false;
        }
        onChatSettingCallback.openSetting(sessionId, sessionType);
        return true;
    }

    @Override
    public boolean isShowGroupSetting(String groupId) {
        if(onChatSettingCallback != null){
            return onChatSettingCallback.isGroupMember(groupId);
        }
        return false;
    }

    @Override
    public void startCallback(String atNo) {
        if(commonCallback != null){
            commonCallback.startCallback(atNo);
        }
    }

    @Override
    public void startVoice(String atNo, int type) {
        if(commonCallback != null){
            commonCallback.startVoice(atNo , type);
        }
    }

    @Override
    public void startLocation(String atNo) {
        if(commonCallback != null){
            commonCallback.startLocation(atNo);
        }
    }
}
