package com.xdja.im.core.repository.interf;

import android.support.annotation.NonNull;

import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.custom.CustomMsgInfo;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.file.bean.FileListType;
import com.xdja.imsdk.model.IMMessage;

import java.io.File;
import java.util.List;
import java.util.Map;

import rx.Observable;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.core.repository.interf
 * @Author: xdjaxa
 * @Creation: 2017-04-11 11:08
 * @Version V1.0
 * @Description:IM模块接口定义
 */
public interface ProxyRepository {

    /**
     * 初始化IMSDK
     *
     * @return 0:表示初始化成功，其他表示失败
     */
    Observable<Integer> initIMProxy();

    /**
     * IMSDK资源释放
     *
     * @return 0:表示初始化成功，其他表示失败
     */
    Observable<Integer> releaseIMProxy();

    /**
     * 为IMSDK设置相关属性
     *
     * @param param 属性相关数据
     * @return 操作结果
     */
    Observable<Boolean> setProxyConfig(Map<String, String> param);

    /**
     * 获取IMSDK相关属性设置
     *
     * @return 属性相关数据
     */
    Observable<String> getProxyConfig(final String key);

    /**
     * 获取会话列表
     * <p>注：begin=0，size=0 则返回所有会话（取决于本地数据库数据量）
     * begin=x, size=y 返回x会话开始之后y个会话
     * begin=x, size=0 返回会话Id为x的会话
     * begin=0, size=y 最后一个更新会话向前取y个会话
     * </p>
     *
     * @param begin      获取会话起始位置会话Id
     * @param size       从起始位置开始获取指定数量会话
     * @param talkType   为会话支持的类型{@link com.xdja.im.core.config.ConstDef.ChatType}
     * @param subscriber 观察者
     * @return 符合条件的会话组成的列表，以会话生成的时间为序以会话最后一次更新时间倒序返回
     */
    void getSessionListBeans(String begin, int size, int talkType, Subscriber<List<TalkSessionBean>> subscriber);

    /**
     * 删除会话
     *
     * @param sessionFlag: 需要删除的会话标识
     * @return 0： 删除会话成功
     */
    void deleteSession(String sessionFlag, Subscriber<Integer> subscriber);

    /**
     * 获取指定会话消息列表
     * <P>注：begin=0，size=0 则返回所有消息（本地数据库数据）
     * begin=x, size=y MsgId=x之后y条消息
     * begin=0, size=y 当前时间点后y条信息
     * </P>
     *
     * @param talkId     会话ID
     * @param talkType   会话类型
     * @param begin      获取消息起始位置
     * @param size       从起始位置开始获取指定数量消息
     * @param subscriber 观察者
     * @return 符合条件的消息组成的列表，时间倒序排列。排序时间可选，服务器接收时间，客户端接收时间，客户端发送时间
     */
    void getMessageList(String talkId,
                        int talkType,
                        long begin,
                        int size,
                        Subscriber<List<TalkMessageBean>> subscriber);

    /**
     * 获取消息的完整信息
     *
     * @param msgId      消息ID
     * @param subscriber
     */
    void getTalkMessage(long msgId, Subscriber<TalkMessageBean> subscriber);

    /**
     * 删除消息
     *
     * @param msgids 列表形式提供删除多条消息
     */
    void deleteMessages(List<Long> msgids,
                        Subscriber<Integer> subscriber);


    /**
     * 获取会话中未读消息数量
     *
     * @param talkId 指定的会话Id
     * @return -1：调用异常;其他：当前会话中未读消息数量
     */
    void getUnReadMsgCount(String talkId,
                           Subscriber<Integer> subscriber);

    /**
     * 清空会话未读消息数量
     *
     * @param talkId 会话ID
     * @return 0：调用成功；其他：错误待定
     */
    void clearUnReadMsgCount(String talkId,
                             Subscriber<Integer> subscriber);

    /**
     * 清除所有消息
     *
     * @param subscriber
     */
    void clearAllData(Subscriber<Integer> subscriber);

    /**
     * 清除某一会话中，所有的聊天消息
     *
     * @param sessionFlag
     * @param subscriber
     */
    void clearAllSessionData(String sessionFlag, Subscriber<Integer> subscriber);

    /**
     * 清空所有会话
     *
     * @return
     */
    void clearAllCacheData(Subscriber<Integer> subscriber);

    /**
     * 获取所有会话未读消息总数
     *
     * @return -1：调用异常;其他：所有会话未读消息数量总和
     */
    void getAllUnReadMsgCount(Subscriber<Integer> subscriber);

    /**
     * 发送消息
     *
     * @param talkMessageBean 待发送的消息对象
     * @param subscriber
     * @return 消息保存在数据库中的数据库id
     */
    void sendMessage(TalkMessageBean talkMessageBean,
                     Subscriber<TalkMessageBean> subscriber);

    /**
     * 消息重发
     *
     * @param talkMessageBean 需要重发的消息信息
     * @param subscriber
     * @return 0：重发成功;其他：错误待定
     */
    void reSendMessage(TalkMessageBean talkMessageBean,
                       Subscriber<Integer> subscriber);

    /**
     * 发送文本消息
     *
     * @param content 消息内容
     * @param to      消息接收方标识
     * @param isShan  是否为闪信
     * @param isGroup 是否为群组
     */
    void sendTextMessage(@NonNull String content,
                         @NonNull String to,
                         boolean isShan,
                         boolean isGroup,
                         Subscriber<TalkMessageBean> subscriber);

    /**
     * 撤回消息
     *
     * @param talkId 待撤回的消息对象id
     * @param action
     * @return 撤回结果
     */
    void recallMessages(long talkId, @ConstDef.RecallAction int action, Subscriber<Integer> subscriber);

    /**
     * 发送文本通知消息
     *
     * @param content 消息内容
     * @param to      消息接收方标识
     * @param isGroup 是否为群组
     */
    void sendNotifyTextMessage(String content,
                               String to,
                               boolean isGroup,
                               Subscriber<TalkMessageBean> subscriber);

    /**
     * 发送自定义消息
     *
     * @param talkMessageBean 自定义消息体
     * @param subscriber
     */
    void sendCustomMessage(TalkMessageBean talkMessageBean,
                           Subscriber<TalkMessageBean> subscriber);

    /**
     * 发送文件消息（语音、图片、视频、文件等）
     *
     * @param to           消息接收方标识
     * @param isShan       是否为闪信
     * @param isGroup      是否为群组
     * @param fileInfoList 文件內容
     * @return
     */
    void sendFileMessage(@NonNull String to,
                         boolean isShan,
                         boolean isGroup,
                         List<FileInfo> fileInfoList,
                         Subscriber<TalkMessageBean> subscriber);

    /**
     * 下载文件
     *
     * @param fileInfo   文件实体信息
     * @param subscriber
     */
    void downloadFile(List<FileInfo> fileInfo, Subscriber<Integer> subscriber);


    /**
     * 暂停文件上传
     *
     * @param fileInfo   需要暂停的文件对象
     * @param subscriber
     * @return 0：暂停文件上传成功;1：暂停处理失败
     */
    void pauseFileSending(FileInfo fileInfo, Subscriber<Integer> subscriber);


    /**
     * 重新开始文件上传
     *
     * @param fileInfo   需要暂停的文件对象
     * @param subscriber
     * @return 0：恢复文件上传;1：恢复文件上传处理失败
     */
    void resumeFileSend(FileInfo fileInfo, Subscriber<Integer> subscriber);

    /**
     * 暂停文件下载
     *
     * @param fileInfo   需要暂停的文件对象
     * @param subscriber
     * @return 0：回调函数未处理
     */
    void pauseFileReceiving(FileInfo fileInfo, Subscriber<Integer> subscriber);


    /**
     * 重新开始文件下载
     *
     * @param fileInfo   需要暂停的文件对象
     * @param subscriber
     * @return 0：恢复文件下载;1：恢复文件下载处理失败
     */
    void resumeFileReceive(FileInfo fileInfo, Subscriber<Integer> subscriber);


    /**
     * 查询当前会话中的文件（目前为图片和视频）
     *
     * @param talkId
     * @param begin
     * @param size
     * @param subscriber
     */
    void getSessionFileList(@NonNull String talkId,
                            int begin, int size,
                            Subscriber<List<TalkMessageBean>> subscriber);

    /**
     * 获取最新和本地文件列表
     *
     * @param tabType
     * @param subscriber
     */
    void queryRecentAndLocalFile(FileListType tabType,
                                 Subscriber<Map<String, List<FileItem>>> subscriber);

    /**
     * 更新消息状态
     *
     * @param message 消息对象
     * @param mState  将要更新的状态
     * @return 0：状态更新成功;其他： 错误待定
     */
    void changeMessageState(final TalkMessageBean message,
                            @ConstDef.MsgState int mState,
                            Subscriber<Integer> subscriber);

    /**
     * 保存图片
     *
     * @param srcFile    源文件路径
     * @param savePath   保存文件
     * @param fileName   保存文件名称
     * @param subscriber
     */
    void saveFile(File srcFile, String savePath, String fileName, Subscriber<String> subscriber);

    /**
     * 保存自定义消息
     *
     * @param msgInfo 消息信息
     */
    void saveCustomIMMessage(CustomMsgInfo msgInfo, Subscriber<IMMessage> subscriber);
}
