package com.xdja.im.lib.audio.manager;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AppOpsManager;
import android.content.Context;
import android.media.MediaRecorder;
import android.os.Build;
import android.os.Handler;

import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.UUID;

public class AudioManager {
    private static final int VERSION = Build.VERSION.SDK_INT;
    private MediaRecorder mMediaRecorder;
    private String mDir;
    private String mCurrentFilePath;
    private static final int REC_LENGTH = 60000;

    private boolean isPrepare;
    private Context mContext;
    private AppOpsManager mAppOpsManager;
    private Method mCheckOpMethod;
    private Handler mHandler = new Handler();
    public int EXCEPTION = 1;//录音权限被禁异常
    public int ILLEGALSTATE_EXCEPTION = 2;//录音被占用时非法异常

    private Activity mActivity;

    @SuppressLint("InlinedApi")
    public AudioManager(Context context, String dir) {
        mDir = dir;
        mContext = context;
        if (VERSION >= Build.VERSION_CODES.KITKAT) {
            mAppOpsManager = (AppOpsManager) mContext.getSystemService(Context.APP_OPS_SERVICE);
            mCheckOpMethod = obtainOpsMethod(mAppOpsManager);
        }
    }

    /**
     * 使用接口 用于回调
     */
    public interface AudioStateListener {
        /**
         * 开始录音
         */
        void wellPrepared();

        /**
         * 录音失败
         *
         * @param exceptionType
         */
        void prepareFailed(int exceptionType);

        void checkCurrentFocusView();
    }

    public AudioStateListener mAudioStateListener;

    public void setActivity(Activity activity) {
        mActivity = activity;
    }

    /**
     * 回调方法
     */
    public void setOnAudioStateListener(AudioStateListener listener) {
        mAudioStateListener = listener;
    }

    private Method obtainOpsMethod(AppOpsManager opm) {
        try {
            Class<?> appOpsClass = opm.getClass();
            return appOpsClass.getDeclaredMethod("checkOp", new Class[]{int.class, int.class, String.class});
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 去准备
     */
    public synchronized void prepareAudio() {
        try {
            isPrepare = false;
            File dir = new File(mDir);
            if (!dir.exists() && !dir.mkdirs()) {
                LogUtil.e("prepare audio failed. create path error.");
                return;
            }

            String fileName = generateFileName();
            File file = new File(dir, fileName);

            mCurrentFilePath = file.getAbsolutePath();

            release();

            mMediaRecorder = new MediaRecorder();
            // 设置输出文件
            mMediaRecorder.setOutputFile(file.getAbsolutePath());
            // 设置MediaRecorder的音频源为麦克风
            mMediaRecorder.setAudioSource(MediaRecorder.AudioSource.MIC);
            // 设置音频格式
            mMediaRecorder.setOutputFormat(MediaRecorder.OutputFormat.DEFAULT);
            // 设置音频编码
            mMediaRecorder.setAudioEncoder(MediaRecorder.AudioEncoder.DEFAULT);
            //设置最大录音时长
            mMediaRecorder.setMaxDuration(REC_LENGTH);

            long beginTime = System.currentTimeMillis();

            // 准备录音
            mMediaRecorder.prepare();
            mMediaRecorder.start();

            long endTime = System.currentTimeMillis();

            if ((endTime - beginTime) > 500) {
                mAudioStateListener.checkCurrentFocusView();
                if (mMediaRecorder != null) {
                    mMediaRecorder.stop();
                    mMediaRecorder.reset();
                }

            } else {
                if (mAudioStateListener != null) {
                    mAudioStateListener.wellPrepared();
                }
                isPrepare = true;
            }

        } catch (IllegalStateException e) {
            e.printStackTrace();
            prepareFaild(ILLEGALSTATE_EXCEPTION);
        } catch (IOException e) {
            e.printStackTrace();
            prepareFaild(EXCEPTION);
        } catch (Exception e) {
            e.printStackTrace();
            prepareFaild(EXCEPTION);
        } finally {
        }

    }

    private void prepareFaild(final int exceptionType) {
        isPrepare = false;
        release();
        mHandler.post(new Runnable() {
            @Override
            public void run() {
                if (mAudioStateListener != null) {
                    mAudioStateListener.prepareFailed(exceptionType);
                }
            }
        });
    }

    /**
     * 随机生成文件的名称
     */
    private String generateFileName() {
        return UUID.randomUUID().toString();
    }

    //获得声音振幅大小
    public synchronized double getAmplitude() {
        try {
            if (isPrepare) {
                if (mMediaRecorder != null) {
                    return mMediaRecorder.getMaxAmplitude();
                }
            }
        } catch (Exception e) {
        }
        return 1;
    }

    /**
     * 释放资源
     */
    public synchronized void release() {
        try {
            if (mMediaRecorder != null) {
                mMediaRecorder.stop();
                mMediaRecorder.reset();
                mMediaRecorder.release();
                mMediaRecorder = null;
            }
        } catch (Exception e) {

        } finally {
            if (mMediaRecorder != null) {
                mMediaRecorder = null;
            }
        }
    }

    /**
     * 取消录音
     */
    public synchronized void cancel() {
        release();
        if (mCurrentFilePath != null) {
            File file = new File(mCurrentFilePath);
            if (!file.delete()) {
                LogUtil.e("delete recorder cache file error.");
            }
            mCurrentFilePath = null;
        }

    }

    public String getCurrentFilePath() {

        return mCurrentFilePath;
    }
}
