package com.xdja.im.lib.file.ui.fragment;

import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v4.app.Fragment;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.lib.file.FileExplorerConfig;
import com.xdja.im.lib.file.FileExplorerManager;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.file.bean.FileListType;
import com.xdja.im.lib.file.ui.adapter.CommonRecyclerAdapter;
import com.xdja.im.lib.file.ui.adapter.FileExplorerFragmentAdapter;
import com.xdja.im.lib.file.ui.presenter.FileListFragmentPresenter;
import com.xdja.im.lib.file.ui.view.ViewChildFileDelegate;
import com.xdja.im.lib.file.ui.view.ViewChildPhotoDelegate;
import com.xdja.im.lib.file.ui.view.ViewDirTitleDelegate;
import com.xdja.im.lib.file.ui.view.ViewFileList;
import com.xdja.im.lib.file.ui.view.vu.FileListVu;
import com.xdja.im.lib.file.utils.FileConstant;
import com.xdja.im.uikit.base.ImBaseFragment;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.lib.file.ui.fragment
 * @Author: xdjaxa
 * @Creation: 2017-04-26 9:02
 * @Version V1.0
 * @Description:
 */
public class FileListFragment extends ImBaseFragment<FileListFragmentPresenter, FileListVu>
        implements FileListFragmentPresenter {

    private int type;
    private FileListType[] childTabLists = {
            FileListType.VIDEO,
            FileListType.PHOTO,
            FileListType.DOCUMENT,
            FileListType.APK,
            FileListType.OTHER
    };

    @Inject
    Lazy<ProxyRepository> mRepository;

    private FileExplorerFragmentAdapter adapter;

    private Map<String, List<FileItem>> dataMaps = new HashMap<>();

    private CommonRecyclerAdapter fragmentAdapter;

    public static FileListFragment newInstance(Bundle bundle) {
        FileListFragment fragment = new FileListFragment();
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    protected Class<? extends FileListVu> getVuClass() {
        return ViewFileList.class;
    }

    @NonNull
    @Override
    protected FileListFragmentPresenter getPresenter() {
        return this;
    }


    @Override
    protected void preBindView(Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);
        Bundle bundle = getArguments();
        type = bundle.getInt(Constant.TAB_TYPE);
    }

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (mComponent == null) {
            return;
        }
        mComponent.inject(this);

        bindViews();
    }

    private void bindViews() {
        getVu().showViewWithType(type);

        initFragmentAdapter(type);

        if (type == FileListType.LOCAL.getType()) {
            bindLocalFragmentView();
        } else {
            bindFragmentView(type);
        }
    }

    private void initFragmentAdapter(int type) {
        fragmentAdapter = new CommonRecyclerAdapter(getActivity(), null, dataMaps);
        fragmentAdapter.addItemViewDelegate(FileConstant.PARENT_DIR, new ViewDirTitleDelegate(type));

        int photoStyle = FileExplorerManager.getInstance().getConfig().getPhotoStyle();
        if (type == FileListType.PHOTO.getType() && photoStyle == FileExplorerConfig.GRID) {
            fragmentAdapter.addItemViewDelegate(FileConstant.CHILD_LEAF, new ViewChildPhotoDelegate());
        } else {
            fragmentAdapter.addItemViewDelegate(FileConstant.CHILD_LEAF_FILE, new ViewChildFileDelegate());
        }
        fragmentAdapter.setRecyclerView(getVu().getRecyclerView());
        fragmentAdapter.setLayoutManager(FileConstant.PARENT_DIR_WIDTH);
        getVu().setRecyclerViewAdapter(fragmentAdapter);
    }

    private void bindFragmentView(int type) {
        if (type == FileListType.RECENT.getType()) {
            getVu().showEmptyView(false);
        }

        FileListType tabType = findFileListTypeWithType(type);
        mRepository.get().queryRecentAndLocalFile(tabType,
                new Subscriber<Map<String, List<FileItem>>>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getVu().showEmptyView(true);
                        LogUtil.d("obtain msg error:" + e.getMessage());
                    }

                    @Override
                    public void onNext(Map<String, List<FileItem>> stringListMap) {
                        if (stringListMap.size() == 0) {
                            getVu().showEmptyView(true);
                        } else {
                            dataMaps.putAll(stringListMap);
                            fragmentAdapter.updateAllItems();
                        }
                    }
                });
    }

    private FileListType findFileListTypeWithType(int type) {
        if (type == FileListType.RECENT.getType()) {
            return FileListType.RECENT;
        } else if (type == FileListType.VIDEO.getType()) {
            return FileListType.VIDEO;
        } else if (type == FileListType.PHOTO.getType()) {
            return FileListType.PHOTO;
        } else if (type == FileListType.DOCUMENT.getType()) {
            return FileListType.DOCUMENT;
        } else if (type == FileListType.APK.getType()) {
            return FileListType.APK;
        } else if (type == FileListType.OTHER.getType()) {
            return FileListType.OTHER;
        } else {
            //其余别的类型
            return FileListType.NON;
        }
    }

    private void bindLocalFragmentView() {
        Bundle bundle = null;
        List<Fragment> fragments = new ArrayList<>();
        for (FileListType listType : childTabLists) {
            bundle = new Bundle();
            bundle.putInt(Constant.TAB_TYPE, listType.getType());
            fragments.add(FileListFragment.newInstance(bundle));
        }
        adapter = buildAdapter(childTabLists, fragments);
        getVu().setFragmentAdapter(adapter);
    }

    private FileExplorerFragmentAdapter buildAdapter(final FileListType[] titles, List<Fragment> fragments) {
        return new FileExplorerFragmentAdapter<FileListType>(getChildFragmentManager(), getActivity(),
                titles, fragments) {
            @Override
            public CharSequence getPageTitle(int position) {
                return context.getString(titles[position].getResId());
            }
        };
    }
}
