package com.xdja.im.uikit.base;

import android.app.Dialog;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.IdRes;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.view.ViewTreeObserver;

import com.xdja.im.base.di.component.DaggerUserComponent;
import com.xdja.im.base.di.component.RepositoryComponent;
import com.xdja.im.base.di.component.UserComponent;
import com.xdja.im.base.frame.Presenter;
import com.xdja.im.base.frame.presenter.BaseFragmentPresenter;
import com.xdja.im.base.frame.view.FragmentBaseView;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.ui.dialog.CustomDialog;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.sys.NetworkUtil;

/**
 * @Package: com.xdja.im.uikit.presenter.base
 * @Author: xdjaxa
 * @Creation: 2017-04-11 10:40
 * @Version V1.0
 * @Description:
 */
public abstract class ImBaseFragment<P extends Presenter, V extends FragmentBaseView>
        extends BaseFragmentPresenter<P, V> {

    protected UserComponent mComponent;

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        RepositoryComponent repositoryComponent = ImUiKit.getInstance().getComponent();
        if (repositoryComponent != null) {
            mComponent = DaggerUserComponent.builder()
                    .repositoryComponent(repositoryComponent)
                    .build();
        }
        if (mComponent == null) {
            LogUtil.e("Fatal :User component is null .");
            getActivity().finish();
        }

        getActivity().getWindow().getDecorView().getViewTreeObserver().addOnWindowFocusChangeListener(
                new ViewTreeObserver.OnWindowFocusChangeListener() {
                    @Override
                    public void onWindowFocusChanged(boolean hasFocus) {
                        onActivityFocusChanged(hasFocus);
                    }
                });
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
    }

    public void switchFragment(ImBaseFragment fragment, boolean addToBackStack) {
        switchFragment(fragment, android.R.id.content, addToBackStack);
    }

    protected void switchFragment(ImBaseFragment fragment, @IdRes int contentRes, boolean addToBackStack) {
        FragmentManager fg = getFragmentManager();
        FragmentTransaction ft = fg.beginTransaction();

        if (addToBackStack) {
            ft.setCustomAnimations(
                    R.anim.im_uikit_open_enter,
                    R.anim.im_uikit_open_exit,
                    R.anim.im_uikit_close_enter,
                    R.anim.im_uikit_close_exit);
        }

        ft.replace(contentRes, fragment, fragment.getName());
        if (addToBackStack) {
            ft.addToBackStack(fragment.getName());
        }
        ft.commit();
    }

    /**
     * 启动申请权限框
     */
    protected void showMissingPermissionDialog() {
        new CustomDialog(getContext(), new CustomDialog.OnCloseListener() {
            @Override
            public void onClick(Dialog dialog, boolean confirm) {
                if (confirm) {
                    startAppSettings();
                } else {
                    finish();
                }
            }
        })
        .setDialogTitle(R.string.im_uikit_hint)
        .setDialogContent(R.string.im_uikit_permission_hint)
        .setPositiveButton(R.string.im_uikit_setting)
        .setNegativeButton(R.string.im_uikit_btn_cancel)
        .show();
    }

    /**
     * 启动应用的设置
     */
    private void startAppSettings() {
        Intent intent = new Intent(
                Settings.ACTION_APPLICATION_DETAILS_SETTINGS);
        intent.setData(Uri.parse("package:" + getPackageName()));
        startActivity(intent);
    }

    /**
     * 监测网络是否需要进行文件发送操作
     *
     * @param result 回调接口
     */
    protected void checkNetType(final ImBaseActivity.OnNetCheckCallback result) {

        boolean isMobileConnected = NetworkUtil.isConnectByMobile(getContext());
        if (isMobileConnected) {
            new CustomDialog(getContext(), new CustomDialog.OnCloseListener() {
                @Override
                public void onClick(Dialog dialog, boolean confirm) {
                    result.onResult(!confirm);
                }
            })
            .setDialogTitle(result.getTitle())
            .setDialogContent(result.getContent())
            .setPositiveButton(result.getPositiveButton())
            .setNegativeButton(result.getNegativeButton())
            .show();
        } else {
            result.onResult(false);
        }
    }

    public abstract class OnNetCheckCallback {

        /**
         * 是否已取消
         *
         * @param isCanceled
         */
        public abstract void onResult(boolean isCanceled);

        protected String getContent() {
            return "";
        }

        protected String getTitle() {
            return getString(R.string.im_uikit_hint);
        }

        protected String getPositiveButton() {
            return getString(R.string.im_uikit_btn_continue);
        }

        protected String getNegativeButton() {
            return getString(R.string.im_uikit_btn_cancel);
        }
    }

    protected void finish() {
        getActivity().finish();
    }

    /**
     * 窗口焦点发生变化
     *
     * @param hasFocus
     */
    protected void onActivityFocusChanged(boolean hasFocus) {
    }

    protected void showToast(String content) {
        CommonTool.showToast(content);
    }

    protected void showToast(int resId) {
        CommonTool.showToast(resId);
    }

    public String getName() {
        return this.getClass().getName();
    }

    public String getPackageName() {
        return getContext().getPackageName();
    }
}
