package com.xdja.im.common.database.dao;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.text.TextUtils;

import com.xdja.im.common.database.DatabaseManager;
import com.xdja.im.common.database.bean.SessionDb;
import com.xdja.im.common.database.builder.SessionEntityBuilder;
import com.xdja.im.common.database.helper.UpdateArgs;

import java.util.ArrayList;
import java.util.List;

/**
 * @Package: com.xdja.im.common.database.dao
 * @Author: xdjaxa
 * @Creation: 2017-05-22 18:15
 * @Version V1.0
 * @Description:
 */
public class SessionEntityDao extends AbstractDao<SessionDb> {

    private SessionEntityDao() {

    }

    private static class SingletonInstance{
        private static final SessionEntityDao mInstance = new SessionEntityDao();
    }

    public static SessionEntityDao getInstance() {
        return SingletonInstance.mInstance;
    }

    public long insert(SessionDb sessionDb) {
        SQLiteDatabase db = getDatabase();
        if (db != null) {
            return insert(db, sessionDb, SessionEntityBuilder.insertSql());
        }
        return -1;
    }

    public int delete(String sessionFlag) {
        if (TextUtils.isEmpty(sessionFlag)) {
            return -1;
        }
        return super.delete(SessionEntityBuilder.delSql(sessionFlag));
    }

    public int deleteAll() {
        return super.delete(SessionEntityBuilder.delAllSql());
    }

    public void deleteOnlyDraft(String sessionFlag) {
        UpdateArgs args = new UpdateArgs(SessionEntityBuilder.TABLE_NAME);
        ContentValues values = new ContentValues();
        values.put(SessionEntityBuilder.DRAFT_CONTENT, "");
        values.put(SessionEntityBuilder.DRAFT_TIME, 0);

        String whereClause[] = new String[]{SessionEntityBuilder.SESSION_FLAG};
        String[] whereArgs = new String[]{sessionFlag};

        args.setValues(values);
        args.setWhereClause(whereClause);
        args.setWhereArgs(whereArgs);
        update(args);
    }

    public SessionDb get(String sessionFlag) {
        SessionDb msgEntryDb = null;
        Cursor cursor = null;
        try {
            cursor = query(SessionEntityBuilder.querySql(sessionFlag));
            if (cursor != null && cursor.moveToNext()) {
                msgEntryDb = readEntry(cursor, 0);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        return msgEntryDb;
    }

    public void update(SessionDb sessionDb) {
        UpdateArgs args = new UpdateArgs(SessionEntityBuilder.TABLE_NAME);
        ContentValues values = new ContentValues();

        values.put(SessionEntityBuilder.SESSION_FLAG, sessionDb.getSessionFlag());
        values.put(SessionEntityBuilder.SESSION_MODE, sessionDb.getSessionMode());
        values.put(SessionEntityBuilder.SESSION_DISTURB, sessionDb.getSessionDisturb());
        values.put(SessionEntityBuilder.RECEIVE_MODE, sessionDb.getReceiveMode());
        values.put(SessionEntityBuilder.DRAFT_CONTENT, sessionDb.getDraftContent());
        values.put(SessionEntityBuilder.DRAFT_TIME, sessionDb.getDraftCreateTime());

        String whereClause[] = new String[]{SessionEntityBuilder.SESSION_FLAG};
        String[] whereArgs = new String[]{sessionDb.getSessionFlag()};

        args.setValues(values);
        args.setWhereClause(whereClause);
        args.setWhereArgs(whereArgs);
        update(args);
    }


    public List<SessionDb> getAll() {
        List<SessionDb> sessionDbList = new ArrayList<>();
        Cursor cursor = null;
        try {
            cursor = query(SessionEntityBuilder.queryAll());
            while (cursor != null && cursor.moveToNext()) {
                sessionDbList.add(readEntry(cursor, 0));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        return sessionDbList;
    }

    @Override
    protected SessionDb readEntry(Cursor cursor, int offset) {
        SessionDb entity = new SessionDb(
                cursor.isNull(offset + 0) ? null : cursor.getLong(offset + 0),
                cursor.isNull(offset + 1) ? null : cursor.getString(offset + 1),
                cursor.isNull(offset + 2) ? null : cursor.getInt(offset + 2),
                cursor.isNull(offset + 3) ? null : cursor.getInt(offset + 3),
                cursor.isNull(offset + 4) ? null : cursor.getInt(offset + 4),
                cursor.isNull(offset + 5) ? null : cursor.getString(offset + 5),
                cursor.isNull(offset + 6) ? null : cursor.getLong(offset + 6)
        );
        return entity;
    }

    @Override
    protected void bindValues(SQLiteStatement stmt, SessionDb entity) {
        stmt.clearBindings();
        Long id = entity.getId();
        if (id != null) {
            stmt.bindLong(1, id);
        }

        String sessionFlag = entity.getSessionFlag();
        if (sessionFlag != null) {
            stmt.bindString(2, sessionFlag);
        }

        Integer sessionMode = entity.getSessionMode();
        if (sessionMode != null) {
            stmt.bindLong(3, sessionMode);
        }

        Integer sessionDisturb = entity.getSessionDisturb();
        if (sessionDisturb != null) {
            stmt.bindLong(4, sessionDisturb);
        }

        Integer receiveMode = entity.getReceiveMode();
        if (receiveMode != null) {
            stmt.bindLong(5, receiveMode);
        }

        String draftContent = entity.getDraftContent();
        if (draftContent != null) {
            stmt.bindString(6, draftContent);
        }

        Long draftCreateTime = entity.getDraftCreateTime();
        if (draftCreateTime != null) {
            stmt.bindLong(7, draftCreateTime);
        }
    }

    /**
     * 获取数据库句柄
     *
     * @return
     * @throws IllegalAccessError 数据库句柄为空异常
     */
    private SQLiteDatabase getDatabase() {
        SQLiteDatabase db = DatabaseManager.getInstance().getDatabase();
        if (db == null) {
            throw new IllegalAccessError("open database failed.");
        }
        return db;
    }
}
