package com.xdja.im.core.model.message.mapper;

import android.text.TextUtils;

import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.utils.FileOption;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.utils.IMFileTools;
import com.xdja.imsdk.constant.IMSessionType;
import com.xdja.imsdk.constant.ImSdkFileConstant;
import com.xdja.imsdk.constant.MsgPackType;
import com.xdja.imsdk.constant.MsgState;
import com.xdja.imsdk.constant.internal.MsgType;
import com.xdja.imsdk.model.IMMessage;
import com.xdja.imsdk.model.body.IMFileBody;

/**
 * @Package: com.xdja.im.common.entity.mapper
 * @Author: xdjaxa
 * @Creation: 2017-04-11 13:57
 * @Version V1.0
 * @Description:
 */
public class ValueConverter {

    /**
     * 转换实体会话类型为业务会话类型
     *
     * @param sessionType 实体会话类型
     * @return 业务会话类型
     */
    @ConstDef.ChatType
    public static int sessionTypeConvert(int sessionType) {
        switch (sessionType) {
            case IMSessionType.SESSION_SINGLE:
                return ConstDef.CHAT_TYPE_P2P;

            case IMSessionType.SESSION_GROUP:
                return ConstDef.CHAT_TYPE_P2G;

            case IMSessionType.SESSION_CUSTOM:
                return ConstDef.CHAT_TYPE_CUSTOM;

            default:
                return ConstDef.CHAT_TYPE_DEFAULT;
        }
    }


    /**
     * 将业务会话类型转换为实体会话类型
     *
     * @param chatType 业务会话类型
     * @return 实体会话类型
     */
    public static int talkListTypeConvert(@ConstDef.ChatType int chatType) {
        return 0;
    }


    /**
     * 将实体消息状态转化为业务消息状态
     *
     * @param imMsgState 实体消息状态
     * @return 业务消息状态
     */
    @ConstDef.MsgState
    public static int imMsgStateConvert(int imMsgState) {

        @ConstDef.MsgState int state = ConstDef.STATE_SENDING;
        switch (imMsgState) {
            case MsgState.MSG_STATE_DEFAULT:
                state = ConstDef.STATE_SENDING;
                break;
            case MsgState.MSG_STATE_FAIL:
                state = ConstDef.STATE_SEND_FAILED;
                break;
            case MsgState.MSG_STATE_SEND:
                state = ConstDef.STATE_SEND_SUCCESS;
                break;
            case MsgState.MSG_STATE_REC:
                state = ConstDef.STATE_ARRIVE;
                break;
            case MsgState.MSG_STATE_READ:
                state = ConstDef.STATE_READED;
                break;
            case MsgState.MSG_STATE_BOMB:
                state = ConstDef.STATE_DESTROY;
                break;
            default:
                break;
        }
        return state;
    }

    /**
     * 将业务消息状态转化为实体消息状态
     *
     * @param talkMsgState 业务消息状态
     * @return 实体消息状态
     */
    public static int talkMsgStateConvert(@ConstDef.MsgState int talkMsgState) {
        int state = MsgState.MSG_STATE_DEFAULT;
        switch (talkMsgState) {
            case ConstDef.STATE_DESTROYING:
                break;
            case ConstDef.STATE_SENDING:
                state = MsgState.MSG_STATE_DEFAULT;
                break;
            case ConstDef.STATE_SEND_FAILED:
                state = MsgState.MSG_STATE_FAIL;
                break;
            case ConstDef.STATE_SEND_SUCCESS:
                state = MsgState.MSG_STATE_SEND;
                break;
            case ConstDef.STATE_ARRIVE:
                state = MsgState.MSG_STATE_REC;
                break;
            case ConstDef.STATE_READED:
                state = MsgState.MSG_STATE_READ;
                break;
            case ConstDef.STATE_DESTROY:
                state = MsgState.MSG_STATE_BOMB;
                break;
            default:
                break;
        }
        return state;
    }

    /**
     * 转化消息类型，将TalkMessageBean的消息类型转换为ImMessage消息类型
     *
     * @param messageBean
     * @return
     */
    public static int talkMsgTypeConvert(TalkMessageBean messageBean) {
        int talkMsgType = messageBean.getMessageType();
        if (!messageBean.isGroupMsg()) {
            if (talkMsgType == ConstDef.MSG_TYPE_WEB) {
                // 单聊图文消息
                return MsgPackType.NORMAL_PP_WEB;
            }
            if (messageBean.isBomb()) {
                if (talkMsgType == ConstDef.MSG_TYPE_TEXT) {
                    //闪信文本 0000 1001
                    return MsgPackType.BOMB_PP_TEXT;
                } else {
                    //闪信文件 0000 1010
                    return MsgPackType.BOMB_PP_FILE;
                }
            } else {
                if (talkMsgType == ConstDef.MSG_TYPE_PRESENTATION) {
                    //单聊提示消息 1000 0001
                    return MsgPackType.NOTICE_PP_TEXT;
                } else if (talkMsgType == ConstDef.MSG_TYPE_TEXT) {
                    //文本消息 0000 0001
                    return MsgPackType.NORMAL_PP_TEXT;
                } else if (talkMsgType == ConstDef.MSG_TYPE_CUSTOM) {
                    return MsgPackType.CUSTOM_PP_TEXT;
                } else {
                    //文件消息 0000 0010
                    return MsgPackType.NORMAL_PP_FILE;
                }
            }
        } else {
            if (messageBean.isBomb()) {
                if (talkMsgType == ConstDef.MSG_TYPE_TEXT) {
                    //群组闪信文本消息 0000 1101
                    return MsgPackType.BOMB_PG_TEXT;
                } else {
                    //群组闪信文件消息 0000 1110
                    return MsgPackType.BOMB_PG_FILE;
                }
            } else {
                if (talkMsgType == ConstDef.MSG_TYPE_PRESENTATION) {
                    //群组提示消息 1000 0101
                    return MsgPackType.NOTICE_PG_TEXT;
                } else if (talkMsgType == ConstDef.MSG_TYPE_TEXT) {
                    //群组文本消息 0000 0101
                    return MsgPackType.NORMAL_PG_TEXT;
                } else if (talkMsgType == ConstDef.MSG_TYPE_CUSTOM) {
                    return MsgPackType.CUSTOM_PG_TEXT;
                } else {
                    //群组文件消息 0000 0110
                    return MsgPackType.NORMAL_PG_FILE;
                }
            }
        }
    }

    /**
     * 将实体消息类型转化为业务消息类型
     * <p>
     * 注意：此处不考虑是否为闪信，在应用处理时，要先判断是否为闪信和群组，然后再找对应的文件类型
     *
     * @return 业务消息类型
     */
    @ConstDef.MsgType
    public static int imMsgTypeConvert(IMMessage imMessage) {

        if ((imMessage.getType() & MsgType.MSG_TYPE_NOTICE) ==
                MsgType.MSG_TYPE_NOTICE) {                          //自定义消息
            return ConstDef.MSG_TYPE_PRESENTATION;

        } else if (imMessage.isTextIMMessage()) {                   //文本消息

            return ConstDef.MSG_TYPE_TEXT;

        } else if (imMessage.isFileIMMessage()) {                   //文件消息
            return imFileTypeConvert((IMFileBody) imMessage.getMessageBody());
        }
        return ConstDef.MSG_TYPE_DEFAULT;
    }

    /**
     * 将实体消息类型转化为业务消息类型
     *
     * @param imFileBody 文件类型消息内容
     * @return 消息类型
     */
    public static int imFileTypeConvert(IMFileBody imFileBody) {
        int fileType = imFileBody.getType();
        switch (fileType) {
            case ImSdkFileConstant.FILE_VOICE: //音频
                return ConstDef.MSG_TYPE_VOICE;

            case ImSdkFileConstant.FILE_VIDEO: //视频
                return ConstDef.MSG_TYPE_VIDEO;

            case ImSdkFileConstant.FILE_IMAGE: //图片
                return ConstDef.MSG_TYPE_PHOTO;

            case ImSdkFileConstant.FILE_NORMAL://普通文件
                return ConstDef.MSG_TYPE_FILE;

            default:
                return ConstDef.MSG_TYPE_PRESENTATION;
        }
    }

    /**
     * 普通文件类型转化
     *
     * @param imFileBody
     * @return
     */
    public static int imNormalFileTypeConvert(IMFileBody imFileBody) {
        String suffix = imFileBody.getSuffix();
        if (TextUtils.isEmpty(suffix) ) {
            suffix = IMFileTools.getSuffixFromFilepath(imFileBody.getLocalPath());
        }
        switch (suffix) {
            case FileOption.TXT_SUFFIX:
                return ConstDef.FILE_TYPE_TXT;

            case FileOption.DOC_SUFFIX:
            case FileOption.DOCX_SUFFIX:
                return ConstDef.FILE_TYPE_WORD;

            case FileOption.PPT_SUFFIX:
            case FileOption.PPTX_SUFFIX:
                return ConstDef.FILE_TYPE_PPT;

            case FileOption.XLS_SUFFIX:
            case FileOption.XLSX_SUFFIX:
                return ConstDef.FILE_TYPE_EXCEL;

            case FileOption.PDF_SUFFIX:
            case FileOption.PPS_SUFFIX:
                return ConstDef.FILE_TYPE_PDF;

            case FileOption.RAR_SUFFIX:
            case FileOption.ZIP_SUFFIX:
            case FileOption.GZ_SUFFIX:
                return ConstDef.FILE_TYPE_ZIP;

            case FileOption.APK_SUFFIX:
                return ConstDef.FILE_TYPE_APK;
        }
        return ConstDef.FILE_TYPE_NORMAL;
    }

    /**
     * add by zya ,20170618.
     * @param type 传入的类型
     * @return
     */
    public static ImSdkFileConstant.FileGetType changeObtainFileType(int type) {
        ImSdkFileConstant.FileGetType getType = ImSdkFileConstant.FileGetType.OTHER;
        switch (type) {
            case ConstDef.FILE_EXIT_ALL:
                getType = ImSdkFileConstant.FileGetType.EXIST_ALL;
                break;
            case ConstDef.FILE_ALL_MEDIA:
                getType = ImSdkFileConstant.FileGetType.CUR_ALL_MEDIA;
                break;
            case ConstDef.FILE_ALL_NORMAL:
                getType = ImSdkFileConstant.FileGetType.OTHER;
                break;
            case ConstDef.FILE_ALL:
                getType = ImSdkFileConstant.FileGetType.CUR_ALL;
        }
        return getType;
    }
}
