package com.xdja.im.core.repository.impl.datasource;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.xdja.im.base.di.DiConfig;
import com.xdja.im.base.di.scope.Scoped;
import com.xdja.im.common.cache.interf.ConfigCache;
import com.xdja.im.common.cache.interf.UserCache;
import com.xdja.im.common.network.ApiFactoryMe;
import com.xdja.im.common.network.body.ReportBody;
import com.xdja.im.core.repository.interf.datasource.CloudDataStore;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.entity.NoDisturbConfig;
import com.xdja.im.uikit.entity.NoDisturbSetter;
import com.xdja.im.uikit.entity.SessionTopSetter;
import com.xdja.im.uikit.entity.SettingTopConfig;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.sys.NetworkUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import okhttp3.ResponseBody;
import retrofit2.Response;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.core.repository.impl.datasource
 * @Author: xdjaxa
 * @Creation: 2017-04-11 13:59
 * @Version V1.0
 * @Description:
 */
public class CloudDataStoreImpl implements CloudDataStore {

    private Context context;

    private ApiFactoryMe apiFactory;

    private UserCache userCache;

    private ConfigCache configCache;

    @Inject
    public CloudDataStoreImpl(@Scoped(value = DiConfig.CONTEXT_SCOPE_APP) Context context,
                              ApiFactoryMe apiFactory,
                              UserCache userCache,
                              ConfigCache configCache) {
        this.context = context;
        this.apiFactory = apiFactory;
        this.userCache = userCache;
        this.configCache = configCache;
    }

    @Override
    public void reportPushClientId(String appId, Subscriber<Boolean> subscriber) {

        String account = userCache.get().getAccount();
        String clientId = userCache.get().getClientId();
        int type = configCache.getDevice().getDeviceType();
        if (TextUtils.isEmpty(account) || TextUtils.isEmpty(clientId)) {
            LogUtil.e("Report push clientId error.");
            return;
        }

        ReportBody body = new ReportBody(
                account,
                type,
                clientId,
                appId
        );
        LogUtil.d("ReportBody:" + body.toString());
        apiFactory.getUserSettingApi().loginReport(body)
                .subscribeOn(Schedulers.newThread())
                .map(new Func1<Object, Boolean>() {
                    @Override
                    public Boolean call(Object response) {
                        return true;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void setCloudSessionTop(final String sessionId, Subscriber<Boolean> subscriber) {
        String account = userCache.get().getAccount();
        if (TextUtils.isEmpty(account)) {
            LogUtil.e("Report push clientId error.");
            return;
        }
        final SessionTopSetter setter = new SessionTopSetter();
        setter.setAccount(account);
        setter.setSessionId(sessionId);
        Observable.just(setter)
                .observeOn(Schedulers.newThread())
                .map(new Func1<SessionTopSetter, Response<ResponseBody>>() {
                    @Override
                    public Response<ResponseBody> call(SessionTopSetter setter) {

                        Response<ResponseBody> response = null;
                        try {
                            response = apiFactory.getUserSettingApi()
                                    .saveSessionTopSettings(setter)
                                    .execute();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        return response;
//                        return apiFactory
//                                .getUserSettingApi()
//                                .saveSessionTopSettings(setter);
                    }
                })
                .map(new Func1<Response<ResponseBody>, Boolean>() {
                    @Override
                    public Boolean call(Response<ResponseBody> response) {
//                        if (response != null && response.code() == 200)
                        return Boolean.TRUE;
//                        else return Boolean.FALSE;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void deleteSettingTopAtCloud(String sessionId, Subscriber<Boolean> subscriber) {
        String account = userCache.get().getAccount();
        if (TextUtils.isEmpty(account)) {
            LogUtil.e("Report push clientId error.");
            return;
        }
        final SessionTopSetter setter = new SessionTopSetter();
        setter.setAccount(account);
        setter.setSessionId(sessionId);
        Observable.just(setter)
                .observeOn(Schedulers.newThread())
                .map(new Func1<SessionTopSetter, Response<ResponseBody>>() {
                    @Override
                    public Response<ResponseBody> call(SessionTopSetter sessionTopSetter) {
                        Response<ResponseBody> response = null;
                        try {
                            response = apiFactory.getUserSettingApi().deleteSessionTopSettings(setter).execute();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        return response;
                    }
                })
                .map(new Func1<Response<ResponseBody>, Boolean>() {
                    @Override
                    public Boolean call(Response<ResponseBody> response) {
//                        if (response != null && response.code() == 200)
                        return Boolean.TRUE;
//                        else return Boolean.FALSE;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public Observable<List<SettingTopConfig>> getSettingTopSettingsAtCloud(@NonNull String account) {
        if (NetworkUtil.isNetworkConnected(ImUiKit.getInstance().getContext())) {
            return apiFactory.getUserSettingApi()
                    .getSettingTopSettings(account)
                    .subscribeOn(Schedulers.newThread())
                    .map(new Func1<List<String>, List<SettingTopConfig>>() {
                        @Override
                        public List<SettingTopConfig> call(List<String> strings) {
                            List<SettingTopConfig> results = new ArrayList<>();
                            for (String str : strings) {
                                SettingTopConfig sConfig = new SettingTopConfig();
                                sConfig.setSessionId(str);

                                if (results.size() == 0 || (results.size() != 0 && results.indexOf(sConfig) < 0)) {
                                    results.add(sConfig);
                                }
                            }
                            return results;
                        }
                    });
        } else {
            List<SettingTopConfig> list = new ArrayList<>();
            return Observable.just(list);
        }

    }

    @Override
    public void saveNoDisturb2Cloud(String sessionFlag, String talkerAccount, int sessionType, Subscriber<Boolean> subscriber) {
        String account = userCache.get().getAccount();
        if (TextUtils.isEmpty(account)) {
            LogUtil.e("Report push clientId error.");
            return;
        }
        final NoDisturbSetter noDisturbSetter = new NoDisturbSetter();
        noDisturbSetter.setSessionId(talkerAccount);
        noDisturbSetter.setSessionType(sessionType);
        noDisturbSetter.setAccount(account);
        noDisturbSetter.setSessionFlag(sessionFlag);
        Observable.just(noDisturbSetter)
                .observeOn(Schedulers.newThread())
                .map(new Func1<NoDisturbSetter, Response<ResponseBody>>() {
                    @Override
                    public Response<ResponseBody> call(NoDisturbSetter noDisturbSetter) {
                        Response<ResponseBody> response = null;
                        try {
                            response = apiFactory.getUserSettingApi().saveNoDisturbSettings(noDisturbSetter).execute();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        return response;
                    }
                })
                .map(new Func1<Response<ResponseBody>, Boolean>() {
                    @Override
                    public Boolean call(Response<ResponseBody> response) {
//                        if (response != null && response.code() == 200)
                        return Boolean.TRUE;
//                        else return Boolean.FALSE;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void deleteNoDisturbAtCloud(String sessionFlag, String talkerAccount, int sessionType, Subscriber<Boolean> subscriber) {
        String account = userCache.get().getAccount();
        if (TextUtils.isEmpty(account)) {
            LogUtil.e("Report push clientId error.");
            return;
        }
        final NoDisturbSetter noDisturbSetter = new NoDisturbSetter();
        noDisturbSetter.setSessionId(talkerAccount);
        noDisturbSetter.setAccount(account);
        noDisturbSetter.setSessionFlag(sessionFlag);
        noDisturbSetter.setSessionType(sessionType);
        Observable.just(noDisturbSetter)
                .observeOn(Schedulers.newThread())
                .map(new Func1<NoDisturbSetter, Response<ResponseBody>>() {
                    @Override
                    public Response<ResponseBody> call(NoDisturbSetter noDisturbSetter) {
                        Response<ResponseBody> response = null;
                        try {
                            response = apiFactory.getUserSettingApi().deleteNoDisturbSettings(noDisturbSetter).execute();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        return response;
//                        return apiFactory.getUserSettingApi().deleteNoDisturbSettings(noDisturbSetter);
                    }
                })
                .map(new Func1<Response<ResponseBody>, Boolean>() {
                    @Override
                    public Boolean call(Response<ResponseBody> response) {
//                        if (response != null && response.code() == 200)
                        return Boolean.TRUE;
//                        else return Boolean.FALSE;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public Observable<List<NoDisturbConfig>> getNoDisturbSettingsAtCloud(@NonNull String account) {
        if (NetworkUtil.isNetworkConnected(ImUiKit.getInstance().getContext())) {
            return apiFactory.getUserSettingApi()
                    .getNoDisturbSettings(account)
                    .subscribeOn(Schedulers.newThread())
                    .map(new Func1<List<NoDisturbConfig>, List<NoDisturbConfig>>() {
                        @Override
                        public List<NoDisturbConfig> call(List<NoDisturbConfig> noDisturbConfigs) {
                            return noDisturbConfigs;
                        }
                    });
        } else {
            List<NoDisturbConfig> noDisturbConfigs = new ArrayList<>();
            return Observable.just(noDisturbConfigs);
        }

    }
}
