package com.xdja.im.core.utils;

import android.database.Cursor;
import android.provider.MediaStore;
import android.text.TextUtils;

import com.xdja.im.core.config.ConstDef;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.imsdk.model.IMMessage;
import com.xdja.imsdk.model.body.IMFileBody;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * <p>Author: leiliangliang   </br>
 * <p>Date: 2016/12/13 14:53   </br>
 * <p>Package: com.xdja.imp.util</br>
 * <p>Description: 本地文件相关处理类</br>
 */
public class IMFileTools {
    private final static String CONTACT_FILES_AUTHORITY = "com.android.contacts.files";//用于判断是否是联系人文件
    private final static String DOWNLOAD_FILES_AUTHORITY = "com.android.providers.downloads.documents";//用于判断是否是系统下载中的文件
    private final static String CONTACT_FILES_PATH = "/Android/data/com.android.contacts/cache/";//联系人文件路径
    //document use suffix
    public static final String[] mImageSuffix = new String[]{
            ".bmp",
            ".png",
            ".jpg",
            ".jpeg",
            ".gif"
    };

    //document use suffix
    public static final String[] mDocSuffix = new String[]{
            ".doc",
            ".docx",
            ".xls",
            ".xlsx",
            ".ppt",
            ".pptx",
            ".pdf",
            ".wps",
            ".pps",
            ".txt"
    };

    //zip use suffix
    public static final String[] mZipSuffix = new String[]{
            ".rar",
            ".tar",
            ".zip",
            ".gz"
    };

    //apk use suffix
    public static final String[] mApkSuffix = new String[]{
            ".apk"
    };

    //video use suffix add by ycm 20161222
    public static String[] mVideoSuffix = new String[]{
            ".asf", ". asx", ".avi",
            ".fvi", ".lsf", ".lsx",
            ".mng", ".mov", ".movie",
            ".mp4", ".mpe", ".mpeg",
            ".mpg", ".mpg4", ".pvx",
            ".qt", ".rv", ".vdo",
            ".viv", ".vivo", ".wm",
            ".wmx", ".wv", ".wvx"
    };

    //audio use suffix add by ycm 20161222
    //fix bug 7661 by zya 20170102
    public static String[] mVoiceSuffix = new String[]{
            ".aif", ".aifc", ".aiff", ".snd", ".smz", ".smd",
            ".als", ".au", ".awb", ".rm", ".ram", ".ra",
            ".es", ".esl", ".imy", ".qcp", ".pae", ".pac",
            ".it", ".xmz", ".xm", ".nsnd", ".mpga", ".mp3",
            ".wmv", ".wma", ".wax", ".mp2", ".mod", ".mio",
            ".wav", ".vql", ".vqf", ".midi", ".mid", ".mdz",
            ".vqe", ".vox", ".vib", ".ma5", ".ma3", ".ma2",
            ".ult", ".tsi", ".stm", ".ma1", ".m3url", ".m3u",
            ".s3z", ".s3m", ".rpm", ".m15", ".itz", ".rmf",
            ".rmvb", ".rmm", ".aac", ".3gpp", ".amr", ".flac",
            ".ota", ".kar", ".rtttl", ".xmf", ".mxmf", ".m4a",
            ".sid", ".gsm", ".mka", ".pls", ".sd2"
    };//end by zya


    //image use mimetype
    public static final String[] mImageArgs = new String[]{
            "image/bmp",
            "image/png",
            "image/jpeg",
            "image/jpeg",
            "image/gif"
    };

    //document select args use mimetype
    public static final String[] mDocMimeArgs = new String[]{
            "application/msword",  //doc
            "application/vnd.ms-excel", //xls
            "application/mspowerpoint", //ppt
            "application/pdf",  //pdf
            "application/vnd.ms-works",
            "application/vnd.ms-powerpoint",
            "text/plain" //txt
    };

    //document select args use suffix
    public static final String[] mDocSuffixArgs = new String[]{
            ".docx",
            ".pptx",
            ".xlsx"
    };


    //zip use mimetype
    public static final String[] mZipMimeArgs = new String[]{
            "application/zip",
    };

    //zip use mimetype
    public static final String[] mZipSuffixArgs = new String[]{
            ".tar",
            ".rar",
            ".gz"
    };

    //im share use mimetype
    public static String[] mFilterType = new String[]{
            "audio/",
            "video/",
            "text/x-vcard",
            "application/",
            "web/",
            "*/*"
    };

    public static String[] mShareType = new String[]{
            "audio/",
            "text/x-vcard",
            "application/",
            "*/*"
    };

    private static final String[] mImageSelectionArgs = new String[mImageSuffix.length];
    private static final String[] mDocSelectionArgs = new String[mDocSuffix.length];
    private static final String[] mZipSelectionArgs = new String[mZipSuffix.length];
    private static final String[] mApkSelectionArgs = new String[mApkSuffix.length];

    /**
     * 文档查询条件
     *
     * @return 图片文件查询条件
     */
    private static String buildImageSelection() {
        StringBuilder selection = new StringBuilder();
        for (int i = 0; i < mImageArgs.length; i++) {
            if (i > 0) {
                selection.append(" OR ");
            }
            selection.append(MediaStore.Images.ImageColumns.MIME_TYPE + " =? ");
            mImageSelectionArgs[i] = mImageArgs[i];
        }
        return selection.toString();
    }

    /**
     * 文档查询条件
     *
     * @return 文档文件查询条件
     */
    private static String buildDocSelection() {
        StringBuilder selection = new StringBuilder();
        //按照mime_type查询
        for (int i = 0; i < mDocMimeArgs.length; i++) {
            if (i > 0) {
                selection.append(" OR ");
            }
            selection.append(MediaStore.Files.FileColumns.MIME_TYPE + " =?");
            mDocSelectionArgs[i] = mDocMimeArgs[i];
        }
        //按照后缀查
        int size = mDocMimeArgs.length;
        for (int i = 0; i < mDocSuffixArgs.length; i++) {
            selection.append(" OR ");
            selection.append(MediaStore.Files.FileColumns.DATA + " LIKE ? ");
            mDocSelectionArgs[size + i] = "%" + mDocSuffixArgs[i];
        }
        return selection.toString();
    }

    /**
     * 压缩包查询条件
     *
     * @return 压缩包文件查询条件
     */
    private static String buildZipSelection() {
        StringBuilder selection = new StringBuilder();
        for (int i = 0; i < mZipMimeArgs.length; i++) {
            if (i > 0) {
                selection.append(" OR ");
            }
            selection.append(MediaStore.Files.FileColumns.MIME_TYPE + " =? ");
            mZipSelectionArgs[i] = mZipMimeArgs[i];
        }
        int size = mZipMimeArgs.length;
        for (int i = 0; i < mZipSuffixArgs.length; i++) {
            selection.append(" OR ");
            selection.append(MediaStore.Files.FileColumns.DATA + " LIKE ? ");
            mZipSelectionArgs[size + i] = "%" + mZipSuffixArgs[i];
        }
        return selection.toString();
    }

    /**
     * apk查询条件
     *
     * @return apk应用程序查询条件
     */
    private static String buildApkSelection() {
        StringBuilder selection = new StringBuilder();
        for (int i = 0; i < mApkSuffix.length; i++) {
            if (i > 0) {
                selection.append(" OR ");
            }
            selection.append(MediaStore.Files.FileColumns.DATA + " LIKE ? ");
            mApkSelectionArgs[i] = "%" + mApkSuffix[i];
        }
        return selection.toString();
    }

    /**
     * 是否为图片类型
     *
     * @param path 图片文件绝对路径
     * @return true 图片支持格式 false不支持格式
     */
    private static boolean isImageType(String path) {
        /*String regularExpression = "(?i)^\\w+\\.(doc|docx|xls|xlsx|ppt|pptx|pdf|wps|pps|ppsx)$";
        return path.matches(regularExpression);*/
        return matchStringInArray(path, mImageSuffix);
    }

    /**
     * 是否为视频类型
     *
     * @param path
     * @return
     */
    private static boolean isVideoType(String path) {
        return matchStringInArray(path, mVideoSuffix);
    }

    /**
     * 是否为音频类型
     *
     * @param path
     * @return
     */
    private static boolean isAudioType(String path) {
        return matchStringInArray(path, mVoiceSuffix);
    }

    /**
     * 是否为文档类型
     *
     * @param path 文档文件绝对路径
     * @return
     */
    private static boolean isDocType(String path) {
        return matchStringInArray(path, mDocSuffix);
    }

    /**
     * 是否为压缩文件类型
     *
     * @param path
     * @return
     */
    private static boolean isZipType(String path) {
        return matchStringInArray(path, mZipSuffix);
    }

    /**
     * 是否为apk类型
     *
     * @param path
     * @return
     */
    private static boolean isApkType(String path) {
        return matchStringInArray(path, mApkSuffix);
    }

    /**
     * 文件类型匹配
     *
     * @param path
     * @param arry
     * @return
     */
    private static boolean matchStringInArray(String path, String[] arry) {
        String pathLower = path.toLowerCase(Locale.getDefault());
        for (String suffix : arry) {
            if (pathLower.endsWith(suffix) && (!pathLower.equals(suffix))) {
                return true;
            }
        }
        return false;
    }

    /**
     * 根据文件路径获取文件后缀名
     *
     * @param filepath
     * @return
     */
    public static String getSuffixFromFilepath(String filepath) {
        if (TextUtils.isEmpty(filepath)) {
            return "";
        }
        int pos = filepath.lastIndexOf('.');
        if (pos != -1) {
            return filepath.substring(pos + 1, filepath.length());
        }
        return "";
    }

    /**
     * 获取文档类型后缀名
     *
     * @return
     */
    private static String getDocFileTypeFromFilepath(String filepath) {
        String docFileType = null;
        String suffix = getSuffixFromFilepath(filepath);
        if (suffix != null) {
            suffix = suffix.toLowerCase(Locale.getDefault());
            if ("docx".equals(suffix) || "doc".equals(suffix)) {
                docFileType = "WORD";
            } else if ("xlsx".equals(suffix) || "xls".equals(suffix)) {
                docFileType = "EXCEL";
            } else if ("ppsx".equals(suffix) || "pps".equals(suffix)) {
                docFileType = "PPS";
            } else if ("pptx".equals(suffix) || "ppt".equals(suffix)) {
                docFileType = "PPT";
            } else if ("pdf".equals(suffix) || "wps".equals(suffix)) {
                docFileType = "PDF";
            } else {
                docFileType = suffix.toUpperCase(Locale.getDefault());
            }
        }
        return docFileType;
    }

    /**
     * 根据文件路径名获取文档类型
     *
     * @param filepath
     * @return
     */
    private static int getDocFileType(String filepath) {
        int fileType = 0;
        String suffix = getSuffixFromFilepath(filepath);
        if (suffix != null) {
            suffix = suffix.toLowerCase(Locale.getDefault());
            if ("docx".equals(suffix) || "doc".equals(suffix)) {
                fileType = ConstDef.FILE_TYPE_WORD;
            } else if ("xlsx".equals(suffix) || "xls".equals(suffix)) {
                fileType = ConstDef.FILE_TYPE_EXCEL;
            } else if ("ppsx".equals(suffix) || "pps".equals(suffix) ||
                    "pdf".equals(suffix) || "wps".equals(suffix)) {
                fileType = ConstDef.FILE_TYPE_PDF;
            } else if ("pptx".equals(suffix) || "ppt".equals(suffix)) {
                fileType = ConstDef.FILE_TYPE_PPT;
            } else if ("txt".equals(suffix)) {
                fileType = ConstDef.FILE_TYPE_TXT;
            } else {
                fileType = ConstDef.FILE_TYPE_NORMAL;
            }
        }
        return fileType;
    }

    /**
     * 获取压缩包文件类型
     *
     * @param filepath
     * @return
     */
    private static String getZipFileTypeFromFilepath(String filepath) {
        /**
         ".tar",
         .rar",
         .gz",
         */
        String docFileType = null;
        String suffix = getSuffixFromFilepath(filepath);
        if (suffix != null) {
            suffix = suffix.toLowerCase(Locale.getDefault());
            if ("tar".equals(suffix) || "gtar".equals(suffix)) {
                docFileType = "TAR";
            } else if ("rar".equals(suffix)) {
                docFileType = "RAR";
            } else if ("gz".equals(suffix) || "tgz".equals(suffix)) {
                docFileType = "GZ";
            }
        }
        return docFileType;
    }

    /**
     * ZHU
     * 查询本地媒体库中图片文件, 按照图片类型进行分类
     *
     * @return 图片文件列表
     */
    public static Map<String, List<FileItem>> queryLocalImages() {
        Map<String, List<FileItem>> imageMap = new HashMap<>();

        // 扫描外部设备中的照片
        String projection[] = {
                MediaStore.Images.ImageColumns.DATA,
                MediaStore.Images.ImageColumns.SIZE,
                MediaStore.Images.ImageColumns.DATE_MODIFIED,
                MediaStore.Images.ImageColumns.MIME_TYPE};
        String sortOrder = MediaStore.Images.Media.DATE_MODIFIED + " desc";
        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Images.Media.EXTERNAL_CONTENT_URI, projection, /*buildImageSelection()*/null,
                /*mImageSelectionArgs*/null, sortOrder);
        if (cursor != null) {
            while (cursor.moveToNext()) {
                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Images.ImageColumns.DATA));
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Images.ImageColumns.SIZE));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATE_MODIFIED)) * 1000;
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }

                // 获取该文件的父路径名
                File parentFile = new File(filePath).getParentFile();
                if (parentFile == null) {
                    continue;
                }
                //按照图片文件夹分类
                String parentName = parentFile.getName();
                if (TextUtils.isEmpty(parentName)) {
                    continue;
                }
                FileItem localFileInfo = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, ConstDef.FILE_TYPE_PHOTO);
                if (imageMap.containsKey(parentName)) {
                    imageMap.get(parentName).add(localFileInfo);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(localFileInfo);
                    imageMap.put(parentName, fileInfoList);
                }
            }
            cursor.close();
        }
        return imageMap;
    }

    /**
     * ZHU
     * 查询媒体库中所有音频文件, 按照文件夹来分类
     *
     * @return 音频文件列表
     */
    public static Map<String, List<FileItem>> queryLocalAudios() {
        Map<String, List<FileItem>> audioMap = new HashMap<>();
        // 扫描外部设备中的音频
        String str[] = {MediaStore.Audio.Media._ID,
                MediaStore.Audio.Media.DATA,
                MediaStore.Audio.Media.SIZE,
                MediaStore.Audio.Media.DATE_MODIFIED};
        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Audio.Media.EXTERNAL_CONTENT_URI, str, null,
                null, MediaStore.Audio.Media.DATE_MODIFIED + " desc");
        if (cursor != null) {
            while (cursor.moveToNext()) {

                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Audio.Media.DATA));
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Audio.Media.SIZE));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Audio.Media.DATE_MODIFIED)) * 1000;
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }
                FileItem FileItem = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, ConstDef.FILE_TYPE_VOICE);

                //音频文件路径
                String path = cursor.getString(cursor.getColumnIndex(MediaStore.Audio.Media.DATA));
                // 获取该文件的父路径名
                File parentFile = new File(path).getParentFile();
                if (parentFile == null) {
                    continue;
                }
                String parentName = parentFile.getName();
                if (TextUtils.isEmpty(parentName)) {
                    continue;
                }
                if (audioMap.containsKey(parentName)) {
                    audioMap.get(parentName).add(FileItem);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(FileItem);
                    audioMap.put(parentName, fileInfoList);
                }
            }
            cursor.close();
        }
        return audioMap;
    }

    /**
     * ZHU
     * 查询媒体库中所有视频文件,按照文件夹进行分裂
     *
     * @return 视频文件列表
     */
    public static Map<String, List<FileItem>> queryLocalVideos() {
        Map<String, List<FileItem>> videoMap = new HashMap<>();
        // 扫描外部设备中的视频
        String str[] = {MediaStore.Video.Media._ID,
                MediaStore.Video.Media.DATA,
                MediaStore.Video.Media.SIZE,
                MediaStore.Video.Media.DATE_MODIFIED,
                MediaStore.Video.Media.DESCRIPTION};
        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Video.Media.EXTERNAL_CONTENT_URI, str, null,
                null, MediaStore.Video.Media.DATE_MODIFIED + " desc");
        if (cursor != null) {
            while (cursor.moveToNext()) {
                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Video.Media.DATA));
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Video.Media.SIZE));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Video.Media.DATE_MODIFIED)) * 1000;
                String desc = cursor.getString(cursor.getColumnIndex(MediaStore.Video.Media.DESCRIPTION));
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }
                // 获取该文件的父路径名
                File parentFile = new File(filePath).getParentFile();
                if (parentFile == null) {
                    continue;
                }
                FileItem localFileInfo = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, ConstDef.FILE_TYPE_VIDEO);
                localFileInfo.setExtraInfo(desc);
                String parentName = parentFile.getName();
                if (videoMap.containsKey(parentName)) {
                    videoMap.get(parentName).add(localFileInfo);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(localFileInfo);
                    videoMap.put(parentName, fileInfoList);
                }
            }
            cursor.close();
        }
        return videoMap;
    }

    /**
     * ZHU
     * 查询本地所有应用列表，包括安装和未安装
     *
     * @return 应用列表
     */
    public static Map<String, List<FileItem>> queryLocalDocuments() {

        Map<String, List<FileItem>> documentMap = new HashMap<>();
        // 扫描外部设备中的文档
        String projection[] = {
                MediaStore.Files.FileColumns.DATA,
                MediaStore.Files.FileColumns.SIZE,
                MediaStore.Files.FileColumns.DATE_MODIFIED,
                MediaStore.Files.FileColumns.MIME_TYPE};
        //排序
        String sortOrder = MediaStore.Files.FileColumns.DATE_MODIFIED + " desc";
        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Files.getContentUri("external"), projection, buildDocSelection(),
                mDocSelectionArgs, sortOrder);
        if (cursor != null) {
            while (cursor.moveToNext()) {
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.SIZE));
                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATA));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATE_MODIFIED)) * 1000;
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }
                //按照类型来进行分类
                String fileType = getDocFileTypeFromFilepath(filePath);
                if (TextUtils.isEmpty(fileType)) {
                    continue;
                }
                FileItem localFileInfo = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, getDocFileType(filePath));
                if (documentMap.containsKey(fileType)) {
                    documentMap.get(fileType).add(localFileInfo);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(localFileInfo);
                    documentMap.put(fileType, fileInfoList);
                }
            }
            cursor.close();
        }
        return documentMap;
    }


    /**
     * ZHU
     * 查询本地所有应用列表，包括安装和未安装
     *
     * @return 应用列表
     */
    public static Map<String, List<FileItem>> queryLocalApplication() {

        Map<String, List<FileItem>> applicationMap = new HashMap<>();
        // 扫描外部设备中的
        String str[] = {
                MediaStore.Files.FileColumns.DATA,
                MediaStore.Files.FileColumns.SIZE,
                MediaStore.Files.FileColumns.DATE_MODIFIED};

        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Files.getContentUri("external"), str, buildApkSelection(),
                mApkSelectionArgs, MediaStore.Files.FileColumns.DATE_MODIFIED + " desc");
        if (cursor != null) {
            while (cursor.moveToNext()) {
                //文件名称
                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATA));
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.SIZE));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATE_MODIFIED)) * 1000;
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }
                // 获取该文件的父路径名
                File parentFile = new File(filePath).getParentFile();
                if (parentFile == null) {
                    continue;
                }
                String fileType = getSuffixFromFilepath(filePath);
                if (TextUtils.isEmpty(fileType)) {
                    continue;
                }
                //获取文件父目录名称
                String parentName = parentFile.getName();
                //apk按照已安装和未安装来区分
                FileItem localFileInfo = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, ConstDef.FILE_TYPE_APK);
                if (applicationMap.containsKey(parentName)) {
                    applicationMap.get(parentName).add(localFileInfo);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(localFileInfo);
                    applicationMap.put(parentName, fileInfoList);
                }
            }
            cursor.close();
        }
        return applicationMap;
    }

    /**
     * ZHU
     * 查询本地所有应用列表，包括安装和未安装
     *
     * @return 应用列表
     */
    public static Map<String, List<FileItem>> queryOtherFiles() {
        Map<String, List<FileItem>> othersMap = new HashMap<>();
        // 扫描外部设备中的视频
        String projection[] = {
                MediaStore.Files.FileColumns.DATA,
                MediaStore.Files.FileColumns.SIZE,
                MediaStore.Files.FileColumns.DATE_MODIFIED,
                MediaStore.Files.FileColumns.MIME_TYPE};
        //排序
        String sortOrder = MediaStore.Files.FileColumns.DATE_MODIFIED + " desc ";
        Cursor cursor = ImUiKit.getInstance().getContext().getContentResolver().query(
                MediaStore.Files.getContentUri("external"), projection, buildZipSelection(),
                mZipSelectionArgs, sortOrder);
        if (cursor != null) {
            while (cursor.moveToNext()) {
                //文件名称
                String filePath = cursor.getString(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATA));
                long fileSize = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.SIZE));
                long dateTime = cursor.getLong(cursor.getColumnIndex(MediaStore.Files.FileColumns.DATE_MODIFIED)) * 1000;
                if (fileSize <= 0 || TextUtils.isEmpty(filePath)) {
                    continue;
                }
                //按照类型来进行分类
                String fileType = getZipFileTypeFromFilepath(filePath);
                if (TextUtils.isEmpty(fileType)) {
                    continue;
                } else {
                    fileType = fileType.toUpperCase(Locale.getDefault());
                }
                FileItem localFileInfo = new FileItem(getNameFromFilepath(filePath),
                        filePath, fileSize, dateTime, ConstDef.FILE_TYPE_ZIP);
                if (othersMap.containsKey(fileType)) {
                    othersMap.get(fileType).add(localFileInfo);
                } else {
                    List<FileItem> fileInfoList = new ArrayList<>();
                    fileInfoList.add(localFileInfo);
                    othersMap.put(fileType, fileInfoList);
                }
            }
            cursor.close();
        }
        return othersMap;
    }

    /**
     * ZHU
     * 根据IM消息，获取最近聊天文件
     *
     * @param imMessageList
     * @return
     */
    public static Map<String, List<FileItem>> getLastFileList(List<IMMessage> imMessageList) {

        Map<String, List<FileItem>> lastFileMap = new LinkedHashMap<>();
        for (IMMessage file : imMessageList) {
            if (file != null && file.isFileIMMessage()) {
                //文件排序时间闲标识
                String mapKey = String.valueOf(IMFileTools.getTimeLineType(file.getIMMessageTime()));
                IMFileBody fileBody = (IMFileBody) file.getMessageBody();
                FileItem localFileInfo = new FileItem(fileBody.getDisplayName(),
                        fileBody.getLocalPath(),
                        fileBody.getFileSize(),
                        file.getIMMessageTime(),
                        fileBody.getType());
                if (lastFileMap.containsKey(mapKey)) {
                    lastFileMap.get(mapKey).add(localFileInfo);
                } else {
                    List<FileItem> localFileList = new ArrayList<>();
                    localFileList.add(localFileInfo);
                    lastFileMap.put(mapKey, localFileList);
                }
            }
        }
        return lastFileMap;

    }

    /**
     * 根据文件路径获取文件名称
     *
     * @param filepath
     * @return
     */
    private static String getNameFromFilepath(String filepath) {
        int pos = filepath.lastIndexOf(File.separator);
        if (pos != -1) {
            return filepath.substring(pos + 1);
        }
        return "";
    }

    //*****************************************************************
    //*           以下为最新文件相关信息
    //*****************************************************************

    /**
     * 一天24小时时间毫秒数
     */
    private static final long ONE_DAY_TIME_MILLIS = 24 * 60 * 60 * 1000;

    public static final int TIME_WITHIN_TODAY = 0;  //今天内
    public static final int TIME_YESTERDAY = 1;     //昨天
    public static final int TIME_WITHIN_WEEK = 2;   //一周内
    public static final int TIME_WITHIN_MONTH = 3;  //一个月内
    public static final int TIME_WITHIN_MONTH_AGO = 4;  //一个月前

    /**
     * 根据时间，获取不同的时间线
     *
     * @param millis 时间毫秒
     * @return 时间线类型
     */
    public static int getTimeLineType(long millis) {
        int timeLineType;
        if (millis > getTimeToMorning()) {
            //一天内，今天0点到现在
            timeLineType = TIME_WITHIN_TODAY;

        } else if (millis > getTimeToYesterdayMorning()) {
            //昨天0点到现在时间
            timeLineType = TIME_YESTERDAY;

        } else if (millis > getTimeToWeekMorning()) {
            //一周内，本周一0点到现在
            timeLineType = TIME_WITHIN_WEEK;

        } else if (millis > getTimeToMonthMorning()) {
            //一个月内，本月一号0点到现在
            timeLineType = TIME_WITHIN_MONTH;
        } else {
            //上个月以及以前
            timeLineType = TIME_WITHIN_MONTH_AGO;
        }
        return timeLineType;
    }

    /**
     * 获得当天0点时间
     *
     * @return 时间毫秒
     */
    private static long getTimeToMorning() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }

    /**
     * 获取两天内时间，到昨天0点时间
     *
     * @return 时间毫秒
     */
    private static long getTimeToYesterdayMorning() {
        return getTimeToMorning() - ONE_DAY_TIME_MILLIS;
    }

    /**
     * 获得本周一0点时间
     *
     * @return 时间毫秒
     */
    private static long getTimeToWeekMorning() {
        Calendar cal = Calendar.getInstance();
        cal.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH),
                cal.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        return cal.getTimeInMillis();
    }

    /**
     * 获得本月第一天0点时间
     *
     * @return 时间毫秒
     */
    private static long getTimeToMonthMorning() {
        Calendar cal = Calendar.getInstance();
        cal.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH),
                cal.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
        cal.set(Calendar.DAY_OF_MONTH, cal.getActualMinimum(Calendar.DAY_OF_MONTH));
        return cal.getTimeInMillis();
    }


    /**
     * 根据文件绝对路径，获取文件类型
     *
     * @param filePath
     * @return
     */
    public static int getFileTypeFromFilepath(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return 0;
        }
        if (isImageType(filePath)) {
            return ConstDef.FILE_TYPE_PHOTO;
        } else if (isVideoType(filePath)) {
            return ConstDef.FILE_TYPE_VIDEO;
        } else if (isAudioType(filePath)) {
            return ConstDef.FILE_TYPE_VOICE;
        } else if (isDocType(filePath)) {
            return getDocFileType(filePath);
        } else if (isApkType(filePath)) {
            return ConstDef.FILE_TYPE_APK;
        } else if (isZipType(filePath)) {
            return ConstDef.FILE_TYPE_ZIP;
        }
        return 0;
    }

    /**
     * 根据文件类型获取对应的资源id
     *
     * @param fileType
     * @return
     */
    public static int getFileIconResId(int fileType) {
        int resId;
        switch (fileType) {
            case ConstDef.FILE_TYPE_VOICE:
                resId = R.mipmap.im_uikit_ic_music;
                break;
            case ConstDef.FILE_TYPE_VIDEO:
                resId = R.mipmap.im_uikit_ic_video;
                break;
            case ConstDef.FILE_TYPE_PHOTO:
                resId = R.mipmap.im_uikit_ic_jpg;
                break;
            case ConstDef.FILE_TYPE_WORD:
                resId = R.mipmap.im_uikit_ic_doc;
                break;
            case ConstDef.FILE_TYPE_TXT:
                resId = R.mipmap.im_uikit_ic_text;
                break;
            case ConstDef.FILE_TYPE_EXCEL:
                resId = R.mipmap.im_uikit_ic_excel;
                break;
            case ConstDef.FILE_TYPE_PDF:
                resId = R.mipmap.im_uikit_ic_pdf;
                break;
            case ConstDef.FILE_TYPE_PPT:
                resId = R.mipmap.im_uikit_ic_ppt;
                break;
            case ConstDef.FILE_TYPE_APK:
                resId = R.mipmap.im_uikit_ic_apk;
                break;
            case ConstDef.FILE_TYPE_ZIP:
                resId = R.mipmap.im_uikit_ic_others;
                break;
            default:
                resId = R.mipmap.im_uikit_ic_others;
        }
        return resId;
    }
}
