package com.xdja.im.lib.historyfile.adapter;

import android.app.Activity;
import android.content.Intent;
import android.os.SystemClock;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.CheckBox;
import android.widget.ImageView;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.lib.filemanage.adapter.BaseRecyclerViewAdapter;
import com.xdja.im.lib.filemanage.model.GridItemBean;
import com.xdja.im.lib.filemanage.util.DateUtils;
import com.xdja.im.lib.filemanage.util.HistoryFileUtils;
import com.xdja.im.lib.filemanage.util.XToast;
import com.xdja.im.lib.filemanage.view.AdapterVu;
import com.xdja.im.lib.historyfile.command.IHistoryPhotoAndVideoAdapterCommand;
import com.xdja.im.lib.historyfile.ui.HistoryFileActivityPresenter;
import com.xdja.im.lib.historyfile.util.HistoryModeUtils;
import com.xdja.im.lib.historyfile.vu.ViewHistoryGridTitleItem;
import com.xdja.im.lib.historyfile.vu.ViewHistoryPhotoItem;
import com.xdja.im.lib.historyfile.vu.ViewHistoryVideoItem;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.activity.FilePreviewActivity;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.view.CircleProgressBar;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;


public class HistoryPhotoAndVideoAdapter extends BaseRecyclerViewAdapter<IHistoryPhotoAndVideoAdapterCommand, GridItemBean>
        implements IHistoryPhotoAndVideoAdapterCommand {

    private static final int PARENT_DIR = 0;
    private static final int CHILD_LEAF = 1;
    private static final int CHILD_LEAF_VIDEO = 2;

    private static final int MIN_CLICK_DELAY_TIME = 500;
    private BusProvider busProvider;

    private List<GridItemBean> dataSources;
    private Activity activity;

    private long lastClickTime = 0;
    private CommonSpanSizeLookup spanSizeLookup;

    private RecyclerView recyclerView;

    private String talkFlag;
    private int mCurrentOperate;

    private List<Class<? extends AdapterVu<IHistoryPhotoAndVideoAdapterCommand, GridItemBean>>> vuClasses;

//    @Inject
//    Lazy<ChangeMsgState> changeMsgState;

    @Inject
    Lazy<ProxyRepository> proxyRepository;

    public HistoryPhotoAndVideoAdapter(String talkFlag, BusProvider busProvider, List<GridItemBean> dataSources) {
        this.talkFlag = talkFlag;
        this.busProvider = busProvider;
        this.dataSources = dataSources;
    }

    @Override
    public GridItemBean getDataSource(int position) {
        return dataSources != null && dataSources.size() > 0 ? dataSources.get(position) : null;
    }

    @Override
    public IHistoryPhotoAndVideoAdapterCommand getCommand() {
        return this;
    }

    @Override
    public void sendReadReceipt(final TalkMessageBean talkMessageBean) {
        proxyRepository.get().changeMessageState(talkMessageBean, ConstDef.STATE_READED, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(Integer integer) {
                if (integer == 0) {
                    LogUtil.d("", "消息已读状态更改成功");
                    for (GridItemBean gridItemBean : dataSources) {
                        if (gridItemBean.isLeaf()) {
                            TalkMessageBean bean = (TalkMessageBean) gridItemBean.getValue();
                            bean.setMessageState(ConstDef.STATE_READED);
                            notifyDataSetChanged();
                        }
                    }
                } else {
                    LogUtil.d("", "消息已读状态更改失败");
                }
            }
        });
    }

    @Override
    protected List<Class<? extends AdapterVu<IHistoryPhotoAndVideoAdapterCommand, GridItemBean>>> getVuClasses() {
        if (vuClasses == null) {
            vuClasses = new ArrayList<>();
            vuClasses.add(ViewHistoryGridTitleItem.class);
            vuClasses.add(ViewHistoryPhotoItem.class);
            vuClasses.add(ViewHistoryVideoItem.class);
        }

        return vuClasses;
    }

    @Override
    public int getItemViewType(int position) {

        if (dataSources.get(position).isLeaf()) {
            TalkMessageBean dataTalk = (TalkMessageBean) dataSources.get(position).getValue();
            FileInfo fileInfo = dataTalk.getFileInfo();
            if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VIDEO) {
                return CHILD_LEAF_VIDEO;
            } else if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO) {
                return CHILD_LEAF;
            }
        }
        return PARENT_DIR;
    }

    @Override
    protected void clickChildLeaf(View view, int position) {
        if (mCurrentOperate != 0) {
            CheckBox checkBox = null;
            switch (getItemViewType(position)) {
                case CHILD_LEAF:
                    checkBox = (CheckBox) view.findViewById(R.id.chx_img_file_select);
                    break;
                case CHILD_LEAF_VIDEO:
                    checkBox = (CheckBox) view.findViewById(R.id.chx_video_file_select);
                    break;
                default:
                    break;
            }
            if (checkBox != null) {
                if (HistoryModeUtils.getInstance().getSelectedFileCount() >= 9 && !checkBox.isChecked() &&
                        mCurrentOperate != ConstDef.FILE_OPERATE_DELETE) {
                    new XToast(getActivity()).display(String.format(activity.getString(R.string.im_uikit_select_file_hint), 9));
                    return;
                }
                checkBox.setChecked(!checkBox.isChecked());
            }
        } else {
            TalkMessageBean talkMessageBean = (TalkMessageBean) dataSources.get(position).getValue();
            //fix bug 19158 by jyg 20170902 start
            //此处保持和聊天页面点击短视频业务一致，如果没有下载的短视频不让点击进入
            if (!HistoryFileUtils.isFileExist(talkMessageBean)) {
                if (DateUtils.isOverdue(talkMessageBean.getShowTime())) {
                    new XToast(activity).display(R.string.im_uikit_history_select_all_overdue);
                    return;
                } else {
                    FileInfo fileInfo = talkMessageBean.getFileInfo();
                    if (fileInfo instanceof VideoFileInfo) {
                        if (((VideoFileInfo) fileInfo).getFileSize() != 0) {
                            List<FileInfo> downloadInfos = new ArrayList<FileInfo>();
                            ((VideoFileInfo) fileInfo).setType(ConstDef.IS_RAW);
                            downloadInfos.add(fileInfo);
                            ((HistoryFileActivityPresenter) getActivity()).showDialogDownloadFile(downloadInfos);
                            return;
                        }
                    }
                }
            }

            //fix bug 19158 by jyg 20170902 end
//            Intent intent = new Intent(getActivity(), FilePreviewActivity.class);
//            intent.putExtra(ConstDef.TAG_TALKFLAG, talkFlag);
//            intent.putExtra(ConstDef.MSG_ID, talkMessageBean.getId());
//            intent.putExtra(ConstDef.FROM_FILE_HISTORY, true);
//            getActivity().startActivity(intent);

            Intent intent = new Intent();
            intent.setClass(getActivity(), FilePreviewActivity.class);
            intent.putExtra(Constant.TAG_TALKERID, talkFlag);
            intent.putExtra(Constant.TAG_MSGID, talkMessageBean.getId());
            intent.putExtra(ConstDef.FROM_FILE_HISTORY, true);
            getActivity().startActivity(intent);
        }
    }

    @Override
    protected void collapseOrExpand(int position) {
        GridItemBean bean = dataSources.get(position);
        List<TalkMessageBean> childLists = bean.getChildList();
        if (childLists == null) {
            return;
        }
        if (bean.isExpand()) {
            //折叠
            for (TalkMessageBean fInfo : childLists) {
                GridItemBean<TalkMessageBean> deleteBean = new GridItemBean();
                deleteBean.setValue(fInfo);
                dataSources.remove(deleteBean);
            }
            bean.setExpand(!bean.isExpand());
            notifyItemRangeRemoved(position + 1, childLists.size());
        } else {
            //显示
            List<GridItemBean> addChildItem = new ArrayList<>();
            for (TalkMessageBean fInfo : childLists) {
                GridItemBean<TalkMessageBean> addBean = new GridItemBean();
                addBean.setValue(fInfo);
                addBean.setChildList(null);
                addChildItem.add(addBean);
            }
            dataSources.addAll(position + 1, addChildItem);
            bean.setExpand(!bean.isExpand());
            notifyItemRangeInserted(position + 1, addChildItem.size());
        }
        notifyItemChanged(position);
    }

    @Override
    public int getItemCount() {
        return dataSources != null ? dataSources.size() : 0;
    }

    public void setActivity(Activity activity) {
        this.activity = activity;
    }

    @Override
    public Activity getActivity() {
        return activity;
    }

    @Override
    public void onClick(View v) {
        int position = recyclerView.getChildAdapterPosition(v);
        if (position < 0 || position >= dataSources.size()) {
            LogUtil.d("", "error position = " + position);
            return;
        }
        switch (getItemViewType(position)) {
            case PARENT_DIR:
                long currentTime = SystemClock.elapsedRealtime();
                if (currentTime - lastClickTime > MIN_CLICK_DELAY_TIME) {
                    lastClickTime = currentTime;
                    collapseOrExpand(position);
                }
                break;
            case CHILD_LEAF:
            case CHILD_LEAF_VIDEO:
                clickChildLeaf(v, position);
                break;
            default:
                break;
        }
    }

    @Override
    public boolean onLongClick(View v) {
        int position = recyclerView.getChildAdapterPosition(v);
        if (position < 0 || position >= dataSources.size()) {
            LogUtil.d("", "error position = " + position);
            return false;
        }
        switch (getItemViewType(position)) {
            case PARENT_DIR:
                break;
            case CHILD_LEAF:
            case CHILD_LEAF_VIDEO:
                if (mCurrentOperate == 0) {
                    TalkMessageBean talkMessageBean = (TalkMessageBean) dataSources.get(position).getValue();
                    List<Long> deleteIds = new ArrayList<>();
                    deleteIds.add(talkMessageBean.getId());
                    List<TalkMessageBean> deleteBeans = new ArrayList<>();
                    deleteBeans.add(talkMessageBean);
                    ((HistoryFileActivityPresenter) getActivity()).showRemoveDialog(deleteIds, deleteBeans);
                    return true;
                }
                break;
            default:
                break;
        }
        return false;
    }

    private CommonSpanSizeLookup getSpanSizeLookup() {
        if (spanSizeLookup == null) {
            spanSizeLookup = new CommonSpanSizeLookup();
        }
        return spanSizeLookup;
    }

    public void setLayoutManager(int gridColumn) {
        GridLayoutManager layoutManager = new GridLayoutManager(activity, gridColumn);
        layoutManager.setSpanSizeLookup(getSpanSizeLookup());
        recyclerView.setLayoutManager(layoutManager);
    }

    public void setRecyclerView(RecyclerView recyclerView) {
        this.recyclerView = recyclerView;
    }

    @Override
    protected Class<? extends AdapterVu<IHistoryPhotoAndVideoAdapterCommand, GridItemBean>> getVuClassByViewType(int itemViewType) {
        Class<? extends AdapterVu<IHistoryPhotoAndVideoAdapterCommand, GridItemBean>> clazz = null;
        switch (itemViewType) {
            case CHILD_LEAF:
                clazz = getVuClasses().get(1);
                break;
            case PARENT_DIR:
                clazz = getVuClasses().get(0);
                break;
            case CHILD_LEAF_VIDEO:
                clazz = getVuClasses().get(2);
                break;
        }
        return clazz;
    }

    @Override
    public void toRefreshSelectHint() {
        IMProxyMessageEvent.HistoryRefreshSelectHintEvent event = new IMProxyMessageEvent.HistoryRefreshSelectHintEvent();
        busProvider.post(event);
    }

    public void updateMode(int operateAction) {
        mCurrentOperate = operateAction;
        Iterator iterator = dataSources.iterator();
        while (iterator.hasNext()) {
            GridItemBean gridItemBean = (GridItemBean) iterator.next();
            if (gridItemBean.isLeaf()) {
                TalkMessageBean bean = (TalkMessageBean) gridItemBean.getValue();
                if (mCurrentOperate == ConstDef.FILE_OPERATE_TRANSMIT) {
                    if (!HistoryFileUtils.isFileExist(bean)) {
                        iterator.remove();
                        continue;
                    }
                } else if (mCurrentOperate == ConstDef.FILE_OPERATE_DOWNLOAD) {
                    if (HistoryFileUtils.isFileExist(bean) || (!HistoryFileUtils.isFileExist(bean) &&
                            !bean.isMine() && DateUtils.isOverdue(bean.getShowTime()))) {
                        iterator.remove();
                        continue;
                    }
                }
                bean.setFileOperate(mCurrentOperate);
                if (mCurrentOperate != 0) {
                    bean.setChecked(false);
                }
            } else {
                List<TalkMessageBean> childList = gridItemBean.getChildList();
                Iterator it = childList.iterator();
                while (it.hasNext()) {
                    TalkMessageBean talkMessageBean = (TalkMessageBean) it.next();
                    if (mCurrentOperate == ConstDef.FILE_OPERATE_TRANSMIT) {
                        if (!HistoryFileUtils.isFileExist(talkMessageBean)) {
                            it.remove();
                            if (childList.isEmpty()) {
                                iterator.remove();
                            }
                            continue;
                        }
                    } else if (mCurrentOperate == ConstDef.FILE_OPERATE_DOWNLOAD) {
                        if (HistoryFileUtils.isFileExist(talkMessageBean) || (!HistoryFileUtils.isFileExist(talkMessageBean) &&
                                !talkMessageBean.isMine() && DateUtils.isOverdue(talkMessageBean.getShowTime()))) {
                            it.remove();
                            if (childList.isEmpty()) {
                                iterator.remove();
                            }
                            continue;
                        }
                    }
                    talkMessageBean.setFileOperate(mCurrentOperate);
                    if (mCurrentOperate != 0) {
                        talkMessageBean.setChecked(false);
                    }
                }
            }
        }
        notifyDataSetChanged();
    }

    private class CommonSpanSizeLookup extends GridLayoutManager.SpanSizeLookup {

        @Override
        public int getSpanSize(int position) {
            int viewType = getItemViewType(position);
            if (viewType == PARENT_DIR) {
                return 4;
            }
            return 1;
        }
    }

    public int getLeafPosition(long msgId) {
        int position = -1;
        for (GridItemBean gridItemBean : dataSources) {
            position++;
            if (gridItemBean.isLeaf()) {
                TalkMessageBean talkMessageBean = (TalkMessageBean) gridItemBean.getValue();
                if (talkMessageBean.getId() == msgId) {
                    return position;
                }
            }
        }
        return -1;
    }

    public void refreshItem(long msgId, FileInfo fileInfo) {
        int leafPosition = getLeafPosition(msgId);
        GridLayoutManager gridLayoutManager = (GridLayoutManager) recyclerView.getLayoutManager();
        int firstVisiblePos = gridLayoutManager.findFirstVisibleItemPosition();
        int lastVisblePos = gridLayoutManager.findLastVisibleItemPosition();
        if (leafPosition < 0 || leafPosition >= dataSources.size() || leafPosition < firstVisiblePos || lastVisblePos > lastVisblePos) {
            return;
        }
        GridItemBean gridItemBean = dataSources.get(leafPosition);
        TalkMessageBean itemBean = (TalkMessageBean) gridItemBean.getValue();
        int aPos = leafPosition - firstVisiblePos;
        View itemView = recyclerView.getChildAt(aPos);
        if (itemView == null || itemBean == null) {
            return;
        }
        itemBean.getFileInfo().setTranslatePercent(fileInfo.getTranslatePercent());
        CircleProgressBar cpb = null;
        switch (getItemViewType(leafPosition)) {
            case CHILD_LEAF:
                cpb = (CircleProgressBar) itemView.findViewById(R.id.image_down_cpb);
                break;
            case CHILD_LEAF_VIDEO:
                cpb = (CircleProgressBar) itemView.findViewById(R.id.video_down_cpb);
                ImageView playButton = (ImageView) itemView.findViewById(R.id.history_file_video_btn_player);
                if (playButton != null) {
                    playButton.setVisibility(fileInfo.getFileState() == ConstDef.LOADING ? View.GONE : View.VISIBLE);
                }
                break;
            default:
                return;
        }
        if (cpb != null) {
            if (fileInfo.getFileState() == ConstDef.LOADING) {
                cpb.setVisibility(View.VISIBLE);
                cpb.setProgress(itemBean.getFileInfo().getTranslatePercent());
            } else {
                cpb.setVisibility(View.GONE);
            }
        }
    }

    public void randomRemoveLeafItem(long msgId) {
        int leafPosition = getLeafPosition(msgId);
        if (leafPosition < 0 || leafPosition >= dataSources.size()) {
            return;
        }
        if (!dataSources.get(leafPosition - 1).isLeaf()) {
            if (leafPosition + 1 == dataSources.size() || (leafPosition + 1 < dataSources.size() &&
                    !dataSources.get(leafPosition + 1).isLeaf())) {
                removeLeafBeanFromDir((TalkMessageBean) dataSources.get(leafPosition).getValue());
                dataSources.remove(leafPosition);
                dataSources.remove(leafPosition - 1);
                notifyItemRangeRemoved(leafPosition - 1, 2);
                return;
            }
        }
        removeLeafBeanFromDir((TalkMessageBean) dataSources.get(leafPosition).getValue());
        dataSources.remove(leafPosition);
        notifyItemRangeRemoved(leafPosition, 1);
    }

    private void removeLeafBeanFromDir(TalkMessageBean deletBean) {
        for (GridItemBean gridItemBean : dataSources) {
            if (!gridItemBean.isLeaf() && gridItemBean.getChildList().contains(deletBean)) {
                gridItemBean.getChildList().remove(deletBean);
            }
        }
    }
}
