package com.xdja.im.lib.audio.player;

import android.media.AudioManager;
import android.media.MediaPlayer;
import android.text.TextUtils;

import com.xdja.im.lib.audio.manager.VoicePlayState;
import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.action.media.MediaManger;
import com.xdja.im.uikit.utils.IoUtils;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.FileInputStream;

/**
 * @Package: com.xdja.im.lib.audio.player
 * @Author: xdjaxa
 * @Creation: 2017-04-24 11:58
 * @Version V1.0
 * @Description: 语音播放管理类
 */
public class AudioPlayerManager {

    /**
     * 媒体播放器
     */
    private MediaPlayer mMediaPlayer;
    /**
     * 当前播放媒体文件路径
     */
    private String mCurPlayPath;
    /**
     * 播放相关参数信息
     */
    private Object[] mArgs;

    private static class SingletonInstance {
        private static final AudioPlayerManager INSTANCE = new AudioPlayerManager();
    }

    public static AudioPlayerManager getInstance() {
        return SingletonInstance.INSTANCE;
    }

    private AudioPlayerManager() {
    }

    /**
     * 播放对应的音频文件
     *
     * @param filePath 文件路径
     */
    public void startToPlay(String filePath, Object... args) {

        synchronized (this){
            LogUtil.d("Start play voice " + filePath);
            if (!checkNullForPath(filePath)) {
                return;
            }

            if (mMediaPlayer == null) {
                mMediaPlayer = new MediaPlayer();
//                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
                mMediaPlayer.setOnErrorListener(errorListener);
                mMediaPlayer.setOnCompletionListener(completionListener);
            } else {
                if (mMediaPlayer.isPlaying()) {
                    stopToPlay(VoicePlayState.STOP, args);
                }
                mMediaPlayer.reset();
            }

            ///播放模式设置
            if (MediaManger.getInstance().isWakeAcquire()) {
                MediaManger.getInstance().changeToEarpieceMode();
                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
            } else {
                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
            }
            if (TextUtils.equals(mCurPlayPath, filePath)) {
                stopToPlay(VoicePlayState.STOP, args);
                AudioFunctions.muteAudioFocus(false);
                return;
            }
            AudioFunctions.muteAudioFocus(true);
            FileInputStream fis = null;
            try {
                File file = new File(filePath);
                fis = new FileInputStream(file);
                mMediaPlayer.setDataSource(fis.getFD());
                mMediaPlayer.prepare();
                mMediaPlayer.start();
                mCurPlayPath = filePath;
                this.mArgs = args;
                AudioListenerManager.getInstance().playStateChange(
                        mCurPlayPath, VoicePlayState.PLAYING, args);
            } catch (Exception e) {
                LogUtil.e("ERROR:Play voice failed. " + e.getMessage());
                stopToPlay(VoicePlayState.ERROR, args);
            } finally {
                IoUtils.closeIO(fis);
            }
        }
    }

    /**
     * 对文件信息进行必要的校验
     *
     * @param filePath
     * @return
     */
    private boolean checkNullForPath(String filePath) {
        if (filePath == null || TextUtils.isEmpty(filePath)) {
            return false;
        }

        File file = new File(filePath);
        return file.exists();
    }

    /**
     * 停止播放
     *
     * @param state 播放状态,因为某种原因停止播放
     */
    private synchronized void stopToPlay(VoicePlayState state, Object... args) {
        LogUtil.d("Stop play voice " + mCurPlayPath);
        if (mMediaPlayer != null) {
            try {
                if (mMediaPlayer.isPlaying()) {
                    mMediaPlayer.stop();
                }
                mMediaPlayer.release();
            } catch (IllegalStateException e) {
                e.printStackTrace();
            }
            mMediaPlayer = null;
            AudioFunctions.muteAudioFocus(false);
        }

        mCurPlayPath = null;
        this.mArgs = null;
        AudioListenerManager.getInstance().playStateChange(
                mCurPlayPath,
                state,
                args);
    }

    public void stopPlay() {
        stopToPlay(VoicePlayState.STOP, mArgs);
    }

    /**
     * 释放MediaPlayer,调用方式在每个页面销毁的时候
     * 尽量释放掉不用的MediaPlayer
     */
    public void releasePlayer() {
        stopToPlay(VoicePlayState.STOP, mArgs);
        if (mMediaPlayer != null) {
            mMediaPlayer.release();
            mMediaPlayer = null;
        }
    }

    public String getCurrentPlayPath() {
        return mCurPlayPath;
    }

    public Object[] getArgs() {
        return mArgs;
    }

    /**
     * 判断当前是否在播放
     *
     * @return
     */
    public boolean isPlaying() {
        return mMediaPlayer != null
                && mMediaPlayer.isPlaying()
                && !TextUtils.isEmpty(mCurPlayPath)
                && mArgs != null && mArgs.length > 0;
    }

    /**
     * 播放出错的监听事件
     */
    private MediaPlayer.OnErrorListener errorListener = new MediaPlayer.OnErrorListener() {
        @Override
        public boolean onError(MediaPlayer mp, int what, int extra) {
            stopToPlay(VoicePlayState.ERROR, mArgs);
            return false;
        }
    };

    /**
     * 播放完成的监听
     */
    private MediaPlayer.OnCompletionListener completionListener = new MediaPlayer.OnCompletionListener() {
        @Override
        public void onCompletion(MediaPlayer mp) {
            stopToPlay(VoicePlayState.COMPLETION, mArgs);
            MediaManger.getInstance().restoreAudioMode();
        }
    };
}
