package com.xdja.im.uikit.ui.activity;

import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;

import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.imageloader.ImageLoader;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyFileEvent;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.repository.impl.im.IMProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.adapter.FilePreviewAdapter;
import com.xdja.im.uikit.ui.presenter.FilePreviewPresenter;
import com.xdja.im.uikit.ui.view.ViewFilePreview;
import com.xdja.im.uikit.ui.view.vu.FilePreviewVu;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.uikit.ui.activity
 * @Author: xdjaxa
 * @Creation: 2017-05-15 10:15
 * @Version V1.0
 * @Description:图片视频文件预览界面
 */
public class HistoryFilePreviewActivity extends ImBaseActivity<FilePreviewPresenter, FilePreviewVu>
        implements FilePreviewPresenter, FilePreviewAdapter.SettingMsgIdCallback {

    @Inject
    Lazy<IMProxyImpl> imProxy;
    /**
     * IMSDK回调
     */
    @Inject
    Lazy<ProxyRepository> proxyRepository;

    @Inject
    ImageLoader imageLoader;
    /**
     * 会话ID
     */
    private String mTalkId;
    /**
     * 当前点击消息的ID
     */
    private long mMsgId;
    /**
     * 文件消息列表
     */
    private List<TalkMessageBean> mMessageList = new ArrayList<>();
    /**
     * 适配器
     */
    private FilePreviewAdapter mAdapter;

    @NonNull
    @Override
    protected Class<? extends FilePreviewVu> getVuClass() {
        return ViewFilePreview.class;
    }

    @NonNull
    @Override
    protected FilePreviewPresenter getPresenter() {
        return this;
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        getWindow().getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
        getWindow().addFlags(View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION);
    }

    @Override
    protected void preBindView(@Nullable Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);
        Intent intent = getIntent();
        if (intent != null) {
            mTalkId = intent.getStringExtra(Constant.TAG_TALKERID);
            mMsgId = intent.getLongExtra(Constant.TAG_MSGID, 0);

            LogUtil.d("Start preview Image, current msgId :" + mMsgId + ", mTalkId:" + mTalkId);
        }
    }

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (mComponent == null) {
            return;
        }
        mComponent.inject(this);
        BusProvider.getInstance().register(this);

        mAdapter = new FilePreviewAdapter(this, mMessageList,
                imProxy.get(), imageLoader, this);
        getVu().setAdapter(mAdapter);

        loadFileData();
    }

    private void loadFileData() {

        proxyRepository.get().getSessionFileList(mTalkId, 0, 0, new Subscriber<List<TalkMessageBean>>() {
            @Override
            public void onCompleted() {
                LogUtil.d("load session file finished.");
                getVu().setProgressVisibility(View.GONE);
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: load session error, " + e.getMessage());
            }

            @Override
            public void onNext(List<TalkMessageBean> talkMessageBeen) {

                mMessageList.clear();
                Collections.reverse(talkMessageBeen);
                mMessageList.addAll(talkMessageBeen);
                mAdapter.notifyDataSetChanged();

                //当前显示位置
                getVu().setCurrentPosition(getCurrentPosition());
            }
        });
    }

    /**
     * 获取当显示图片的位置
     *
     * @return
     */
    private int getCurrentPosition() {
        int size = mMessageList.size();
        for (int i = 0; i < size; i++) {
            TalkMessageBean bean = mMessageList.get(i);
            if (bean != null && bean.getId() == mMsgId) {
                return i;
            }
        }
        LogUtil.w("Not math the current msgId :" + mMsgId);
        return 0;
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (imageLoader != null) {
            imageLoader.clearMemoryCache(this);
        }
    }

    /**
     * 事件和消息匹配
     *
     * @param messageId 事件ID
     * @return
     */
    private int mathMessage(long messageId) {
        int size = mMessageList.size();
        for (int i = size - 1; i >= 0; i--) {
            if (messageId == mMessageList.get(i).getId()) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 更新文件实体信息类
     *
     * @param fileInfo
     * @param eventFile
     */
    private void updateFileInfo(FileInfo fileInfo, FileInfo eventFile) {
        if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO) {
            ImageFileInfo eventFileInfo = (ImageFileInfo) eventFile;
            ImageFileInfo imageFileInfo = (ImageFileInfo) fileInfo;

            FileInfo hdFileInfo = eventFileInfo.getHdFileInfo();
            if (hdFileInfo != null) {
                imageFileInfo.getHdFileInfo().setFilePath(hdFileInfo.getFilePath());
                imageFileInfo.getHdFileInfo().setTranslateSize(hdFileInfo.getTranslateSize());
                imageFileInfo.getHdFileInfo().setFileState(hdFileInfo.getFileState());
            }

            FileInfo rawFileInfo = eventFileInfo.getRawFileInfo();
            if (rawFileInfo != null) {
                imageFileInfo.getRawFileInfo().setFilePath(rawFileInfo.getFilePath());
                imageFileInfo.getRawFileInfo().setTranslateSize(rawFileInfo.getTranslateSize());
                imageFileInfo.getRawFileInfo().setFileState(rawFileInfo.getFileState());
            }

        } else if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VIDEO) {

            VideoFileInfo eventFileInfo = (VideoFileInfo) eventFile;
            VideoFileInfo videoFileInfo = (VideoFileInfo) fileInfo;

            FileInfo rawFileInfo = eventFileInfo.getRawFileInfo();
            if (rawFileInfo != null) {
                videoFileInfo.getRawFileInfo().setFilePath(rawFileInfo.getFilePath());
                videoFileInfo.getRawFileInfo().setTranslateSize(rawFileInfo.getTranslateSize());
                videoFileInfo.getRawFileInfo().setFileState(rawFileInfo.getFileState());
            }
        }
    }

    /**
     * 更新消息文件信息
     *
     * @param position
     * @param fileInfo
     */
    private void updateFileInfo(int position, FileInfo fileInfo) {
        if (position < 0 || position > mMessageList.size()) {
            return;
        }
        mMessageList.get(position).setFileInfo(fileInfo);
        if (getVu().getViewPager().getCurrentItem() == position) {
            getVu().fileDownloadFinish(position);
        }
    }

    /**
     * 文件下载完成接收事件
     *
     * @param event
     */
    @Subscribe
    public void onReceiveFileFinished(IMProxyFileEvent.ReceiveFileFinishedEvent event) {

        LogUtil.d("EVENT: ReceiveFileFinishedEvent.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Receive file completed, position:" + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file completed, but fileInfo is null.");
            return;
        }

        FileInfo eventFile = event.getFileInfo();
        if (eventFile == null) {
            LogUtil.e("ERROR: Receive file completed, but event fileInfo is null.");
            return;
        }
        mMessageList.get(position).getFileInfo().setFileState(ConstDef.DONE);
        updateFileInfo(fileInfo, eventFile);
        updateFileInfo(position, fileInfo);

        IMProxyFileEvent.HistoryFileEvent historyFileEvent = new IMProxyFileEvent.HistoryFileEvent();
        BusProvider.getInstance().post(historyFileEvent);
    }

    /**
     * 文件进度更新事件
     *
     * @param event
     */
    @Subscribe
    public void onReceiveFileProgressUpdate(IMProxyFileEvent.ReceiveFileProgressUpdateEvent event) {
        LogUtil.d("EVENT: onReceiveFileProgressUpdate.");

//        int position = mathMessage(event.getAttachedMsgId());
//        if (position < 0) {
//            LogUtil.e("ERROR: Receive file completed, position:" + position);
//            return;
//        }
//
//        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
//        if (fileInfo == null) {
//            LogUtil.e("ERROR: Receive file progress update, but fileInfo is null.");
//            return;
//        }
//        FileInfo eventFile = event.getFileInfo();
//        if (eventFile == null) {
//            LogUtil.e("ERROR: Receive file progress update, but event fileInfo is null.");
//            return;
//        }
//
//        updateFileInfo(fileInfo, eventFile);
//        updateFileInfo(position, fileInfo);
    }

    /**
     * 文件下载暂停事件
     *
     * @param event
     */
    @Subscribe
    public void onReceiveFilePaused(IMProxyFileEvent.ReceiveFilePausedEvent event) {
    }

    /**
     * 文件下载失败事件
     *
     * @param event
     */
    @Subscribe
    public void onReceiveFileFailed(IMProxyFileEvent.ReceiveFileFailedEvent event) {
        LogUtil.d("EVENT: ReceiveFileFailedEvent.");

        int position = mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Receive file failed, position:" + position);
            return;
        }

        FileInfo fileInfo = mMessageList.get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file failed, but fileInfo is null.");
            return;
        }
        FileInfo eventFile = event.getFileInfo();
        if (eventFile == null) {
            LogUtil.e("ERROR: Receive file failed, but event fileInfo is null.");
            return;
        }
        mMessageList.get(position).getFileInfo().setFileState(ConstDef.FAIL);
        updateFileInfo(fileInfo, eventFile);
        updateFileInfo(position, fileInfo);
    }


    @Override
    public void setCurMsgId(long msgId) {
        this.mMsgId = msgId;
    }

    @Override
    public void updateUI() {
        getVu().setAdapter(mAdapter);
    }

    @Override
    public void showEmptyView() {
        getVu().showEmptyView();
    }

    @Override
    public void selectPage(int index, TalkMessageBean bean) {
        if (index > 0) {
            getVu().getViewPager().setCurrentItem(index);
        } else if (index == 0) {
            getVu().getViewPager().setCurrentItem(index);
        }
        deleteItem(bean);
    }

    public void deleteItem(final TalkMessageBean messageBean) {
        if (messageBean != null) {
            final long id = messageBean.getId();
            List<Long> ids = new ArrayList<>();
            ids.add(id);
            proxyRepository.get().deleteMessages(ids, new Subscriber<Integer>() {
                @Override
                public void onCompleted() {
                    LogUtil.d("Delete message successfully. " + id);
                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.e("delete message error, " + e.getMessage());
                }

                @Override
                public void onNext(Integer integer) {
                    if (integer == 0) {
                        onDeleteMessage(messageBean);
                    }
                }
            });
        }
    }

    private int onDeleteMessage(TalkMessageBean talkMessageBean) {
        try {
            //构建事件对象
            IMProxyMessageEvent.DeleteMessageEvent messageEvent = new IMProxyMessageEvent.DeleteMessageEvent();
            messageEvent.setMsgAccount(talkMessageBean.getFrom());
            messageEvent.setTalkMessageBean(talkMessageBean);

            //发送事件
            BusProvider.getInstance().post(messageEvent);
            //返回正确处理结果
            return ConstDef.CALLBACK_HANDLED;
        } catch (Exception ex) {
            //处理异常信息
            return ConstDef.CALLBACK_NOT_HANDLED;
        }
    }
}
