package com.xdja.im.uikit.ui.fragment;

import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.View;
import android.widget.Toast;

import com.bumptech.glide.Glide;
import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.account.AccountDesc;
import com.xdja.im.core.model.event.IMContactEvent;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.core.model.event.IMProxyFileEvent;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.message.MeetInfo;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ChatListCallBack;
import com.xdja.im.lib.action.bean.OptionBean;
import com.xdja.im.lib.album.ImagePickerManager;
import com.xdja.im.lib.audio.player.AudioListener;
import com.xdja.im.lib.audio.player.AudioListenerManager;
import com.xdja.im.lib.camera.CameraManager;
import com.xdja.im.lib.file.FileExplorerManager;
import com.xdja.im.lib.video.VideoRecorderManager;
import com.xdja.im.uikit.IntentParam;
import com.xdja.im.uikit.action.media.MediaManger;
import com.xdja.im.uikit.action.media.ReceiverModeManger;
import com.xdja.im.uikit.base.ImBaseFragment;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.notification.NotificationUtil;
import com.xdja.im.uikit.ui.adapter.ChatListAdapter;
import com.xdja.im.uikit.ui.presenter.ChatListPresenter;
import com.xdja.im.uikit.ui.view.ViewChatList;
import com.xdja.im.uikit.ui.view.vu.ChatListVu;
import com.xdja.im.uikit.utils.ImageCache;
import com.xdja.im.uikit.utils.audio.AudioPlayerManagerProxy;
import com.xdja.im.uikit.utils.file.TranslateInfoUtil;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.uikit.presenter.fragment
 * @Author: xdjaxa
 * @Date: 2017-04-10 17:15
 * @Version V1.0
 * @Comment: 聊天详情界面列表
 */
public class ChatListFragment extends ImBaseFragment<ChatListPresenter, ChatListVu>
        implements ChatListPresenter, AudioListener {

    @Inject
    Lazy<ChatListCallBack> chatListCallBack;

    @Override
    protected Class<? extends ChatListVu> getVuClass() {
        return ViewChatList.class;
    }

    @NonNull
    @Override
    protected ChatListPresenter getPresenter() {
        return this;
    }

    @Override
    public String getTalkFlag() {
        return chatListCallBack.get().getTalkId() + "_" + chatListCallBack.get().getTalkType();
    }

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        if (mComponent == null) {
            finish();
            return;
        }
        //注册回调
        mComponent.inject(this);
        chatListCallBack.get().initChatPram(getArguments(), getVu(), this);
        if (TextUtils.isEmpty(chatListCallBack.get().getTalkId())) {
            finish();
            return;
        }
        BusProvider.getInstance().register(this);
        NotificationUtil.getInstance().setCurTalkAccount(chatListCallBack.get().getTalkId());
        AudioListenerManager.getInstance().addListener(this);
        //听筒相关
        new ReceiverModeManger().init(getActivity());
        //初始化
        initView();
    }

    @Override
    public void onStart() {
        super.onStart();
        chatListCallBack.get().requestPermissions(getActivity());
    }

    /**
     * 初始化
     */
    private void initView() {
        chatListCallBack.get().clearAdapter();
        chatListCallBack.get().loadMessageList(0);
        //清除未读消息数
        chatListCallBack.get().clearUnReadMsgCount();
        //获取草稿信息
        chatListCallBack.get().getDraftContent();
        //加载消息Option数据
        chatListCallBack.get().startLoadOptions(getActivity());
        initToolBar();
    }

    /**
     * ToolBar初始化
     */
    private void initToolBar() {
        getVu().setTalkType(chatListCallBack.get().getTalkType());
        getVu().setChatMeet(getTalkFlag());
        getVu().setToolBarVisibility(View.GONE);
    }

    /**
     * 构建Adapter
     *
     * @return
     */
    private ChatListAdapter buildAdapter() {
        return chatListCallBack.get().buildAdapter();
    }

    /**
     * 加载当前会话中的所有聊天消息
     *
     * @param msgId 消息id，如果为0，则表示拉取最新消息
     */
    private void loadMessageList(final long msgId) {
        chatListCallBack.get().loadMessageList(msgId);
    }

    @Override
    public void downRefreshList() {
        chatListCallBack.get().downRefreshList();
    }

    /**
     * 退出聊天界面，通知更新会话内容
     */
    private void notifyUpdateLastSession() {
        chatListCallBack.get().notifyUpdateLastSession();
    }

    /**
     * 清除未读消息标识
     */
    private void clearUnReadMsgCount() {
        chatListCallBack.get().clearUnReadMsgCount();
    }

    @Override
    public int getMessageSize() {
        return chatListCallBack.get().getListTalkMessageBean().size();
    }

    @Override
    public boolean sendTextMessage(String message) {
        chatListCallBack.get().sendTextMessage(message);
        return true;
    }

    @Override
    public void sendFileMessage(List<FileInfo> fileList) {
        chatListCallBack.get().sendFileMessage(fileList);
    }

    @Override
    public void sendVoiceMessage(String path, int seconds) {
        chatListCallBack.get().sendVoiceMessage(path, seconds);
    }

    //***********************************************************************
    //*            以下为chatAction启动第三方界面，以及返回结果
    //***********************************************************************
    @Override
    public void startToOptionView(int position, OptionBean optionBean) {
        //chatListCallBack.get().startToOptionView(getActivity(), position, optionBean);
        LogUtil.d("startToOptionView:" + optionBean.toString());
        Intent intent = chatListCallBack.get().buildIntent(getActivity(), optionBean);
        try {
            //捕获异常，防止
            if (optionBean.getRequestCode() > 0) {
                startActivityForResult(intent, optionBean.getRequestCode());
            } else {
                startActivity(intent);
            }
            //恢复面板状态
            getVu().restoreActionState();
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @Override
    public void startToAlbum() {
        getVu().restoreActionState();
        //打开相册
        ImagePickerManager.getInstance().openAlbum(this, Constant.RESULT_CODE_ALBUM);
    }

    @Override
    public void startToCamera() {
        getVu().restoreActionState();
        CameraManager.getInstance().startTakePhoto(this, Constant.RESULT_CODE_CAMERA);
    }

    @Override
    public void startToFileExplorer() {
        getVu().restoreActionState();
        FileExplorerManager.getInstance().openFileExplorer(this, Constant.RESULT_CODE_FILE);
    }

    @Override
    public void startToVideo() {
        getVu().restoreActionState();
        VideoRecorderManager.getInstance().startVideoRecorder(this, Constant.RESULT_CODE_VIDEO);
    }

    @Override
    public void startSetting() {
        getVu().restoreActionState();
        try {
            boolean result = IMUiKitProxyImpl.getInstance().openChatSettingCallback(chatListCallBack.get().getTalkId(), chatListCallBack.get().getTalkType());
            if (!result) {
                IntentParam.startSettingActivity(getActivity(), chatListCallBack.get().getTalkId(), chatListCallBack.get().getTalkType());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onLongClickAtUser(String groupId, String account, String name) {
        chatListCallBack.get().onLongClickAtUser(ConstDef.CLICK_IMAGE,groupId, account,name);
    }

    @Override
    public void startLocation(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startLocation(chatListCallBack.get().getTalkId());
    }

    @Override
    public void startCallback(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startCallback(chatListCallBack.get().getTalkId());
    }

    @Override
    public void startVoice(int position, OptionBean optionBean) {
        getVu().restoreActionState();
        IMUiKitProxyImpl.getInstance().startVoice(chatListCallBack.get().getTalkId(), chatListCallBack.get().getTalkType());
    }

    @Override
    public void listScroll(int firstVisibleItem, int endVisibleItem) {
        for (int i = 0; i < chatListCallBack.get().getListTalkMessageBean().size(); i++) {
            if (chatListCallBack.get().getListTalkMessageBean().get(i).getMessageType() == ConstDef.MSG_TYPE_PHOTO) {
                if (i < firstVisibleItem || i > endVisibleItem) {
                    chatListCallBack.get().getListTalkMessageBean().get(i).setGifPlayed(true);
                } else {
                    chatListCallBack.get().getListTalkMessageBean().get(i).setGifPlayed(false);
                }
                buildAdapter().updateItem(i, chatListCallBack.get().getListTalkMessageBean().get(i));
            }
        }
    }

    @Override
    public void addMeeting() {
        IMUiKitProxyImpl.getInstance().addMeetting(chatListCallBack.get().getAccount(), chatListCallBack.get().getTalkId());
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        LogUtil.d("onActivityResult requestCode:" + requestCode);
        if (data == null) {
            LogUtil.w("ERROR: on bad activity result, data is null.");
            return;
        }

        switch (resultCode) {
            case Constant.RESULT_CODE_EMOTION:
                break;
            case Constant.RESULT_CODE_ALBUM:
                handleAlbumResult(data);
                break;
            case Constant.RESULT_CODE_CAMERA:
                handleCameraResult(data);
                break;
            case Constant.RESULT_CODE_AUDIO:
                break;
            case Constant.RESULT_CODE_VIDEO:
                handleVideoRecorderResult(data);
                break;
            case Constant.RESULT_CODE_FILE:
                handleFileResult(data);
                break;
            case Constant.RESULT_CODE_LOCATION:
                break;
            case Constant.RESULT_CODE_WEB:
                break;
            case Constant.RESULT_CODE_FILE_DOWNLOAD:
                handleFileLoadResult(data);
                break;
        }
    }

    /**
     * 相册返回值处理
     *
     * @param data
     */
    private void handleAlbumResult(Intent data) {
        chatListCallBack.get().handleAlbumResult(data);
    }

    /**
     * 拍照返回值处理
     *
     * @param data
     */
    private void handleCameraResult(Intent data) {
        chatListCallBack.get().handleCameraResult(data);
    }

    /**
     * 视频录制返回值处理
     *
     * @param data
     */
    private void handleVideoRecorderResult(Intent data) {
        chatListCallBack.get().handleVideoRecorderResult(data);
    }

    /**
     * 文件选择返回值处理
     *
     * @param data
     */
    private void handleFileResult(Intent data) {
        chatListCallBack.get().handleFileResult(data);
    }

    /**
     * 文件返回值结果处理
     *
     * @param data
     */
    private void handleFileLoadResult(Intent data) {
        chatListCallBack.get().handleFileLoadResult(data);
    }
    //***********************************************************************
    //*            启动第三方界面结束
    //***********************************************************************

    @Override
    public void onActivityFocusChanged(boolean hasFocus) {
        super.onActivityFocusChanged(hasFocus);
        LogUtil.d("Window focus changed " + hasFocus);
        if (!hasFocus) {
            AudioPlayerManagerProxy.stopPlay();
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        chatListCallBack.get().setAtType();
        IMProxySessionEvent.ClearUnReadMsgEvent event =
                new IMProxySessionEvent.ClearUnReadMsgEvent();
        event.setTalkId(getTalkFlag());
        BusProvider.getInstance().post(event);
        MediaManger.getInstance().restoreAudioMode();
        LogUtil.d("onPause.");
        //通知更新最有一条会话消息
        notifyUpdateLastSession();
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();

        LogUtil.d("onDestroyView.");

        //清除未读消息
        clearUnReadMsgCount();


        //更新文件传输进度
        chatListCallBack.get().updateFileTranslateSize();
        chatListCallBack.get().clearAdapter();
        AudioPlayerManagerProxy.stopPlay();
        AudioListenerManager.getInstance().clearAll();

        //TranslateInfoUtil.clear();

        //清除缓存信息
        ImageCache.getInstance().clearAllCache();
        Glide.get(getContext()).clearMemory();

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    //*****************************************************************
    //*     以下为OTTO事件处理模块，该模块所有的回调均在主线程中
    //*****************************************************************
    @Subscribe
    public void onReceiveNewMessage(IMProxyMessageEvent.ReceiveNewMessageEvent event) {

        LogUtil.d("EVENT: Receive a new message.");

        //消息内容校验
        List<TalkMessageBean> talkMessageList = event.getTalkMessageList();
        if (talkMessageList == null || talkMessageList.isEmpty()) {
            LogUtil.w("receive new message event, but messageBean is null or empty.");
            return;
        }
        if (chatListCallBack.get().getTalkType() == ConstDef.CHAT_TYPE_P2G) {
            for (TalkMessageBean bean : talkMessageList) {
                LogUtil.d("ChatListActivity", "bean.MsgType:" + bean.getMessageType());
                if (bean.getMessageType() == ConstDef.MSG_TYPE_PRESENTATION) {
                    if (IMUiKitProxyImpl.getInstance().isShowGroupSetting(chatListCallBack.get().getTalkId())) {
                        getVu().setToolBarAddVisibility(View.VISIBLE);
                    } else {
                        getVu().setToolBarAddVisibility(View.INVISIBLE);
                    }
                }
            }
        }
        //发送方账号校验
        String from = event.getMsgAccount();
        if (TextUtils.isEmpty(from) || !TextUtils.equals(from, chatListCallBack.get().getTalkId())) {
            LogUtil.w("receive new message event, but account is empty.");
            return;
        }
        chatListCallBack.get().addNewMessageToList(talkMessageList);
    }

    @Subscribe
    public void onDeleteMessage(IMProxyMessageEvent.DeleteMessageEvent event) {

        LogUtil.d("EVENT: Delete a new message.");

        TalkMessageBean talkMessageBean = event.getTalkMessageBean();

        //如果当时语音正在播放，则停止当前播放语音
        FileInfo fileInfo = talkMessageBean.getFileInfo();
        if (fileInfo != null && fileInfo.getFileType() == ConstDef.FILE_TYPE_VOICE) {
            AudioPlayerManagerProxy.stopPlay();
        }

        chatListCallBack.get().getListTalkMessageBean().remove(talkMessageBean);
        chatListCallBack.get().notifyDataSetChanged();
    }

    @Subscribe
    public void onRefreshMessage(IMProxyMessageEvent.RefreshMessageEvent event) {

        LogUtil.d("EVENT: refresh single message.");

        TalkMessageBean bean = event.getTalkMessageBean();
        if (bean == null) {
            LogUtil.e("refresh single message, but messageBean is null.");
            return;
        }
        chatListCallBack.get().changeMsgState(bean);
    }

    @Subscribe
    public void onRefreshMessageList(IMProxyMessageEvent.RefreshMessageListEvent event) {
        LogUtil.d("EVENT: refresh message list.");

        if (event.isNeedClearInput()) {
            getVu().setInputContent("");
        }
        loadMessageList(0);
    }

    @Subscribe
    public void onRecallMessage(final IMProxyMessageEvent.RecallMessageEvent event) {
        TalkMessageBean bean = event.getTalkMessageBean();
        if (bean == null) {
            LogUtil.e("refresh single message, but messageBean is null.");
            return;
        }

        int position = chatListCallBack.get().mathMessage(bean.getId());
        if (position < 0) {
            LogUtil.w("math message failed." + bean.getId());
            return;
        }

        chatListCallBack.get().getListTalkMessageBean().set(position, bean);
        buildAdapter().updateItem(position, bean);
    }

    @Subscribe
    public void onSendFileComplete(IMProxyFileEvent.SendFileFinishedEvent event) {
        LogUtil.d("EVENT: Send file complete.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Send file completed. position:" + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file completed, but fileInfo is null.");
            return;
        }

        fileInfo.setFileState(ConstDef.DONE);
        chatListCallBack.get().updateFileInfo(position, fileInfo);


        if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VOICE) {
            LogUtil.d("Voice send finished.");
            return;
        }
        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onSendFilePause(IMProxyFileEvent.SendFilePausedEvent event) {

        LogUtil.d("EVENT: Send file pause.");
    }

    @Subscribe
    public void onSendFileProgressUpdate(IMProxyFileEvent.SendFileProgressUpdateEvent event) {

        LogUtil.d("EVENT: Send file progress update.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Send file progress update, position: " + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file progress update, but post fileInfo is null.");
            return;
        }

        //视频和图片，只有在上传原图时，才更新进度
        if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO ||
                fileInfo.getFileType() == ConstDef.FILE_TYPE_VIDEO) {

            int fileProperty = event.getFileProperty();
            if (fileProperty == ConstDef.IS_RAW) {
                int percent = event.getPercent();
                fileInfo.setTranslatePercent(percent);
            }
        }
        FileInfo eventFileInfo = event.getFileInfo();
        if (eventFileInfo != null) {
            fileInfo.setTranslateSize(eventFileInfo.getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.STATE_SENDING);
        chatListCallBack.get().updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onSendFileFailed(IMProxyFileEvent.SendFileFailedEvent event) {
        LogUtil.d("EVENT: Send file failed.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: send file failed, position:" + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file failed , but fileInfo is null.");
            return;
        }
        LogUtil.w("Send failed file info:" + fileInfo.toString());

        fileInfo.setFileState(ConstDef.FAIL);
        chatListCallBack.get().updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onReceiveFileComplete(IMProxyFileEvent.ReceiveFileFinishedEvent event) {
        LogUtil.d("EVENT: Receive file complete.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.e("ERROR: Receive file completed, position:" + position);
            return;
        }


        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file completed, but fileInfo is null.");
            return;
        }

        //更新状态
        fileInfo.setFileState(ConstDef.DONE);
        chatListCallBack.get().updateFileInfo(position, fileInfo);

        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onReceiveFilePause(IMProxyFileEvent.ReceiveFilePausedEvent event) {
        LogUtil.d("EVENT: Receive file pause.");
        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Receive file PAUSE, position: " + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file PAUSE, but post fileInfo is null.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.PAUSE);
        chatListCallBack.get().updateFileInfo(position, fileInfo);

    }

    @Subscribe
    public void onReceiveFileProgressUpdate(IMProxyFileEvent.ReceiveFileProgressUpdateEvent event) {
        LogUtil.d("EVENT: Receive file progress update.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Send file progress update, position: " + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Send file progress update, but post fileInfo is null.");
            return;
        }

        if (fileInfo.getFileType() != ConstDef.FILE_TYPE_NORMAL) {
            LogUtil.d("No need update file progress.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.LOADING);
        chatListCallBack.get().updateFileInfo(position, fileInfo);
    }

    @Subscribe
    public void onReceiveFileFailed(IMProxyFileEvent.ReceiveFileFailedEvent event) {
        LogUtil.d("EVENT: Receive file failed.");

        int position = chatListCallBack.get().mathMessage(event.getAttachedMsgId());
        if (position < 0) {
            LogUtil.w("Error: Receive file Failed, position: " + position);
            return;
        }

        FileInfo fileInfo = chatListCallBack.get().getListTalkMessageBean().get(position).getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: Receive file Failed, but post fileInfo is null.");
            return;
        }

        if (event.getFileInfo() != null) {
            fileInfo.setTranslateSize(event.getFileInfo().getTranslateSize());
        }
        fileInfo.setFileState(ConstDef.FAIL);
        chatListCallBack.get().updateFileInfo(position, fileInfo);
        TranslateInfoUtil.remove(fileInfo.getTalkMessageId());
    }

    @Subscribe
    public void onRefreshGroupEvent(IMContactEvent.RefreshGroupEvent event) {
        LogUtil.d("EVENT: Receive group refresh event.");

        String groupId = event.getAccount();
        //刷新对应的会话列表
        //刷新对应的聊天对象，刷新对应导航栏标题
        chatListCallBack.get().notifyDataSetChanged();

        //chatListCallBack.get().getTalkId()与groupId不同等异常状态
        if (!chatListCallBack.get().getTalkId().equals(groupId)) {
            return;
        }

        if (chatListCallBack.get().getTalkId().equals(groupId)) {
            //获取群信息
            IMUiKitProxyImpl.getInstance().getAccount(groupId, ConstDef.CHAT_TYPE_P2G, new Subscriber<AccountBean>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.d("获取群信息失败");
                }

                @Override
                public void onNext(AccountBean accountBean) {
                    String displayName = accountBean.getDisplayName();
                    AccountDesc accountDesc = accountBean.getAccountDesc();
                    if (accountDesc != null && accountDesc.getMemberNum() > 0) {
                        displayName += " (" + accountDesc.getMemberNum() + ")";
                    }
                    getVu().setTitle(displayName);
                }
            });

        }
    }

    @Subscribe
    public void onDissolutionGroupEvent(IMContactEvent.DissolutionGroupEvent event) {
        LogUtil.d("EVENT: dissolution group event.");

        getActivity().finish();
    }

    @Subscribe
    public void onDataTimeChanged(IMProxyEvent.DataTimeChangedEvent event) {

        LogUtil.d("EVENT: Receive date time changed event.");

        chatListCallBack.get().notifyDataSetChanged();
    }

    @Subscribe
    public void onRefreshSessionList(IMProxySessionEvent.RefreshSessionListEvent event) {

        LogUtil.d("EVENT: onRefreshSessionList");
        chatListCallBack.get().notifyDataSetChanged();
    }

    /**
     * 接收到撤回事件
     * 界面收到撤回事件后进行刷新
     *
     * @param event
     */
    @Subscribe
    public void onRefreshRecallMessage(final IMProxyEvent.RecallMessageEvent event) {
        if (event == null || event.getTalkMessageBean() == null) {
            return;
        }
        TalkMessageBean bean = event.getTalkMessageBean();

        if (chatListCallBack.get().getListTalkMessageBean() != null) {
            for (int i = chatListCallBack.get().getListTalkMessageBean().size() - 1; i >= 0; i--) {
                TalkMessageBean messageBean = chatListCallBack.get().getListTalkMessageBean().get(i);
                if (messageBean != null) {
                    if (messageBean.getId() == bean.getId()) {
                        messageBean.setMessageState(ConstDef.STATE_RECALL_SUCCESS);
                        messageBean.setContent(bean.getContent());
                        messageBean.setMessageType(bean.getMessageType());
                        chatListCallBack.get().notifyDataSetChanged();
                        getVu().setListSelection(getVu().getListView().getFirstVisiblePosition());
                        break;
                    }
                }
            }
            //[S] add by licong ,对于语音播放情况下的处理
            if (AudioPlayerManagerProxy.isPlaying()) {
                AudioPlayerManagerProxy.stopPlay();
            }
            //[E] add by licong ,对于语音播放情况下的处理
        }
    }

    @Subscribe
    public void onRefreshMeetState(IMProxyMessageEvent.ChatMeetRefreshStateHintEvent event) {
        if (event == null || event.getMeetInfo() == null) {
            return;
        }
        MeetInfo meetInfo = event.getMeetInfo();
        if (meetInfo.isShow()) {
            getVu().setMeetState(meetInfo.isShow(),chatListCallBack.get().getTalkType());
        } else {
            getVu().setMeetState(meetInfo.isShow(),chatListCallBack.get().getTalkType());
        }
    }

    /**
     * ------------------------------------语音播放回调-----------------start
     */
    @Override
    public void playingCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }
        long msgId = (long) args[0];
        buildAdapter().startAnimation(filePath, msgId);
    }

    @Override
    public void playingStopCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        buildAdapter().stopAnimation(filePath, msgId);
    }

    @Override
    public void playingErrorCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        buildAdapter().stopAnimation(filePath, msgId);
    }

    @Override
    public void playingCompletionCallback(String filePath, Object... args) {
        if (TextUtils.isEmpty(filePath) &&
                (args == null || args.length == 0)) {
            return;
        }

        long msgId = (long) args[0];
        boolean isNext = (boolean) args[1];
        buildAdapter().playNextAudio(filePath, msgId, isNext);
    }
    /**
     * ------------------------------------语音播放回调-----------------end
     */
}
