package com.xdja.im.uikit.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.RequiresApi;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.widget.ImageView;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.ImageCache;
import com.xdja.im.uikit.utils.gif.GifDecoder;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

@SuppressLint("AppCompatCustomView")
public class GifImageView extends ImageView implements Runnable {

    private static final String TAG = "GifDecoderView";
    private GifDecoder gifDecoder;
    private Bitmap tmpBitmap;
    private final Handler handler = new Handler(Looper.getMainLooper());
    private boolean animating;
    private boolean renderFrame;
    private boolean shouldClear;
    private Thread animationThread;
    private Thread toBytesThread;
    private OnFrameAvailable frameCallback = null;
    private long framesDisplayDuration = -1L;
    private OnAnimationStop animationStopCallback = null;
    private OnAnimationStart animationStartCallback = null;
    private String url;
    private byte[] bytes;

    private final Runnable loadGifImage = new Runnable() {
        @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
        @Override
        public void run() {
            if (setBytes(bytes)) {
                startAnimation();
            } else {
                loadImageFrame(url);
            }
        }
    };
    private final Runnable getGifBytes = new Runnable() {
        @Override
        public void run() {
            if (bytes == null) {
                File file = new File(url);
                try {
                    bytes = getByte(file);
                    handler.post(loadGifImage);
                } catch (Exception e) {
                    LogUtil.e(TAG, "load gif fail");
                }
            }
        }
    };
    private final Runnable updateResults = new Runnable() {
        @Override
        public void run() {
            if (animating && tmpBitmap != null) {
                setImageBitmap(tmpBitmap);
            }
        }
    };

    private final Runnable cleanupRunnable = new Runnable() {
        @Override
        public void run() {
            tmpBitmap = null;
            gifDecoder = null;
            animationThread = null;
            shouldClear = false;
            setImageBitmap(null);
        }
    };

    public GifImageView(final Context context, final AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public GifImageView(final Context context) {
        this(context, null);
    }

    public GifImageView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    public void loadImage(String url) {
        if (checkIsCiphert()) {
            return;
        }
        if (!animating) {
            this.url = url;
            toBytesThread = new Thread(getGifBytes);
            toBytesThread.setName("ToGifBytesThread");
            toBytesThread.start();
        }
    }

    public void loadImageFrame(String url) {
        if (checkIsCiphert()) {
            return;
        }
        if (TextUtils.isEmpty(url)){
            setImageResource(R.drawable.im_uikit_pic_default);
        } else {
            try {
                Bitmap bitmap = ImageCache.getInstance().get(url);
                if (bitmap == null || bitmap.isRecycled()) {
                    bitmap = BitmapFactory.decodeFile(url);
                    ImageCache.getInstance().put(url, bitmap);
                }
                super.setImageBitmap(bitmap);
            } catch (Exception e) {
                setImageResource(R.drawable.im_uikit_pic_default);
            }
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    public boolean setBytes(final byte[] bytes) {
        setBackground(null);
        this.bytes = bytes;
        if (bytes != null) {
            gifDecoder = new GifDecoder();
            try {
                gifDecoder.read(bytes);
                if (animating) {
                    startAnimationThread();
                } else {
                    gotoFrame(0);
                }
            } catch (final Exception e) {
                gifDecoder = null;
                LogUtil.e(TAG, "load gif fail");
                setImageResource(R.drawable.im_uikit_pic_default);
                return false;
            }
        } else {
            animating = false;
            tmpBitmap = null;
        }
        return true;
    }

    public long getFramesDisplayDuration() {
        return framesDisplayDuration;
    }

    /**
     * Sets custom display duration in milliseconds for the all frames. Should be called before {@link
     * #startAnimation()}
     *
     * @param framesDisplayDuration Duration in milliseconds. Default value = -1, this property will
     *                              be ignored and default delay from gif file will be used.
     */
    public void setFramesDisplayDuration(long framesDisplayDuration) {
        this.framesDisplayDuration = framesDisplayDuration;
    }

    public void startAnimation() {
        animating = true;
        startAnimationThread();
    }

    public boolean isAnimating() {
        return animating;
    }

    public void stopAnimation() {
        animating = false;
        if (animationThread != null) {
            animationThread.interrupt();
            animationThread = null;
        }
        if (toBytesThread != null) {
            toBytesThread.interrupt();
            toBytesThread = null;
        }
        if (gifDecoder != null) {
            gifDecoder.clear();
        }
        gifDecoder = null;
        bytes = null;
        shouldClear = false;
    }

    public void gotoFrame(int frame) {
        if (gifDecoder.getCurrentFrameIndex() == frame) return;
        if (gifDecoder.setFrameIndex(frame - 1) && !animating) {
            renderFrame = true;
            startAnimationThread();
        }
    }

    public void resetAnimation() {
        gifDecoder.resetLoopIndex();
        gotoFrame(0);
    }

    public void clear() {
        animating = false;
        renderFrame = false;
        shouldClear = true;
        stopAnimation();
        handler.post(cleanupRunnable);
    }

    /**
     * 把一个文件转化为字节
     *
     * @param file
     * @return byte[]
     * @throws Exception
     */
    public byte[] getByte(File file) throws IOException {
        byte[] bytes = new byte[1024*4];
        InputStream in = new FileInputStream(file.getAbsoluteFile());
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int n;
        while((n = in.read(bytes)) != -1){
            out.write(bytes, 0, n);
        }
        return out.toByteArray();
    }

    private boolean canStart() {
        return (animating || renderFrame) && gifDecoder != null && animationThread == null;
    }

    public int getGifWidth() {
        return gifDecoder.getWidth();
    }

    public int getGifHeight() {
        return gifDecoder.getHeight();
    }

    @Override
    public void run() {
        if (animationStartCallback != null) {
            animationStartCallback.onAnimationStart();
        }
        boolean advance = false;
        //milliseconds spent on frame decode
        long frameDecodeTime = 0;
        int delay;
        long before;
        do {
            if (!animating && !renderFrame) {
                break;
            }
            try {
                advance = gifDecoder.advance();
                before = System.nanoTime();
                tmpBitmap = gifDecoder.getNextFrame();
                if (frameCallback != null) {
                    tmpBitmap = frameCallback.onFrameAvailable(tmpBitmap);
                }
                frameDecodeTime = (System.nanoTime() - before) / 1000000;
                handler.post(updateResults);
            } catch (Exception e) {
                Log.w(TAG, e);
            }

            renderFrame = false;
            if (!animating || !advance) {
                animating = false;
                break;
            }
            try {
                delay = gifDecoder.getNextDelay();
                // Sleep for frame duration minus time already spent on frame decode
                // Actually we need next frame decode duration here,
                // but I use previous frame time to make code more readable
                delay -= frameDecodeTime;
                if (delay > 0) {
                    Thread.sleep(framesDisplayDuration > 0 ? framesDisplayDuration : delay);
                }
            } catch (final Exception e) {
                // suppress exception
            }
        } while (animating);

        if (shouldClear) {
            handler.post(cleanupRunnable);
        }
        animationThread = null;

        if (animationStopCallback != null) {
            animationStopCallback.onAnimationStop();
        }
    }

    public OnFrameAvailable getOnFrameAvailable() {
        return frameCallback;
    }

    public void setOnFrameAvailable(OnFrameAvailable frameProcessor) {
        this.frameCallback = frameProcessor;
    }

    public interface OnFrameAvailable {
        Bitmap onFrameAvailable(Bitmap bitmap);
    }

    public OnAnimationStop getOnAnimationStop() {
        return animationStopCallback;
    }

    public void setOnAnimationStop(OnAnimationStop animationStop) {
        this.animationStopCallback = animationStop;
    }

    public void setOnAnimationStart(OnAnimationStart animationStart) {
        this.animationStartCallback = animationStart;
    }

    public interface OnAnimationStop {
        void onAnimationStop();
    }

    public interface OnAnimationStart {
        void onAnimationStart();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        clear();
    }

    private void startAnimationThread() {
        if (canStart()) {
            animationThread = new Thread(this);
            animationThread.setName("StartAnimationThread");
            animationThread.start();
        }
    }
    private boolean checkIsCiphert(){
//        boolean isCiphert = UniversalUtil.isCiphert(getContext());
//        if(isCiphert){
//            setImageResource(R.drawable.encrypt_image);
//            return true;
//        }
        return false;
    }
    @Override
    protected void onDraw(Canvas canvas) {
        try{
            super.onDraw(canvas);
        }catch (Exception e){
            LogUtil.d("","GifImageView onDraw Exception.");
        }
    }
}
