package com.xdja.im.core.proxy;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.widget.Toast;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.database.helper.MXSettingHelper;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.message.MeetInfo;
import com.xdja.im.core.model.message.TalkCustomBean;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkNotifyBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.param.SessionParam;
import com.xdja.im.core.service.KitService;
import com.xdja.im.core.utils.ToolUtils;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.notification.NotificationUtil;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;

import rx.Subscriber;

/**
 * @Package: com.xdja.im.core.proxy
 * @Author: xdjaxa
 * @Creation: 2017-05-04 17:28
 * @Version V1.0
 * @Description:
 */
public class IMModuleProxyImpl implements ModuleProxy {


    private IMModuleProxyImpl() {
    }

    private static class SingletonInstance {
        private static final IMModuleProxyImpl mInstance = new IMModuleProxyImpl();
    }

    public static IMModuleProxyImpl getInstance() {
        return SingletonInstance.mInstance;
    }

    @Override
    public void initIMProxy() {
        LogUtil.d("Start IM Kit service.");
        Context context = ImUiKit.getInstance().getContext();
        if (context == null) {
            LogUtil.e("Start Kit Service failed, context is null.");
            return;
        }
        try {
            Intent intent = new Intent();
            intent.setClass(context, KitService.class);
            intent.setAction("android.intent.action.RESPOND_VIA_MESSAGE");
            intent.setPackage(context.getPackageName());
            context.startService(intent);
        } catch (Exception e) {
            LogUtil.w("Start service failed.");
        }
    }

    @Override
    public void releaseProxy() {
        Context context = ImUiKit.getInstance().getContext();
        if (context == null) {
            LogUtil.e("Stop Kit Service failed, context is null.");
            return;
        }
        try {
            Intent intent = new Intent();
            intent.setClass(context, KitService.class);
            intent.setAction("android.intent.action.RESPOND_VIA_MESSAGE");
            intent.setPackage(context.getPackageName());
            context.stopService(intent);
        } catch (Exception e) {
            LogUtil.w("Stop service failed.");
        }
    }

    @Override
    public void sendNotifyMessage(String sessionId, int sessionType, TalkNotifyBean bean) {
        if (bean == null) {
            LogUtil.w("ERROR: talk notify bean is null.");
            return;
        }

        boolean isGroup = sessionType == ConstDef.CHAT_TYPE_P2G;
        ImUiKit.getInstance().getComponent().proxyRepository().sendNotifyTextMessage(
                bean.getContent(), bean.getTo(), isGroup, new Subscriber<TalkMessageBean>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Send notify message failed. " + e.getMessage());
                    }

                    @Override
                    public void onNext(TalkMessageBean talkMessageBean) {
                        LogUtil.d("Send notify message successfully.");

                        IMProxyMessageEvent.ReceiveNewMessageEvent event =
                                new IMProxyMessageEvent.ReceiveNewMessageEvent();
                        List<TalkMessageBean> list = new ArrayList<>();
                        list.add(talkMessageBean);
                        event.setTalkMessageList(list);
                        event.setMsgAccount(talkMessageBean.getTo());
                        BusProvider.getInstance().post(event);
                    }
                }
        );
    }

    @Override
    public void sendCustomTextMessage(String sessionId, int sessionType, TalkCustomBean bean) {
        if (bean == null) {
            LogUtil.w("ERROR: Send custom message failed, bean is null.");
            return;
        }
        boolean isGroup = sessionType == ConstDef.CHAT_TYPE_P2G;
        TalkMessageBean talkMessageBean = new TalkMessageBean();
        talkMessageBean.setTo(sessionId);
        talkMessageBean.setGroupMsg(isGroup);
        talkMessageBean.setMessageType(ConstDef.MSG_TYPE_PRESENTATION);
        talkMessageBean.setContent(bean.getContent());

        ImUiKit.getInstance().getComponent().proxyRepository().sendCustomMessage(talkMessageBean, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Send custom message failed, " + e.getMessage());
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {
                LogUtil.d("Send custom message successfully.");
            }
        });
    }

    private void clearSessionParamData(String sessionFlag) {
        ImUiKit.getInstance().getComponent().diskDataStore().deleteSessionParam(sessionFlag,
                new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Clear session data failed." + e.getMessage());
                    }

                    @Override
                    public void onNext(Boolean aBoolean) {
                        IMProxyMessageEvent.RefreshMessageListEvent event =
                                new IMProxyMessageEvent.RefreshMessageListEvent();
                        event.setNeedClearInput(true);
                        BusProvider.getInstance().post(event);
                    }
                });
    }

    private void clearAllSessionParamsData() {
        ImUiKit.getInstance().getComponent().diskDataStore().deleteAllSessionParams(new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Clear all session data failed." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                IMProxySessionEvent.RefreshSessionListEvent event =
                        new IMProxySessionEvent.RefreshSessionListEvent();
                BusProvider.getInstance().post(event);
            }
        });

    }

    @Override
    public void clearSessionData(final String sessionId, int chatType) {
        //清除会话消息
        if (TextUtils.isEmpty(sessionId)) {
            LogUtil.e("Session id is null error.");
            return;
        }
        if (ImUiKit.getInstance().getComponent() == null && ImUiKit.getInstance().getComponent().proxyRepository() == null){
            LogUtil.e("getComponent is null error.");
            return;
        }
        final String sessionFlag = ToolUtils.getSessionTag(sessionId, chatType);
        ImUiKit.getInstance().getComponent().proxyRepository().clearAllSessionData(sessionFlag,
                new Subscriber<Integer>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Clear message list failed." + e.getMessage());
                    }

                    @Override
                    public void onNext(Integer integer) {
                        IMProxySessionEvent.ClearUnReadMsgEvent event =
                                new IMProxySessionEvent.ClearUnReadMsgEvent();
                        event.setTalkId(sessionId);
                        BusProvider.getInstance().post(event);
                        clearSessionParamData(sessionFlag);
                        Toast.makeText(ImUiKit.getInstance().getContext() , "聊天记录清除成功" , Toast.LENGTH_LONG).show();
                    }
                });
    }

    @Override
    public void clearAllSessionData() {
        //清除所有会话消息
        ImUiKit.getInstance().getComponent().proxyRepository().clearAllData(
                new Subscriber<Integer>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Clear all message failed." + e.getMessage());
                    }

                    @Override
                    public void onNext(Integer integer) {
                        IMProxySessionEvent.DeleteSessionEvent event =
                                new IMProxySessionEvent.DeleteSessionEvent();
                        BusProvider.getInstance().post(event);
                        clearAllSessionParamsData();
                    }
                });
    }

    @Override
    public void clearCacheData() {

    }

    @Override
    public void setNoDisturbMode(String sessionId, int sessionType, final boolean isDisturb) {
        if (TextUtils.isEmpty(sessionId)) {
            LogUtil.e("Session id is null error.");
            return;
        }
        final String sessionFlag = ToolUtils.getSessionTag(sessionId, sessionType);
        if (isDisturb) {
            ImUiKit.getInstance().getComponent().cloudDataStore().saveNoDisturb2Cloud(sessionFlag, sessionId, sessionType, new Subscriber<Boolean>() {
                @Override
                public void onCompleted() {
                }

                @Override
                public void onError(Throwable e) {
                }

                @Override
                public void onNext(Boolean aBoolean) {
                    if (aBoolean) {
                        setLocalNoDisturbMode(sessionFlag, isDisturb);
                    }
                }
            });
        } else {
            ImUiKit.getInstance().getComponent().cloudDataStore().deleteNoDisturbAtCloud(sessionFlag, sessionId, sessionType, new Subscriber<Boolean>() {
                @Override
                public void onCompleted() {
                }

                @Override
                public void onError(Throwable e) {
                }

                @Override
                public void onNext(Boolean aBoolean) {
                    if (aBoolean) {
                        setLocalNoDisturbMode(sessionFlag, isDisturb);
                    }
                }
            });
        }
    }

    private void setLocalNoDisturbMode(String sessionFlag, boolean isDisturb) {
        ImUiKit.getInstance().getComponent().diskDataStore().setSessionDisturb(sessionFlag,
                isDisturb, new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Set disturb failed." + e.getMessage());
                    }

                    @Override
                    public void onNext(Boolean aBoolean) {
                        IMProxySessionEvent.RefreshSessionListEvent event =
                                new IMProxySessionEvent.RefreshSessionListEvent();
                        BusProvider.getInstance().post(event);
                    }
                });
    }

    @Override
    public void setSessionTop(String sessionId, int sessionType, final boolean isTop) {
        if (TextUtils.isEmpty(sessionId)) {
            LogUtil.e("Session id is null error.");
            return;
        }
        final String sessionFlag = ToolUtils.getSessionTag(sessionId, sessionType);
        if (isTop) {
            ImUiKit.getInstance()
                    .getComponent()
                    .cloudDataStore()
                    .setCloudSessionTop(sessionFlag, new Subscriber<Boolean>() {
                        @Override
                        public void onCompleted() {
                        }

                        @Override
                        public void onError(Throwable e) {
                            LogUtil.e("ERROR: set session top error." + e.getMessage());
                        }

                        @Override
                        public void onNext(Boolean aBoolean) {
                            setLocalTopState(aBoolean, sessionFlag, isTop);
                        }
                    });
        } else {
            ImUiKit.getInstance()
                    .getComponent()
                    .cloudDataStore()
                    .deleteSettingTopAtCloud(sessionFlag, new Subscriber<Boolean>() {
                        @Override
                        public void onCompleted() {
                        }

                        @Override
                        public void onError(Throwable e) {
                        }

                        @Override
                        public void onNext(Boolean aBoolean) {
                            setLocalTopState(aBoolean, sessionFlag, isTop);
                        }
                    });
        }


    }

    private void setLocalTopState(boolean aBoolean, String sessionFlag, boolean isTop) {
        if (aBoolean) {
            ImUiKit.getInstance().getComponent().diskDataStore().setSessionTop(sessionFlag,
                    isTop, new Subscriber<Boolean>() {
                        @Override
                        public void onCompleted() {
                        }

                        @Override
                        public void onError(Throwable e) {
                            LogUtil.e("ERROR: Set session top failed." + e.getMessage());
                        }

                        @Override
                        public void onNext(Boolean aBoolean) {
                            IMProxySessionEvent.RefreshSessionListEvent event =
                                    new IMProxySessionEvent.RefreshSessionListEvent();
                            BusProvider.getInstance().post(event);
                        }
                    });
        }
    }

    @Override
    public void setReceiveMode(String sessionId, int sessionType, boolean isOpen) {
        if (TextUtils.isEmpty(sessionId)) {
            LogUtil.e("Session id is null error.");
            return;
        }
        String sessionFlag = ToolUtils.getSessionTag(sessionId, sessionType);
        ImUiKit.getInstance().getComponent().diskDataStore().setReceiveMode(sessionFlag,
                isOpen, new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Set receive mode failed." + e.getMessage());
                    }

                    @Override
                    public void onNext(Boolean aBoolean) {
                        LogUtil.d("Set receive mode successfully.");
                        /*IMProxySessionEvent.RefreshSessionListEvent event =
                                new IMProxySessionEvent.RefreshSessionListEvent();
                        BusProvider.getInstance().post(event);*/
                    }
                });
    }

    @Override
    public SessionParam getSessionParam(String sessionId, int sessionType) {
        String sessionFlag = ToolUtils.getSessionTag(sessionId, sessionType);
        return MXSettingHelper.getSessionParam(sessionFlag);
    }

    @Override
    public void notify(final TalkSessionBean sessionBean, final int newMsgCnt, final boolean isDisturb) {
        IMUiKitProxyImpl.getInstance().getAccount(sessionBean.getTalkerAccount(), sessionBean.getTalkType(),
                new Subscriber<AccountBean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: Notify get account failed.");
                    }

                    @Override
                    public void onNext(AccountBean accountBean) {
                        String account = sessionBean.getTalkerAccount();
                        String talkId = sessionBean.getTalkFlag();
                        String imgUrl = accountBean.getAvatarUrl();
                        String userName = accountBean.getDisplayName();
                        int talkType = sessionBean.getTalkType();
                        int msgCount = newMsgCnt;
                        NotificationUtil.getInstance().remindMessage(account,
                                imgUrl,
                                userName,
                                talkId,
                                talkType,
                                msgCount,
                                sessionBean.getLastTime(),
                                isDisturb);
                    }
                });
    }

    @Override
    public void notifyMeet(MeetInfo meetInfo) {
        IMProxyMessageEvent.ChatMeetRefreshStateHintEvent event =  new IMProxyMessageEvent.ChatMeetRefreshStateHintEvent();
        event.setMeetInfo(meetInfo);
        BusProvider.getInstance().post(event);
    }
}
