package com.xdja.im.uikit.ui.fragment;

import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.View;

import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMContactEvent;
import com.xdja.im.core.model.event.IMNotificationEvent;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.message.MeetInfo;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.param.SessionParam;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.core.repository.interf.UserOperateRepository;
import com.xdja.im.core.repository.interf.datasource.CloudDataStore;
import com.xdja.im.core.repository.interf.datasource.DiskDataStore;
import com.xdja.im.core.utils.ToolUtils;
import com.xdja.im.uikit.IntentParam;
import com.xdja.im.uikit.base.ImBaseFragment;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.adapter.SessionListAdapter;
import com.xdja.im.uikit.ui.presenter.SessionListPresenter;
import com.xdja.im.uikit.ui.view.ViewSessionList;
import com.xdja.im.uikit.ui.view.vu.SessionListVu;
import com.xdja.im.uikit.utils.SpUtils;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.uikit.presenter.fragment
 * @Author: xdjaxa
 * @Date: 2017-04-10 17:14
 * @Version V1.0
 * @Comment: 聊天回话列表显示界面
 */
public class SessionListFragment extends ImBaseFragment<SessionListPresenter, SessionListVu>
        implements SessionListPresenter {
    /**
     * IMSDK回调
     */
    @Inject
    Lazy<ProxyRepository> proxyRepository;
    /**
     * 数据存储
     */
    @Inject
    Lazy<DiskDataStore> diskDataStore;
    /**
     * 云数据存储
     */
    @Inject
    Lazy<CloudDataStore> cloudDataStore;
    /**
     * 用户数据处理
     */
    @Inject
    Lazy<UserOperateRepository> userOperateRepositoryLazy;
    /**
     * 适配器
     */
    private SessionListAdapter mAdapter;
    /**
     * 会话数据集合
     */
    private List<TalkSessionBean> mSessionList = new ArrayList<>();

    /**
     * 会话数据类型(默认加载全部数据)
     */
    private int mTalkType = ConstDef.CHAT_TYPE_DEFAULT;
    /**
     * 未读消息数
     */
    private int mUnReadCount;

    /**
     * 是否需要通知未读消息
     */
    private boolean isNotify = true;

    private MeetInfo mMeetInfo;

    @NonNull
    @Override
    protected Class<? extends SessionListVu> getVuClass() {
        return ViewSessionList.class;
    }

    @NonNull
    @Override
    protected SessionListPresenter getPresenter() {
        return this;
    }

    @Override
    protected void preBindView(Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);

        //会话数据类型
        Bundle arguments = getArguments();
        if (arguments != null) {
            mTalkType = arguments.getInt(Constant.TAG_TALKTYPE, ConstDef.CHAT_TYPE_DEFAULT);
        }
    }

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (mComponent == null) {
            return;
        }
        mComponent.inject(this);
        BusProvider.getInstance().register(this);

        mAdapter = new SessionListAdapter(getActivity(), mSessionList, this);
        getVu().setAdapter(mAdapter);

        //加载会话数据
        startLoadSessionData();
    }

    @Override
    public void onResume() {
        super.onResume();
        isNotify = true;
    }

    @Override
    public void onPause() {
        super.onPause();
        isNotify = false;
    }

    /**
     * 计算会话所有未读消息数
     *
     * @param sessionBeanList
     */
    private void calculateReadMessageCount(List<TalkSessionBean> sessionBeanList) {
        int unReadCount = 0;
        for (TalkSessionBean bean : sessionBeanList) {
            unReadCount += bean.getNotReadCount();//未读消息处理
            SessionParam param = bean.getParam();
            if (null != param && param.isDisturb()) {
                unReadCount -= bean.getNotReadCount();
            }
        }
        IMUiKitProxyImpl.getInstance().updateUnReadMessageCount(unReadCount);
    }

    /**
     * 更新未读消息数
     *
     * @param talkSessionBean
     */
    private void updateUnReadMessageCount(TalkSessionBean talkSessionBean) {
        int allUnreadCount = 0;
        if (talkSessionBean == null) {
            return;
        }
        for (TalkSessionBean bean : mSessionList) {
            allUnreadCount += bean.getNotReadCount();//未读消息处理
            SessionParam param = bean.getParam();
            if (null != param && param.isDisturb()) {
                allUnreadCount -= bean.getNotReadCount();
            }
        }
        if (talkSessionBean.getParam() != null && talkSessionBean.getParam().isDisturb()) {
            LogUtil.e("ERROR: talkSessionBean  getParam has  disturb set");
        } else {
            allUnreadCount -= talkSessionBean.getNotReadCount();
        }
        talkSessionBean.setNotReadCount(0);
        if (allUnreadCount < 0) {
            LogUtil.e("ERROR: Update unRead message count " + allUnreadCount);
            return;
        }
        IMUiKitProxyImpl.getInstance().updateUnReadMessageCount(allUnreadCount);
    }

    @Override
    public void onListItemClick(int position) {
        TalkSessionBean talkSessionBean = mSessionList.get(position);

        if (talkSessionBean == null) {
            LogUtil.e("ERROR: obtain session info failed!!");
            return;
        }

        switch (talkSessionBean.getTalkType()) {
            case ConstDef.CHAT_TYPE_P2P:
                break;
            case ConstDef.CHAT_TYPE_P2G:
                break;
            case ConstDef.CHAT_TYPE_CUSTOM:
                break;
            case ConstDef.CHAT_TYPE_DEFAULT:
                break;
            default:
                break;
        }

        IntentParam.startChatActivity(getContext(),
                talkSessionBean.getTalkerAccount(),
                talkSessionBean.getTalkType());

        //发送取消未接来电以及未接视频通话通知栏事件
        BusProvider.getInstance().post(new IMNotificationEvent(talkSessionBean));
        updateUnReadMessageCount(talkSessionBean);
    }

    /**
     * 加载会话列表数据
     */
    private void startLoadSessionData() {
        //加载动画
        getVu().setProgressBarVisibility(View.VISIBLE);
        if (proxyRepository == null || proxyRepository.get() == null) {
            return;
        }

        proxyRepository.get().getSessionListBeans("", 0, mTalkType, new Subscriber<List<TalkSessionBean>>() {
            @Override
            public void onCompleted() {
                //getVu().setProgressBarVisibility(View.GONE);
            }

            @Override
            public void onError(Throwable e) {
                getVu().setProgressBarVisibility(View.GONE);
                LogUtil.e("ERROR: Load session data error. " + e.getMessage());
            }

            @Override
            public void onNext(List<TalkSessionBean> sessionBeanList) {
                LogUtil.d("Load session data finished.");

                //加载会话的配置信息
                loadSessionParams(sessionBeanList);
            }
        });
    }

    /**
     * 加载会话中所有的配置信息
     *
     * @param sessionBeanList
     */
    private void loadSessionParams(final List<TalkSessionBean> sessionBeanList) {

        userOperateRepositoryLazy.get().getAllSessionParams(new Subscriber<List<SessionParam>>() {
            @Override
            public void onCompleted() {
                getVu().setProgressBarVisibility(View.GONE);
            }
            @Override
            public void onError(Throwable e) {
                //剔除重复的会话
                updateSessionList(sessionBeanList);
                getVu().setProgressBarVisibility(View.GONE);
            }
            @Override
            public void onNext(List<SessionParam> sessionParams) {
                if (sessionParams == null || sessionParams.isEmpty()) {
                    //剔除重复的会话
                    updateSessionList(sessionBeanList);
                    return;
                }

                for (int i = 0; i < sessionParams.size(); i++) {
                    SessionParam param = sessionParams.get(i);
                    addParamToSessionList(param, sessionBeanList);
                }
                Collections.sort(sessionBeanList);
                updateSessionList(sessionBeanList);
            }
        });
    }

    /**
     * 添加配置信息
     *
     * @param param
     * @param sessionBeanList
     */
    private void addParamToSessionList(SessionParam param, List<TalkSessionBean> sessionBeanList) {
        TalkSessionBean tempBean = new TalkSessionBean();
        tempBean.setTalkFlag(param.getSessionFlag());
        int index = sessionBeanList.indexOf(tempBean);
        if (index >= 0) {
            //更新草稿信息
            TalkSessionBean talkSessionBean = sessionBeanList.get(index);
            talkSessionBean.setParam(param);

            if (TextUtils.isEmpty(talkSessionBean.getLastMsgAccount()) &&
                    talkSessionBean.getLastTime() > 0) {
                if (param.getDraftTime() > talkSessionBean.getLastTime()) {
                    talkSessionBean.setLastTime(param.getDraftTime());
                }
            }
            if (!TextUtils.isEmpty(param.getDraftContent()) && param.getDraftTime() > 0) {
                talkSessionBean.setLastTime(param.getDraftTime());
            }
        } else {
            //会话不存在，则生成新会话
            if (param.getDraftTime() > 0) {
                TalkSessionBean talkSessionBean = new TalkSessionBean();
                talkSessionBean.setTalkFlag(param.getSessionFlag());
                talkSessionBean.setLastTime(param.getDraftTime());
                talkSessionBean.setParam(param);
                sessionBeanList.add(talkSessionBean);
            }
        }
    }

    /**
     * 创建新会话
     *
     * @param sessionBean
     */
    private void createNewSession(final TalkSessionBean sessionBean) {
        diskDataStore.get().loadSessionParam(sessionBean.getTalkFlag(), new Subscriber<SessionParam>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Load session param failed." + sessionBean.getTalkFlag());
                mSessionList.add(sessionBean);
                Collections.sort(mSessionList);
                mAdapter.notifyDataSetChanged();

                //通知更新未读消息数
                calculateReadMessageCount(mSessionList);
            }

            @Override
            public void onNext(SessionParam sessionParam) {
                if (sessionParam != null) {
                    LogUtil.d("SessionParam:" + sessionParam.toString());
                    sessionBean.setParam(sessionParam);
                }
                mSessionList.add(sessionBean);
                removeExists(mSessionList);
                Collections.sort(mSessionList);
                mAdapter.notifyDataSetChanged();

                //通知更新未读消息数
                calculateReadMessageCount(mSessionList);
            }
        });
    }

    /**
     * 更新会话列表
     *
     * @param sessionBeanList
     */
    private void updateSessionList(List<TalkSessionBean> sessionBeanList) {
        mSessionList.clear();
        //剔除重复的会话
        removeExists(sessionBeanList);
        mSessionList.addAll(sessionBeanList);
        mAdapter.notifyDataSetChanged();

        //通知更新未读消息数
        calculateReadMessageCount(mSessionList);
    }

    /**
     * 移除重复消息
     *
     * @param beans
     */
    private void removeExists(List<TalkSessionBean> beans) {
        Iterator<TalkSessionBean> iterator = beans.iterator();
        for (; iterator.hasNext(); ) {
            TalkSessionBean bean = iterator.next();

            int index = beans.indexOf(bean);
            if (index > -1 && bean != beans.get(index)) {
                iterator.remove();
            }
        }
    }

    @Override
    public void deleteSession(final int position) {
        final TalkSessionBean sessionBean = (TalkSessionBean) mAdapter.getItem(position);
        if (sessionBean != null) {
            getVu().setProgressBarVisibility(View.VISIBLE);

            proxyRepository.get().deleteSession(sessionBean.getTalkFlag(), new Subscriber<Integer>() {
                @Override
                public void onCompleted() {
                    LogUtil.d("Delete session successfully.");
                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.e("Delete session error." + e.getMessage());
                    getVu().setProgressBarVisibility(View.GONE);
                }

                @Override
                public void onNext(Integer integer) {

                    if (integer == 0) {
                        getVu().setProgressBarVisibility(View.GONE);
                        mSessionList.remove(sessionBean);
                        mAdapter.notifyDataSetChanged();

                        //会话删除成功后，需要同步更新SessionCache和数据库
                        deleteSessionParam(sessionBean.getTalkFlag());

                        //更新通知栏信息
                        //该数据已经从mSessionList中移除
                        sessionBean.setNotReadCount(0);
                        updateUnReadMessageCount(sessionBean);
                    }
                }
            });
        }
    }

    @Override
    public void deleteSession(String sessionFlag) {
        if (TextUtils.isEmpty(sessionFlag)) {
            LogUtil.e("ERROR:Delete session, but sessionFlag is im_uikit_empty.");
            return;
        }
        final TalkSessionBean sessionBean = new TalkSessionBean();
        sessionBean.setTalkFlag(sessionFlag);

        proxyRepository.get().deleteSession(sessionBean.getTalkFlag(), new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
                LogUtil.d("Delete session successfully.");
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("Delete session error." + e.getMessage());
                getVu().setProgressBarVisibility(View.GONE);
            }

            @Override
            public void onNext(Integer integer) {

                if (integer == 0) {
                    getVu().setProgressBarVisibility(View.GONE);
                    mSessionList.remove(sessionBean);
                    mAdapter.notifyDataSetChanged();

                    //会话删除成功后，需要同步更新SessionCache和数据库
                    deleteSessionParam(sessionBean.getTalkFlag());
                }
            }
        });
    }

    @Override
    public void deleteSessionParam(final String sessionFlag) {
        diskDataStore.get().deleteSessionParam(sessionFlag, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: Delete session param failed.");
            }

            @Override
            public void onNext(Boolean aBoolean) {
                LogUtil.d("Delete session param successfully. sessionFlag:" + sessionFlag);
                IMProxyMessageEvent.RefreshMessageListEvent event =
                        new IMProxyMessageEvent.RefreshMessageListEvent();
                event.setNeedClearInput(true);
                BusProvider.getInstance().post(event);
            }
        });
    }

    @Override
    public void setSessionTop(final int position) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }
        cloudDataStore.get()
                .setCloudSessionTop(sessionBean.getTalkFlag(), new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR: set session top error." + e.getMessage());
                    }

                    @Override
                    public void onNext(Boolean aBoolean) {
                        if (aBoolean)
                            setLocalSessionTop(position);
                    }
                });
    }

    @Override
    public void deleteSessionTop(final int position) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }
        cloudDataStore.get().deleteSettingTopAtCloud(sessionBean.getTalkFlag(), new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: set session top error." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                if (aBoolean)
                    setLocalSessionTop(position);
            }
        });
    }

    private void setLocalSessionTop(final int position) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }
        SessionParam param = sessionBean.getParam();
        if (param == null) {
            param = new SessionParam();
            sessionBean.setParam(param);
        }

        //当前设置的置顶模式
        final boolean isTop = !param.isTop();
        param.setShowOnTopTime(System.currentTimeMillis());

        diskDataStore.get().setSessionTop(sessionBean.getTalkFlag(), isTop, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: set session top error." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                //重新进行排序即可，因为TalkSessionBean已经重写了Compare方法，重新排序即可
                mSessionList.get(position).getParam().setTop(isTop);
                Collections.sort(mSessionList);
                mAdapter.notifyDataSetChanged();
            }
        });
    }

    @Override
    public boolean isSessionTop(int position) {
        TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return false;
        }

        if (sessionBean.getParam() == null) {
            return false;
        }
        return sessionBean.getParam().isTop();
    }

    @Override
    public void setDisturb(final int position) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }
        SessionParam param = sessionBean.getParam();
        if (param == null) {
            param = new SessionParam();
            sessionBean.setParam(param);
        }
        //当前设置的置顶模式
        final boolean isDisturb = !param.isDisturb();
        cloudDataStore.get().saveNoDisturb2Cloud(sessionBean.getTalkFlag(), sessionBean.getTalkerAccount(), sessionBean.getTalkType(), new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(Boolean aBoolean) {
                if (aBoolean)
                    setLocalDisturb(position, isDisturb);
            }
        });
    }

    @Override
    public void deleteDisturb(final int position) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }
        SessionParam param = sessionBean.getParam();
        if (param == null) {
            param = new SessionParam();
            sessionBean.setParam(param);
        }
        //当前设置的置顶模式
        final boolean isDisturb = !param.isDisturb();
        cloudDataStore.get().deleteNoDisturbAtCloud(sessionBean.getTalkFlag(), sessionBean.getTalkerAccount(), sessionBean.getTalkType(), new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(Boolean aBoolean) {
                if (aBoolean)
                    setLocalDisturb(position, isDisturb);
            }
        });
    }

    public void setLocalDisturb(final int position, final boolean isDisturb) {
        final TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null || TextUtils.isEmpty(sessionBean.getTalkFlag())) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return;
        }

        SessionParam param = sessionBean.getParam();
        if (param == null) {
            param = new SessionParam();
            sessionBean.setParam(param);
        }
        diskDataStore.get().setSessionDisturb(sessionBean.getTalkFlag(), isDisturb, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: set session disturb error." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                mSessionList.get(position).getParam().setDisturb(isDisturb);
                Collections.sort(mSessionList);
                mAdapter.notifyDataSetChanged();
                calculateReadMessageCount(mSessionList);
            }
        });
    }

    @Override
    public boolean isDisturb(int position) {
        TalkSessionBean sessionBean = mSessionList.get(position);
        if (sessionBean == null) {
            LogUtil.e("ERROR: set session top, but session info is null.");
            return false;
        }

        if (sessionBean.getParam() == null) {
            return false;
        }
        return sessionBean.getParam().isDisturb();
    }

    @Override
    public MeetInfo ifShowMeetTip() {
        return mMeetInfo;
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 新消息到来，更新会话所有内容
     *
     * @param eventBean
     * @param index
     * @return
     */
    private TalkSessionBean addNewSession(TalkSessionBean eventBean, int index) {
        SessionParam param = mSessionList.get(index).getParam();
        //更新配置信息
        if (param == null) {
            param = new SessionParam();
            param.setDraftContent("");
            param.setDraftTime(0L);
        }

        //需要清除之前的草稿信息
        eventBean.setParam(param);
        return eventBean;
    }

    /**
     * 聊天界面返回，更新会话草稿和时间
     *
     * @param eventBean
     * @param index
     * @return
     */
    private TalkSessionBean updateSession(TalkSessionBean eventBean, int index) {
        TalkSessionBean bean = mSessionList.get(index);
        SessionParam param = bean.getParam();
        if (param == null) {
            param = new SessionParam();
        }

        if (eventBean.getParam() != null) {
            //草稿信息有更新
            param.setDraftContent(eventBean.getParam().getDraftContent());
            param.setDraftTime(eventBean.getParam().getDraftTime());
            //草稿清空不更新时间
            if (0 != eventBean.getParam().getDraftTime()) {
                bean.setLastTime(eventBean.getParam().getDraftTime());
            }
        } else {
            //消息内容有更新
            bean.setContent(eventBean.getContent());
            bean.setLastTime(eventBean.getLastTime());
            bean.setLastMsg(eventBean.getLastMsg());
            param.setDraftContent("");
            param.setDraftTime(0L);
        }
        bean.setParam(param);
        //退出聊天界面，将所有消息置为已读
        bean.setNotReadCount(0);
        return bean;
    }

    //*****************************************************************
    //*     以下为OTTO事件处理模块，该模块所有的回调均在主线程中
    //*****************************************************************
    @Subscribe
    public void onCreateNewSession(IMProxySessionEvent.CreateNewSessionEvent event) {

        LogUtil.d("EVENT: onCreateNewSession.");

        TalkSessionBean talkSessionBean = event.getSessionBean();
        if (talkSessionBean == null) {
            LogUtil.e("ERROR: create new session failed.");
            return;
        }

        if (mSessionList.contains(talkSessionBean)) {
            LogUtil.w("This session is exits. do not need to create.");
            return;
        }

        createNewSession(talkSessionBean);
    }

    @Subscribe
    public void onDeleteSession(IMProxySessionEvent.DeleteSessionEvent event) {
        LogUtil.d("EVENT: onDeleteSession.");

        TalkSessionBean talkSessionBean = event.getSessionBean();
        if (talkSessionBean == null) {
            LogUtil.e("ERROR: delete session failed.");
            return;
        }
        String sessionFlag = ToolUtils.getSessionTag(talkSessionBean.getTalkerAccount(),
                talkSessionBean.getTalkType());
        deleteSessionParam(sessionFlag);
    }

    @Subscribe
    public void onRefreshSession(IMProxySessionEvent.RefreshSessionEvent event) {
        LogUtil.d("EVENT: onRefreshSession");

        TalkSessionBean eventBean = event.getSessionBean();
        if (eventBean == null) {
            LogUtil.w("ERROR: refresh session, but bean is null.");
            return;
        }

        int index = mSessionList.indexOf(eventBean);
        if (index < 0 || index >= mSessionList.size()) {
            LogUtil.w("ERROR: Index outOfBounds error.");
            startLoadSessionData();
            return;
        }

        TalkSessionBean bean;
        if (event.isUpdateData()) {
            isNotify = true;
            bean = updateSession(eventBean, index);
        } else {
            bean = addNewSession(eventBean, index);
        }
        //列表排序
        mSessionList.set(index, bean);
        Collections.sort(mSessionList);
        mAdapter.notifyDataSetChanged();

        LogUtil.d("onRefreshSession is need notify:" + isNotify);
        calculateReadMessageCount(mSessionList);
    }

    @Subscribe
    public void onRefreshSessionList(IMProxySessionEvent.RefreshSessionListEvent event) {

        LogUtil.d("EVENT: onRefreshSessionList");

        startLoadSessionData();
    }

    @Subscribe
    public void onDissolutionGroupEvent(IMContactEvent.DissolutionGroupEvent event) {
        LogUtil.d("EVENT: dissolution group event.");

        LogUtil.d("Dissolution groupId " + event.getAccount());

        //群组解散或者退出群组，删除当前会话
        String sessionFlag = ToolUtils.getSessionTag(event.getAccount(), ConstDef.CHAT_TYPE_P2G);
        deleteSession(sessionFlag);
    }

    @Subscribe
    public void onDataTimeChanged(IMProxyEvent.DataTimeChangedEvent event) {
        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }

    @Subscribe
    public void onDeleteAllMessage(IMProxySessionEvent.ClearUnReadMsgEvent event) {
        LogUtil.d("EVENT: onDeleteAllMessage");
        for (TalkSessionBean sessionBean : mSessionList) {
            if (sessionBean.getTalkFlag().equals(event.getTalkId())) {
                updateUnReadMessageCount(sessionBean);
            }
        }
    }

    @Subscribe
    public void NetworkStateChange(IMProxyEvent.NetworkStateEvent event) {
        LogUtil.d("EVENT: NetworkStateChange");
        getVu().changeViewSate(event.getState());
    }


    @Subscribe
    public void onRefreshMeetState(IMProxyMessageEvent.ChatMeetRefreshStateHintEvent event) {
        if (event == null || event.getMeetInfo() == null) {
            return;
        }
        MeetInfo meetInfo = event.getMeetInfo();
        if (meetInfo != null){
            mMeetInfo = meetInfo;
            if (meetInfo.isShow()){
                SpUtils.getInstance().saveBoolean(SpUtils.meetChat,true);
            } else {
                SpUtils.getInstance().saveBoolean(SpUtils.meetChat,false);
            }
            SpUtils.getInstance().saveString(SpUtils.meetTalkFlag,meetInfo.getTaklFlag());
        }
        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }
}
