package com.xdja.im.common.utils;

import android.text.TextUtils;

import java.io.File;
import java.io.FileInputStream;
import java.text.DecimalFormat;

/**
 * @Package: com.xdja.im.common.utils
 * @Author: xdjaxa
 * @Creation: 2017-05-02 15:03
 * @Version V1.0
 * @Description:文件大小计算工具类
 */
public class FileSizeUtils {

    public static final int SIZETYPE_B = 1;//获取文件大小单位为B的double值
    public static final int SIZETYPE_KB = 2;//获取文件大小单位为KB的double值
    public static final int SIZETYPE_MB = 3;//获取文件大小单位为MB的double值
    public static final int SIZETYPE_GB = 4;//获取文件大小单位为GB的double值

    private static DecimalFormat format_W1 = new DecimalFormat("#.0");
    private static DecimalFormat format_W2 = new DecimalFormat("#.00");

    /**
     * 获取文件指定文件的指定单位的大小
     *
     * @param filePath 文件路径
     * @param sizeType 获取大小的类型1为B、2为KB、3为MB、4为GB
     * @return double值的大小
     */
    public static double getFileOrFilesSize(String filePath, int sizeType) {
        File file = new File(filePath);
        long blockSize = 0;
        try {
            if (file.isDirectory()) {
                blockSize = getFileSizes(file);
            } else {
                blockSize = getFileSize(file);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Double.valueOf(formatFileSize(blockSize, sizeType, format_W2));
    }

    /**
     * 调用此方法自动计算指定文件或指定文件夹的大小
     *
     * @param filePath 文件路径
     * @return 计算好的带B、KB、MB、GB的字符串
     */
    public static String getAutoFileOrFilesSize(String filePath) {
        File file = new File(filePath);
        long blockSize = 0;
        try {
            if (file.isDirectory()) {
                blockSize = getFileSizes(file);
            } else {
                blockSize = getFileSize(file);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return formatFileSize(blockSize);
    }

    /**
     * 根据文件路径获取文件大小
     *
     * @param filePath
     * @return
     */
    public static long getFileSize(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return 0;
        }

        File file = new File(filePath);
        try {
            if (file.exists()) {
                return getFileSize(file);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * 获取指定文件大小
     *
     * @param file
     * @return
     * @throws Exception
     */
    public static long getFileSize(File file) throws Exception {
        long size = 0;
        if (file.exists()) {
            FileInputStream fis = null;
            fis = new FileInputStream(file);
            size = fis.available();
        } else {
            file.createNewFile();
        }
        return size;
    }

    /**
     * 获取指定文件夹
     *
     * @param f
     * @return
     * @throws Exception
     */
    public static long getFileSizes(File f) throws Exception {
        long size = 0;
        File flist[] = f.listFiles();
        for (int i = 0; i < flist.length; i++) {
            if (flist[i].isDirectory()) {
                size = size + getFileSizes(flist[i]);
            } else {
                size = size + getFileSize(flist[i]);
            }
        }
        return size;
    }

    /**
     * 文件大小位数自适应
     *
     * @param fileS
     * @return
     */
    public static String formatAutoFileSize(long fileS) {
        String fileSizeString;
        String wrongSize = "0B";
        if (fileS == 0) {
            return wrongSize;
        }
        if (fileS < 1024) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_B, format_W1);
        } else if (fileS < 1048576) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_KB, format_W1);
        } else if (fileS < 1073741824) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_MB, format_W2);
        } else {
            fileSizeString = formatFileSize(fileS, SIZETYPE_GB, format_W2);
        }
        return fileSizeString;
    }


    /**
     * 转换文件大小
     *
     * @param fileS
     * @return
     */
    public static String formatFileSize(long fileS) {
        String fileSizeString;
        String wrongSize = "0B";
        if (fileS == 0) {
            return wrongSize;
        }
        if (fileS < 1024) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_B, format_W2);
        } else if (fileS < 1048576) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_KB, format_W2);
        } else if (fileS < 1073741824) {
            fileSizeString = formatFileSize(fileS, SIZETYPE_MB, format_W2);
        } else {
            fileSizeString = formatFileSize(fileS, SIZETYPE_GB, format_W2);
        }
        return fileSizeString;
    }

    /**
     * 转换文件大小,指定转换的类型
     *
     * @param fileS
     * @param sizeType
     * @return
     */
    public static String formatFileSize(long fileS, int sizeType) {
        return formatFileSize(fileS, sizeType, format_W2);
    }

    /**
     * 转换文件大小,指定转换的类型
     *
     * @param fileS
     * @param sizeType
     * @return
     */
    public static String formatFileSize(long fileS, int sizeType, DecimalFormat format) {
        String fileSizeString = "";
        switch (sizeType) {
            case SIZETYPE_B:
                fileSizeString = format.format((double) fileS) + "B";
                break;
            case SIZETYPE_KB:
                fileSizeString = format.format((double) fileS / 1024) + "KB";
                break;
            case SIZETYPE_MB:
                fileSizeString = format.format((double) fileS / 1048576) + "MB";
                break;
            case SIZETYPE_GB:
                fileSizeString = format.format((double) fileS / 1073741824) + "GB";
                break;
            default:
                break;
        }
        return fileSizeString;
    }


    /**
     * 转换文件大小
     *
     * @param fileS
     * @return
     */
    public static String FormetFileSize(long fileS) {
        DecimalFormat df = new DecimalFormat("#.00");
        String fileSizeString;
        String wrongSize = "0B";
        if (fileS == 0) {
            return wrongSize;
        }
        if (fileS < 1024) {
            fileSizeString = df.format((double) fileS) + "B";
            //guorong@xdja.com 大小为B或者KB，略去小数点后的数值 bug :5076. begin
            if(fileSizeString.contains(".")){
                String[] s = fileSizeString.split("\\.");
                fileSizeString = s[0] + "B";
            }
            //guorong@xdja.com 大小为B或者KB，略去小数点后的数值 bug :5076. end
        } else if (fileS < 1048576) {
            fileSizeString = df.format((double) fileS / 1024) + "KB";
            //guorong@xdja.com 大小为B或者KB，略去小数点后的数值 bug :5076. begin
            if(fileSizeString.contains(".")){
                String[] s = fileSizeString.split("\\.");
                fileSizeString = s[0] + "KB";
            }
            //guorong@xdja.com 大小为B或者KB，略去小数点后的数值 bug :5076. end
        } else if (fileS < 1073741824) {
            fileSizeString = df.format((double) fileS / 1048576) + "MB";
        } else {
            fileSizeString = df.format((double) fileS / 1073741824) + "GB";
        }
        return fileSizeString;
    }

}
