package com.xdja.im.lib.album.ui.activity;

import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.MotionEvent;

import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.lib.album.ImageDataSource;
import com.xdja.im.lib.album.bean.ImageItem;
import com.xdja.im.lib.album.bean.ImageThumbBean;
import com.xdja.im.lib.album.ui.adapter.ImagePreviewAdapter;
import com.xdja.im.lib.album.ui.presenter.ImagePreviewPresenter;
import com.xdja.im.lib.album.ui.view.ViewImagePreview;
import com.xdja.im.lib.album.ui.view.vu.ImagePreviewVu;
import com.xdja.im.lib.album.utils.DataTools;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package:com.xdja.im.lib.album
 * @Author:Lei
 * @Creation:2017-04-23 12:22
 * @Version 1.0
 * @Description:图片预览界面
 */
public class ImagePreviewActivity extends ImBaseActivity<ImagePreviewPresenter, ImagePreviewVu>
        implements ImagePreviewPresenter {

    /**
     * 预览所有图片
     */
    public static final int TYPE_ALL = 0;
    /**
     * 预览已选图片
     */
    public static final int TYPE_SELECTED = 1;
    /**
     * 类型标记
     */
    public static final String TAG_TYPE = "P_Type";
    /**
     * 位置标记
     */
    public static final String TAG_POSITION = "P_Position";

    /**
     * 适配器
     */
    private ImagePreviewAdapter mAdapter;
    /**
     * 数据集合
     */
    private List<ImageItem> mImageItemList = new ArrayList<>();

    /**
     * 当前首次进入图片显示位置
     */
    private int mCurPosition;

    /**
     * 标志位，如果正在发送图片中，则为true
     */
    private boolean bSendingImage = false;

    @NonNull
    @Override
    protected Class<? extends ImagePreviewVu> getVuClass() {
        return ViewImagePreview.class;
    }

    @NonNull
    @Override
    protected ImagePreviewPresenter getPresenter() {
        return this;
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        int type = getIntent().getIntExtra(TAG_TYPE, 0);

        mImageItemList.clear();
        if (type == TYPE_ALL) {
            mImageItemList.addAll(ImageDataSource.getInstance().getAll());
            mCurPosition = getIntent().getIntExtra(TAG_POSITION, 0);
        } else {
            mImageItemList.addAll(ImageDataSource.getInstance().getSelects());
        }

        mAdapter = new ImagePreviewAdapter(this, mImageItemList);
        getVu().setAdapter(mAdapter);

        getVu().setCurrentPosition(mCurPosition);
    }

    @Override
    public int getImageSize() {
        return mImageItemList.size();
    }

    @Override
    public ImageItem getImageItem(int position) {
        if (mImageItemList != null && mImageItemList.size() > 0)
            return mImageItemList.get(position);
        else return null;
    }

    @Override
    public void updateItem(int position, ImageItem imageItem) {
        mImageItemList.remove(position);
        mImageItemList.add(position, imageItem);
        mAdapter.notifyDataSetChanged();
    }

    @Override
    public void sendImages(List<ImageItem> dataList) {

        getVu().setProgressBarVisible(true);

        bSendingImage = true;
        getVu().setProgressBarVisible(true);

        Observable.from(dataList)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<ImageItem, Observable<ImageThumbBean>>() {
                    @Override
                    public Observable<ImageThumbBean> call(ImageItem imageItem) {
                        return DataTools.createThumbNails(imageItem);
                    }
                })
                .toList()
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<ImageThumbBean>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("Create thumbnail error, " + e.getMessage());

                        bSendingImage = false;
                        getVu().setProgressBarVisible(false);
                        CommonTool.showToast(R.string.im_uikit_send_image_failed);
                    }

                    @Override
                    public void onNext(List<ImageThumbBean> imageThumbBeen) {
                        setResult(imageThumbBeen);
                    }
                });
    }

    /**
     * 返回结果
     *
     * @param thumbBean 缩略图信息
     */
    private void setResult(List<ImageThumbBean> thumbBean) {
        Intent intent = new Intent();

        //数据绑定
        Bundle bundle = new Bundle();
        ArrayList<ImageThumbBean> bundleList = new ArrayList<>();
        bundleList.addAll(thumbBean);
        //发送图片数据
        bundle.putParcelableArrayList(Constant.TAG_PIC_SELECT, bundleList);
        //添加数据到Intent
        intent.putExtras(bundle);
        //返回结果
        setResult(Constant.RESULT_CODE_ALBUM, intent);
        finish();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        bSendingImage = false;
    }

    @Override
    public boolean dispatchTouchEvent(MotionEvent ev) {
        if (bSendingImage) {
            return true;
        }
        return super.dispatchTouchEvent(ev);
    }
}
