package com.xdja.im.lib.album.ui.view;

import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.support.v4.view.ViewPager;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ProgressBar;

import com.xdja.im.common.utils.FileSizeUtils;
import com.xdja.im.lib.album.ImageDataSource;
import com.xdja.im.lib.album.ImagePickerConfig;
import com.xdja.im.lib.album.bean.ImageItem;
import com.xdja.im.lib.album.ui.adapter.ImagePreviewAdapter;
import com.xdja.im.lib.album.ui.presenter.ImagePreviewPresenter;
import com.xdja.im.lib.album.ui.view.vu.ImagePreviewVu;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImActivitySuperView;
import com.xdja.im.uikit.utils.log.LogUtil;

/**
 * @Package: com.xdja.im.lib.album.ui.view
 * @Author: xdjaxa
 * @Creation: 2017-04-25 17:49
 * @Version V1.0
 * @Description:
 */
public class ViewImagePreview extends ImActivitySuperView<ImagePreviewPresenter>
        implements ImagePreviewVu, ViewPager.OnPageChangeListener, View.OnClickListener {

    private ViewPager mViewPager;
    /**
     * 发送按钮
     */
    private Button mSendBtn;
    /**
     * 原图选择按钮
     */
    private Button mOriginBtn;
    /**
     * 选择图片按钮
     */
    private Button mSelectBtn;
    /**
     * 加载进度
     */
    private ProgressBar mLoadingPBar;
    /**
     * 当前选中item索引
     */
    private int mCurrentItemIndex;

    @Override
    protected int getLayoutRes() {
        return R.layout.im_uikit_activity_image_preview;
    }

    @Override
    protected int getToolbarType() {
        return ToolbarDef.NAVIGATE_BACK;
    }

    @Override
    protected void injectView() {
        super.injectView();
        mViewPager = (ViewPager) findViewById(R.id.viewpager);
        mSendBtn = (Button) findViewById(R.id.btn_send);
        mOriginBtn = (Button) findViewById(R.id.btn_original);
        mSelectBtn = (Button) findViewById(R.id.btn_select);
        mLoadingPBar = (ProgressBar) findViewById(R.id.pbar_loading);

        //设置监听器
        mSendBtn.setOnClickListener(this);
        mSelectBtn.setOnClickListener(this);
        mViewPager.addOnPageChangeListener(this);
        mOriginBtn.setOnClickListener(this);
    }

    @Override
    public void setAdapter(ImagePreviewAdapter adapter) {
        mViewPager.setAdapter(adapter);

        adapter.notifyDataSetChanged();

        //刷新标题指示器内容
        refreshTitleIndicator();
        //刷新已选图片指示器
        refreshSelectPictureIndicator();
        //更新图片信息
        refreshLocalPictureInfo(0);
    }

    @Override
    public void setCurrentPosition(int position) {
        mViewPager.setCurrentItem(position);
    }

    @Override
    public void refreshTitleIndicator() {

        getActivity().setTitle(String.format(getStringRes(R.string.im_uikit_selest),
                (mCurrentItemIndex + 1), getPresenter().getImageSize()));
    }

    @Override
    public int getSelectedCount() {
        return ImageDataSource.getInstance().size();
    }

    @Override
    public void setProgressBarVisible(boolean bVisible) {
        mLoadingPBar.setVisibility(bVisible ? View.VISIBLE : View.GONE);
    }

    @Override
    public void refreshSelectPictureIndicator() {
        //当前已经选择的图片个数
        int selectCnt = getSelectedCount();
        if (selectCnt > 0 ) {
            setSendBtnClickable(true);
            mSendBtn.setText(String.format(getStringRes(R.string.im_uikit_picture_send_indicator), selectCnt,
                    ImagePickerConfig.MAX_SELECT_NUM));
        } else {
            setSendBtnClickable(false);
            mSendBtn.setText(R.string.im_uikit_send);
        }
    }

    /**
     * 发送按钮是否可点击
     *
     * @param clickable
     */
    private void setSendBtnClickable(boolean clickable) {
        if (clickable) {
            //设置可点击
            mSendBtn.setClickable(true);
            //动态设置大小
            ViewGroup.LayoutParams params = mSendBtn.getLayoutParams();
            params.width = ViewGroup.LayoutParams.WRAP_CONTENT;
            mSendBtn.setLayoutParams(params);
            mSendBtn.setTextColor(Color.parseColor("#F3000000"));
        } else {
            //设置不可点击
            mSendBtn.setClickable(false);
            //动态设置大小，默认长度 46dp
            ViewGroup.LayoutParams params = mSendBtn.getLayoutParams();
            params.width = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 46,
                    getContext().getResources().getDisplayMetrics());
            mSendBtn.setLayoutParams(params);
            //文字颜色
            mSendBtn.setTextColor(Color.parseColor("#77000000"));
        }
    }

    @Override
    public void refreshLocalPictureInfo(int position) {
        ImageItem imageItem = getPresenter().getImageItem(position);
        if (imageItem == null) {
            LogUtil.e("ERROR: get image info error. " + position);
            return;
        }

        mOriginBtn.setText(String.format(getStringRes(R.string.im_uikit_original_image),
                FileSizeUtils.formatFileSize(imageItem.getSize())));

        boolean isSelect = ImageDataSource.getInstance().contains(imageItem);

        //图片选中状态信息
        mSelectBtn.setBackgroundResource(isSelect ? R.mipmap.im_uikit_icon_selected_on :
                R.mipmap.im_uikit_icon_selected_off);

        if (imageItem.isOriginal()) {
            mOriginBtn.setTextColor(Color.parseColor("#F3000000"));
            setOriginalPictureResource(R.mipmap.im_uikit_icon_selected_on);
        } else {
            mOriginBtn.setTextColor(Color.parseColor("#77000000"));
            setOriginalPictureResource(R.mipmap.im_uikit_icon_selected_off);
        }
    }

    @Override
    public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {

    }

    @Override
    public void onPageSelected(int position) {

        mCurrentItemIndex = position;
        //更新标题指示器
        refreshTitleIndicator();
        //更新当前已选图片信息
        refreshLocalPictureInfo(position);
    }

    @Override
    public void onPageScrollStateChanged(int state) {

    }

    @Override
    public void onClick(View v) {
        int resId = v.getId();

        if (resId == R.id.btn_send) {
            mSendBtn.setEnabled(false);
            mOriginBtn.setEnabled(false);
            getPresenter().sendImages(ImageDataSource.getInstance().getSelects());

        } else if (resId == R.id.btn_select) {//选择按钮
            ImageItem imageItem = getPresenter().getImageItem(mCurrentItemIndex);
            if (imageItem == null) {
                LogUtil.e("ERROR: get image info error. " + mCurrentItemIndex);
                return;
            }
            boolean isSelect = ImageDataSource.getInstance().contains(imageItem);

            if(!isSelect) {
                if (ImageDataSource.getInstance().size() >= ImagePickerConfig.MAX_SELECT_NUM) {
                    showToast(String.format(getStringRes(R.string.im_uikit_img_select_limit),
                            ImagePickerConfig.MAX_SELECT_NUM));
                    return;
                }
            }

            if (isSelect) { //图片已选
                ImageDataSource.getInstance().remove(imageItem);
                setSelectPicture(false);
                setOriginalPictureResource(R.mipmap.im_uikit_icon_selected_off);
            } else {
                ImageDataSource.getInstance().add(imageItem);
                setSelectPicture(true);
            }
            refreshSelectPictureIndicator();
        } else if (resId == R.id.btn_original) {//原图选择按钮
            ImageItem imageItem = getPresenter().getImageItem(mCurrentItemIndex);
            if (imageItem == null) {
                LogUtil.e("ERROR: get image info error. " + mCurrentItemIndex);
                return;
            }
            boolean isOriginal = imageItem.isOriginal();

            if (isOriginal) { //已经选择原图
                //取消选择原图
                mOriginBtn.setTextColor(Color.parseColor("#77000000"));
                setOriginalPictureResource(R.mipmap.im_uikit_icon_selected_off);
            } else {
                boolean isSelect = ImageDataSource.getInstance().contains(imageItem);
                if(!isSelect) {
                    if (ImageDataSource.getInstance().size() >= ImagePickerConfig.MAX_SELECT_NUM) {
                        showToast(String.format(getStringRes(R.string.im_uikit_img_select_limit),
                                ImagePickerConfig.MAX_SELECT_NUM));
                        return;
                    }
                }

                //选择原图
                mOriginBtn.setTextColor(Color.parseColor("#F3000000"));
                mOriginBtn.setText(String.format(getStringRes(R.string.im_uikit_original_image),
                        FileSizeUtils.formatFileSize(imageItem.getSize())));
                //更新选择图标
                mSelectBtn.setBackgroundResource(R.mipmap.im_uikit_icon_selected_on);
                //更新原图显示图片
                setOriginalPictureResource(R.mipmap.im_uikit_icon_selected_on);
            }
            imageItem.setOriginalState(!isOriginal);
            ImageDataSource.getInstance().updateItem(imageItem);
            //更新选中原图状态
            getPresenter().updateItem(mCurrentItemIndex, imageItem);
            refreshSelectPictureIndicator();
        }
    }

    /**
     * 设置图片选中状态
     * @param isSelect
     */
    private void setSelectPicture(boolean isSelect){
        //设置状态
        if (isSelect) {
            mSelectBtn.setBackgroundResource(R.mipmap.im_uikit_icon_selected_on);
        } else {
            mSelectBtn.setBackgroundResource(R.mipmap.im_uikit_icon_selected_off);
        }
    }

    /**
     * 更改原图选中图标
     * @param resId
     */
    private void setOriginalPictureResource(int resId){
        //更新原图显示图片
        Drawable unSelectDrawable = getDrawableRes(resId);
        if (unSelectDrawable != null) {
            unSelectDrawable.setBounds(0, 0, unSelectDrawable.getIntrinsicWidth(),
                    unSelectDrawable.getIntrinsicHeight());
            mOriginBtn.setCompoundDrawables(unSelectDrawable, null, null, null);
        }
    }
}
