package com.xdja.im.lib.filemanage.fragment;

import android.os.Bundle;
import android.support.annotation.NonNull;
import android.view.View;
import android.widget.ExpandableListView;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.filemanage.adapter.FileListAdapter;
import com.xdja.im.lib.filemanage.commend.IFileListCommand;
import com.xdja.im.lib.filemanage.presenter.IMFragmentPresenter;
import com.xdja.im.lib.filemanage.util.FileInfoCollection;
import com.xdja.im.lib.filemanage.util.XToast;
import com.xdja.im.lib.filemanage.view.IFileListVu;
import com.xdja.im.lib.filemanage.view.ViewFileList;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import rx.Subscriber;


/**
 * <p>Author: leiliangliang   </br>
 * <p>Date: 2016/12/6 19:25   </br>
 * <p>Package: com.xdja.imp.presenter.fragment</br>
 * <p>Description:            </br>
 */
public class FileListPresenter extends IMFragmentPresenter<IFileListCommand, IFileListVu>
        implements IFileListCommand {

    /**
     * 最多可选择文件个数
     */
    private static final int MAX_SELECT_COUNT = 9;
    private static final long MAX_SELECT_SIZE = 30 * 1024 * 1024;

    //事件总线
    BusProvider busProvider;

//    @Inject
//    Lazy<QueryLocalFiles> queryLocalFiles;

    private FileListAdapter mAdapter;

    /**
     * 分组标题
     */
    private final List<String> mGroupTitles = new ArrayList<>();

    /**
     * 分组对应的文件列表
     */
    private final List<List<FileItem>> mLocalFileList = new ArrayList<>();

    /**
     * type值，根据不同的类型，加载不同的文件数据
     */
    private int mFileType = 0;

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (null == useCaseComponent) {
            LogUtil.e("FileListPresenter", "useCaseComponent is null");
            return;
        }

        //初始化注入
        useCaseComponent.inject(this);

        //初始化事件总线
        busProvider = BusProvider.getInstance();
        busProvider.register(this);

        Bundle bundle = getArguments();
        if (bundle != null) {
            mFileType = bundle.getInt(ConstDef.ARGS_FILE_TYPE);
        }

        //控件相关
        initViews();

        //加载数据
        loadData();
    }

    private void initViews() {

        getVu().getListView().setOnChildClickListener(new ExpandableListView.OnChildClickListener() {
            @Override
            public boolean onChildClick(ExpandableListView parent, View v,
                                        int groupPosition, int childPosition, long id) {

                FileItem fileInfo = mLocalFileList.get(groupPosition).get(childPosition);
                if (fileInfo != null) {

                    //文件选择个数限制
                    if (!fileInfo.isSelected()) {
                        if (fileInfo.getFileSize() >= MAX_SELECT_SIZE) {
                            new XToast(getContext()).display(getString(R.string.im_uikit_file_is_too_larger));
                            return false;
                        }
                        if (FileInfoCollection.getInstance().getSelectedFileCount() >= MAX_SELECT_COUNT) {
                            new XToast(getContext()).display(String.format(getString(R.string.im_uikit_select_file_hint),
                                    MAX_SELECT_COUNT));
                            return false;
                        }
                    }

                    //通知刷新界面
                    fileInfo.setSelected(!fileInfo.isSelected());
                    if (fileInfo.isSelected()) {
                        FileInfoCollection.getInstance().putFileToSelectedCache(
                                fileInfo.getFilePath() + fileInfo.getModifiedDate(),
                                fileInfo);
                    } else {
                        FileInfoCollection.getInstance().removeToSelectedCache(
                                fileInfo.getFilePath() + fileInfo.getModifiedDate());
                    }
                    IMProxyEvent.FileSelectedEvent event = new IMProxyEvent.FileSelectedEvent(fileInfo);
                    busProvider.post(event);


                    //更新数据
                    mLocalFileList.get(groupPosition).remove(childPosition);
                    mLocalFileList.get(groupPosition).add(childPosition, fileInfo);
                    mAdapter.notifyDataSetChanged();
                }
                return false;
            }
        });
    }

    /**
     * 获取数据
     */
    private void loadData() {

        ImUiKit.getInstance().getComponent()
                .proxyRepository()
                .queryLocalFile(mFileType, new Subscriber<Map<String, List<FileItem>>>() {
                    @Override
                    public void onCompleted() {
                    }

                    @Override
                    public void onError(Throwable e) {
                        getVu().setEmptyView();
                    }

                    @Override
                    public void onNext(Map<String, List<FileItem>> stringListMap) {
                        if (stringListMap.size() == 0) {

                            getVu().setEmptyView();

                        } else {
                            mGroupTitles.clear();
                            mLocalFileList.clear();
                            for (String key : stringListMap.keySet()) {
                                mGroupTitles.add(key);
                                mLocalFileList.add(stringListMap.get(key));
                            }
                            mAdapter = new FileListAdapter(getContext(), mGroupTitles);
                            mAdapter.addLocalFiles(mLocalFileList);
                            getVu().setListAdapter(mAdapter);
                        }
                    }
                });
    }

    @NonNull
    @Override
    protected Class<? extends IFileListVu> getVuClass() {
        return ViewFileList.class;
    }

    @NonNull
    @Override
    protected IFileListCommand getCommand() {
        return this;
    }


    @Override
    public void onDestroy() {
        super.onDestroy();
        if (busProvider != null){
            busProvider.unregister(this);
        }
    }
}
