package com.xdja.im.lib.share;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;

import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.filemanage.util.IMFileUtils;
import com.xdja.im.lib.filemanage.util.TextUtil;
import com.xdja.im.lib.filemanage.util.ToolUtil;
import com.xdja.im.lib.share.adapter.ShareSessionListAdapter;
import com.xdja.im.lib.share.model.ConvertBean;
import com.xdja.im.lib.share.model.IntentBean;
import com.xdja.im.lib.share.model.ShareBean;
import com.xdja.im.lib.share.model.ShareSelectType;
import com.xdja.im.lib.share.presenter.ShareSessionListPresenter;
import com.xdja.im.lib.share.view.ShareViewSessionList;
import com.xdja.im.lib.share.view.vu.ShareViewSessionListVu;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.utils.DataConvertUtils;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * Created by wenqi on 2018/3/1.
 * 分享
 */

public class ShareSessionListActivity extends ImBaseActivity<ShareSessionListPresenter, ShareViewSessionListVu> implements ShareSessionListPresenter {


    /**
     * IMSDK回调
     */
    @Inject
    Lazy<ProxyRepository> proxyRepository;


    private ShareSessionListAdapter adapter;

    private Intent intent;


    /**
     * 会话数据集合
     */
    private List<TalkSessionBean> mSessionList = new ArrayList<>();

    /**
     * 会话数据类型(默认加载全部数据)
     */
    private int mTalkType = ConstDef.CHAT_TYPE_DEFAULT;

    @Override
    protected void preBindView(Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);
        intent = getIntent();
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        if (mComponent == null) {
            return;
        }
        mComponent.inject(this);

        BusProvider.getInstance().register(this);
        adapter = new ShareSessionListAdapter(mSessionList, this);
        getVu().setAdapter(adapter);
        getVu().setToolBarVisibility(View.VISIBLE);
        getVu().setTitle("选择");
        getVu().setTitleColor(R.color.im_uiKit_share_title_color);
        getVu().setToolBarAddVisibility(View.INVISIBLE);
        //加载会话数据
        startLoadSessionData();
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        this.intent = intent;
    }

    @NonNull
    @Override
    protected Class<? extends ShareViewSessionListVu> getVuClass() {
        return ShareViewSessionList.class;
    }

    @NonNull
    @Override
    protected ShareSessionListPresenter getPresenter() {
        return this;
    }

    @Override
    public Activity getActivity() {
        return this;
    }

    /**
     * 搜索
     */
    @Override
    public void onSearchChanged(String searchKey) {
        IMUiKitProxyImpl.getInstance().callSessionSearch(getActivity(), searchKey, ShareSelectType.SHARE_SEARCH_CONTACT);
    }

    /**
     * 创建新会话
     */
    @Override
    public void onCreateSessionClick() {
        IMUiKitProxyImpl.getInstance().callSessionContact(getActivity(), ShareSelectType.SHARE_NEW_CONTACT);
    }

    /**
     * 更多联系人
     */
    @Override
    public void onMoreContactClick() {
        IMUiKitProxyImpl.getInstance().callSessionContact(getActivity(), ShareSelectType.SHARE_MORE_CONTACT);
    }

    /**
     * 列表项被点击
     *
     * @param item
     */
    @Override
    public void onListItemClick(TalkSessionBean item) {
        if (item != null) {
            getVu().showDialog(intent, item);
        }
    }

    /**
     * 对话框确认发送
     *
     * @param shareBeanList
     * @param intentBean
     * @param dialogContent
     */
    @Override
    public void onDialogRequestSend(final List<ShareBean> shareBeanList, final IntentBean intentBean, final String dialogContent) {

        Observable.from(shareBeanList)
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(intentBean.getContent())) {
                            sendTextMessage(intentBean.getContent(), shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(dialogContent)) {
                            sendTextMessage(dialogContent, shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (intentBean.getUriArrayList() != null && intentBean.getUriArrayList().size() > 0) {
                            sendIntentFile(shareBean, intentBean);
                            return true;
                        } else
                            return false;
                    }
                })
                .subscribe(new Subscriber<ShareBean>() {
                    @Override
                    public void onCompleted() {
                        getVu().showSimpleDialog();
                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                    }

                    @Override
                    public void onNext(ShareBean shareBean) {
                        LogUtil.d("shareBean" + shareBean.toString());
                    }
                });
    }


    private void sendIntentFile(final ShareBean shareBean, IntentBean intentBean) {

        Observable.just(intentBean)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<IntentBean, Observable<String>>() {
                    @Override
                    public Observable<String> call(IntentBean intentBean) {
                        return Observable.from(intentBean.getUriArrayList());
                    }
                })
                .filter(new Func1<String, Boolean>() {
                    @Override
                    public Boolean call(String s) {
                        Uri uri = Uri.parse(Uri.decode(s));
                        String path = uri.getPath();
                        if (!TextUtils.isEmpty(path)) {
                            File file = new File(path);
                            return file.exists() && file.canRead();
                        } else
                            return false;
                    }
                })
                .map(new Func1<String, FileInfo>() {
                    @Override
                    public FileInfo call(String s) {
                        FileItem info = IMFileUtils.queryLocalFiles(getContext(), Uri.parse(Uri.decode(s)));
                        FileInfo fileInfo = new FileInfo();
                        fileInfo.setFilePath(info.getFilePath());
                        fileInfo.setFileName(info.getFileName());
                        fileInfo.setFileSize(info.getFileSize());
                        String name = info.getFileName();
                        String suffix = ToolUtil.getLastString(name, ".");
                        if (!TextUtils.isEmpty(suffix)) {
                            fileInfo.setSuffix(suffix);
                        }
                        fileInfo.setFileType(info.getFileType());
                        boolean isImage = IMFileUtils.isImageFileType(name);
                        boolean isVideo = IMFileUtils.isVideoFileType(name);

                        if (isImage) {
                            fileInfo = DataConvertUtils.mapImagePicFileInfo(fileInfo);
                        }

                        if (isVideo) {
                            fileInfo = DataConvertUtils.mapVideoFileInfo(fileInfo);
                            if (fileInfo != null) {
                                fileInfo.setFileType(ConstDef.FILE_TYPE_VIDEO);
                            }
                        }
                        return fileInfo;
                    }
                })
                .toList()
                .subscribeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<FileInfo>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                    }

                    @Override
                    public void onNext(List<FileInfo> fileInfos) {

                        if (fileInfos != null) {
                            sendFileMessage(fileInfos, shareBean.getAccount(), shareBean.getType());
                        }

                    }
                });

    }


    /**
     * 对话框取消发送
     */
    @Override
    public void onDialogCancleSend() {
        showToast("取消发送");
    }

    /**
     * 对话框分享完成确认
     */
    @Override
    public void onTipsDialogTrueClick() {
        IMUiKitProxyImpl.getInstance().shareOverCallback(getActivity());
        finish();
    }

    /**
     * 对话框分享完成取消
     */
    @Override
    public void onTipsDialogClick() {
        finish();
    }

    /**
     * 发送文本消息
     *
     * @param message 文本消息内容
     * @return {@code true} 消息可发送，并且发送的动作操作成功 <br />
     * {@code false} 消息不可发送，或者发送的动作操作失败
     */
    @Override
    public boolean sendTextMessage(String message, String sendTo, int type) {

        if (proxyRepository == null || proxyRepository.get() == null) {
            return false;
        }
        proxyRepository.get().sendTextMessage(message, sendTo, false, type == ConstDef.CHAT_TYPE_P2G, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {
                showToast("发送成功");
            }
        });

        return false;
    }

    /**
     * 发送文件消息
     *
     * @param fileList 文件列表
     * @return {@code true} 消息可发送，并且发送的动作操作成功 <br />
     * {@code false} 消息不可发送，或者发送的动作操作失败
     */
    @Override
    public void sendFileMessage(List<FileInfo> fileList, String sendTo, int type) {

        if (proxyRepository == null || proxyRepository.get() == null) {
            return;
        }
        proxyRepository.get().sendFileMessage(sendTo, false, type == ConstDef.CHAT_TYPE_P2G, fileList, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {
                showToast("文件发送成功");
            }
        });
    }


    /**
     * 加载会话列表数据
     */
    private void startLoadSessionData() {
        //加载动画
        getVu().setProgressBarVisibility(View.VISIBLE);
        if (proxyRepository == null || proxyRepository.get() == null) {
            return;
        }
        proxyRepository.get().getSessionListBeans("", 0, mTalkType, new Subscriber<List<TalkSessionBean>>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                getVu().setProgressBarVisibility(View.GONE);
                LogUtil.e("ERROR: Load session data error. " + e.getMessage());
            }

            @Override
            public void onNext(List<TalkSessionBean> sessionBeanList) {
                LogUtil.d("Load session data finished.");
                if (sessionBeanList != null && sessionBeanList.size() > 0) {
                    mSessionList.clear();
                    //剔除重复的会话
                    removeExists(sessionBeanList);
                    mSessionList.addAll(sessionBeanList);
                    adapter.notifyDataSetChanged();
                    getVu().setShareSessionEmptyVisibility(View.GONE);
                } else {
                    getVu().setShareSessionEmptyVisibility(View.VISIBLE);
                }
                getVu().setProgressBarVisibility(View.GONE);
            }
        });
    }


    /**
     * 移除重复消息
     *
     * @param beans
     */
    private void removeExists(List<TalkSessionBean> beans) {
        Iterator<TalkSessionBean> iterator = beans.iterator();
        for (; iterator.hasNext(); ) {
            TalkSessionBean bean = iterator.next();

            int index = beans.indexOf(bean);
            if (index > -1 && bean != beans.get(index)) {
                iterator.remove();
            }
        }
    }


    @Subscribe
    public void onShareContactRefreshEvent(IMProxyMessageEvent.ShareContactRefreshEvent event) {
        int type = event.getType();
        List<ShareBean> list = event.getList();
        switch (type) {
            case ShareSelectType.SHARE_NEW_CONTACT:
            case ShareSelectType.SHARE_MORE_CONTACT:

                break;
            case ShareSelectType.SHARE_SEARCH_CONTACT:

                break;
        }
        getVu().showMultiContactDialog(intent, list);


    }


}
