package com.xdja.im.base.executor;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import rx.Scheduler;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.base.executor
 * @Author: xdjaxa
 * @Creation: 2017-04-11 15:33
 * @Version V1.0
 * @Description:异步线程接口定义
 */
public class ThreadExecutor implements Executor {

    private static final int INITIAL_POOL_SIZE = 3;

    private static final int MAX_POOL_SIZE = 5;

    // Sets the amount of time an idle thread waits before terminating
    private static final int KEEP_ALIVE_TIME = 10;

    // Sets the Time Unit to seconds
    private static final TimeUnit KEEP_ALIVE_TIME_UNIT = TimeUnit.SECONDS;

    @NonNull
    private final BlockingQueue<Runnable> workQueue;

    @NonNull
    private final ThreadPoolExecutor threadPoolExecutor;

    @NonNull
    private final ThreadFactory threadFactory;

    @Inject
    public ThreadExecutor() {
        this.workQueue = new LinkedBlockingQueue<>();
        this.threadFactory = new JobThreadFactory();
        this.threadPoolExecutor = new ThreadPoolExecutor(INITIAL_POOL_SIZE, MAX_POOL_SIZE,
                KEEP_ALIVE_TIME, KEEP_ALIVE_TIME_UNIT, this.workQueue, this.threadFactory);
    }

    public void execute(@Nullable Runnable runnable) {
        if (runnable == null) {
            throw new IllegalArgumentException("Runnable to execute cannot be null");
        }
        this.threadPoolExecutor.execute(runnable);
    }

    public void submite(@Nullable Runnable runnable) {
        if (runnable == null) {
            throw new IllegalArgumentException("Runnable to submite cannot be null");
        }
        this.threadPoolExecutor.submit(runnable);
    }

    private static class JobThreadFactory implements ThreadFactory {
        private static final String THREAD_NAME = "android_";
        private int counter = 0;

        @NonNull
        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = new Thread(runnable, THREAD_NAME + counter);
            counter++;
            return thread;
        }
    }

    public Scheduler getScheduler() {
        return Schedulers.from(this);
    }

}
