package com.xdja.im.common.database.dao;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.text.TextUtils;

import com.xdja.im.common.database.DatabaseManager;
import com.xdja.im.common.database.builder.MeetInfoEntityBuilder;
import com.xdja.im.common.database.helper.UpdateArgs;
import com.xdja.im.core.model.message.MeetInfo;

import java.util.ArrayList;
import java.util.List;

/**
 * @Package: com.xdja.im.common.database.dao
 * @Author: xdjaxa
 * @Creation: 2017-05-22 18:15
 * @Version V1.0
 * @Description:
 */
public class MeetInfoEntityDao extends AbstractDao<MeetInfo> {

    private MeetInfoEntityDao() {

    }

    private static class SingletonInstance{
        private static final MeetInfoEntityDao mInstance = new MeetInfoEntityDao();
    }

    public static MeetInfoEntityDao getInstance() {
        return SingletonInstance.mInstance;
    }

    public long insert(MeetInfo meetInfo) {
        SQLiteDatabase db = getDatabase();
        if (db != null) {
            return insert(db, meetInfo, MeetInfoEntityBuilder.insertSql());
        }
        return -1;
    }

    public int delete(String sessionFlag) {
        if (TextUtils.isEmpty(sessionFlag)) {
            return -1;
        }
        return super.delete(MeetInfoEntityBuilder.delSql(sessionFlag));
    }

    public int deleteAll() {
        return super.delete(MeetInfoEntityBuilder.delAllSql());
    }

    public MeetInfo get(String sessionFlag) {
        MeetInfo msgEntryDb = null;
        Cursor cursor = null;
        try {
            cursor = query(MeetInfoEntityBuilder.querySql(sessionFlag));
            if (cursor != null && cursor.moveToNext()) {
                msgEntryDb = readEntry(cursor, 0);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        return msgEntryDb;
    }

    public void update(MeetInfo meetInfo) {
        UpdateArgs args = new UpdateArgs(MeetInfoEntityBuilder.TABLE_NAME);
        ContentValues values = new ContentValues();
        values.put(MeetInfoEntityBuilder.ID, meetInfo.getId());
        values.put(MeetInfoEntityBuilder.MEETINFO_ISSHOW, meetInfo.getIsShow());
        values.put(MeetInfoEntityBuilder.MEETINFO_FLAG, meetInfo.getTaklFlag());
        values.put(MeetInfoEntityBuilder.MEETINFO_TIPS, meetInfo.getTips());

        String whereClause[] = new String[]{MeetInfoEntityBuilder.MEETINFO_FLAG};
        String[] whereArgs = new String[]{meetInfo.getTaklFlag()};

        args.setValues(values);
        args.setWhereClause(whereClause);
        args.setWhereArgs(whereArgs);
        update(args);
    }


    public List<MeetInfo> getAll() {
        List<MeetInfo> meetInfoList = new ArrayList<>();
        Cursor cursor = null;
        try {
            cursor = query(MeetInfoEntityBuilder.queryAll());
            while (cursor != null && cursor.moveToNext()) {
                meetInfoList.add(readEntry(cursor, 0));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        return meetInfoList;
    }

    @Override
    protected MeetInfo readEntry(Cursor cursor, int offset) {
        MeetInfo entity = new MeetInfo(
                cursor.isNull(offset + 0) ? null : cursor.getLong(offset + 0),
                cursor.isNull(offset + 1) ? null : cursor.getInt(offset + 1),
                cursor.isNull(offset + 2) ? null : cursor.getString(offset + 2),
                cursor.isNull(offset + 3) ? null : cursor.getString(offset + 3)
        );
        return entity;
    }

    @Override
    protected void bindValues(SQLiteStatement stmt, MeetInfo entity) {
        stmt.clearBindings();
        Long id = entity.getId();
        if (id != null) {
            stmt.bindLong(1, id);
        }

        Integer isShow = entity.getIsShow();
        if (isShow != null) {
            stmt.bindLong(2, isShow);
        }

        String talkFlag = entity.getTaklFlag();
        if (talkFlag != null) {
            stmt.bindString(3, talkFlag);
        }

        String tips = entity.getTips();
        if (tips != null) {
            stmt.bindString(4, tips);
        }
    }

    /**
     * 获取数据库句柄
     *
     * @return
     * @throws IllegalAccessError 数据库句柄为空异常
     */
    private SQLiteDatabase getDatabase() {
        SQLiteDatabase db = DatabaseManager.getInstance().getDatabase();
        if (db == null) {
            throw new IllegalAccessError("open database failed.");
        }
        return db;
    }
}
