package com.xdja.im.core.proxy;

import android.text.TextUtils;
import android.util.Log;

import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.account.ActionDesc;
import com.xdja.im.core.model.account.GroupAction;
import com.xdja.im.core.model.event.IMContactEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.utils.DataCache;
import com.xdja.im.core.utils.ToolUtils;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.Iterator;
import java.util.List;

import rx.Subscriber;

/**
 * @Package: com.xdja.im.core.proxy
 * @Author: xdjaxa
 * @Creation: 2017-08-21 11:43
 * @Version V1.0
 * @Description:联系人群组相关接口
 */
public class ImUiKitContactProxyImpl implements ImUiKitContactProxy {

    private String mCurAccount;

    private static class SingletonInstance {
        private static final ImUiKitContactProxyImpl mInstance = new ImUiKitContactProxyImpl();
    }

    public static ImUiKitContactProxyImpl getInstance() {
        return SingletonInstance.mInstance;
    }

    private ImUiKitContactProxyImpl() {
        mCurAccount = ImUiKit.getInstance().getCurAccount();
    }

    @Override
    public void notifyGroupChanged(String groupId, GroupAction action, ActionDesc actionDesc) {
        switch (action) {
            case ACTION_ADD:    //添加群成员
                onAddMemberAction(groupId, actionDesc);
                break;

            case ACTION_REMOVE: //移除群成员
                onRemoveMemberAction(groupId, actionDesc);
                break;

            case ACTION_CREATE: //群组创建
                onCreateGroupAction(groupId, actionDesc);
                break;

            case ACTION_UPDATE: //群组更新
                onGroupUpdateAction(groupId, actionDesc);
                break;

            case ACTION_DISSOLUTION: //群组解散
                onDissolutionAction(groupId, actionDesc);
                break;
            case ACTION_QUIT:
                onQuitAction(groupId, actionDesc);
                break;
        }
    }

    /**
     * 生成群会话标识
     *
     * @param groupId
     * @return
     */
    private String getGroupFlag(String groupId) {
        return ToolUtils.getSessionTag(groupId, ConstDef.CHAT_TYPE_P2G);
    }

    /**
     * 添加群成员事件
     * 如果提供message自定义消息内容，内部不再组织自定义消息内容
     * @param groupId    群组Id
     * @param actionDesc 相关数据信息
     * @Note:1）群主：提示拉取XX进入通知 2）群成员：提示被加入群组
     */
    private void onAddMemberAction(String groupId, ActionDesc actionDesc) {

        String content = actionDesc.getMessage();
        if (TextUtils.isEmpty(content)) {
            content = buildAddMemberMessage(actionDesc.getMaster(), actionDesc.getAccountList());
        }
        DataCache.getInstance().remove(getGroupFlag(groupId));
        sendNotifyMessage(groupId, content);

    }

    /**
     * 移除群成员事件
     * 如果提供message自定义消息内容，内部不再组织自定义消息内容
     * @param groupId    群组Id
     * @param actionDesc 相关数据信息
     * @Note:1)群主：将XX人移除群组 2）群成员：被移除群组
     */
    private void onRemoveMemberAction(String groupId, ActionDesc actionDesc) {
        String content = actionDesc.getMessage();
        if (TextUtils.isEmpty(content)) {
            content = buildRemoveMemberNotifyContent(actionDesc.getMaster(), actionDesc.getAccountList());
        }
        sendNotifyMessage(groupId, content);
    }


    /**
     * 创建群组事件
     * 如果提供message自定义消息内容，内部不再组织自定义消息内容
     * @param groupId    群组Id
     * @param actionDesc 相关数据信息<br/>
     * @Note：创建群组事件，只需要向imsdk发送一条自定义消息即可。 <br><br/>
     * 1）imsdk回查询如果当前没有改会话（grioupid对应的群组），则会话创建新会话，并回调接口自定义消息；<br><br/>
     * 2）imsdk查询存在改会话，则直接发送自定义消息即可<br><br/>
     */
    private void onCreateGroupAction(String groupId, ActionDesc actionDesc) {

        //上层已经定义好发送自定义消息内容
        String content = actionDesc.getMessage();
        if (TextUtils.isEmpty(content)) {
            content = buildCreateGroupMessage(actionDesc.getMaster(), actionDesc.getAccountList());
        }

        DataCache.getInstance().remove(getGroupFlag(groupId));
        sendNotifyMessage(groupId, content);
    }

    /**
     * 群组更新事件
     * 群组更新事件中需要增加说明，请提供 message 自定义内容
     * @param groupId    群组Id
     * @param actionDesc 相关数据信息
     */
    private void onGroupUpdateAction(String groupId, ActionDesc actionDesc) {

        String content = actionDesc.getMessage();
        if (!TextUtils.isEmpty(content)) {
            sendNotifyMessage(groupId, content);
        }

        DataCache.getInstance().remove(getGroupFlag(groupId));
        IMContactEvent.RefreshGroupEvent refreshGroupEvent = new IMContactEvent.RefreshGroupEvent();
        refreshGroupEvent.setAccount(groupId);
        BusProvider.getInstance().post(refreshGroupEvent);
    }

    /**
     * 群组解散事件
     *
     * @param groupId    群组Id
     * @param actionDesc 相关数据信息
     * @Note:1）群主：直接退出当前聊天界面，并删除会话 2）群成员：提示群主解散群组
     */
    private void onDissolutionAction(String groupId, ActionDesc actionDesc) {
        String content = actionDesc.getMessage();
        if (TextUtils.isEmpty(content)) {
            content = buildDissolutionNotifyContent(actionDesc.getMaster(), actionDesc.getAccountList());
        }
        AccountBean master = actionDesc.getMaster();
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            DataCache.getInstance().remove(getGroupFlag(groupId));
            IMContactEvent.DissolutionGroupEvent dissolutionGroupEvent = new IMContactEvent.DissolutionGroupEvent();
            dissolutionGroupEvent.setAccount(groupId);
            BusProvider.getInstance().post(dissolutionGroupEvent);
        } else {
            sendNotifyMessage(groupId, content);
        }
    }

    /**
     * 主动退出群组事件
     *
     * @param groupId
     * @param actionDesc
     */
    private void onQuitAction(String groupId, ActionDesc actionDesc) {
        String content = actionDesc.getMessage();
        if (TextUtils.isEmpty(content)) {
            content = buildQuitMessage(actionDesc.getMaster(), actionDesc.getAccountList());
        }

        AccountBean master = actionDesc.getMaster();
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            DataCache.getInstance().remove(getGroupFlag(groupId));
            IMContactEvent.DissolutionGroupEvent dissolutionGroupEvent = new IMContactEvent.DissolutionGroupEvent();
            dissolutionGroupEvent.setAccount(groupId);
            BusProvider.getInstance().post(dissolutionGroupEvent);
        } else {
            sendNotifyMessage(groupId, content);
        }
    }

    /**
     * 发送自定义通知事件
     *
     * @param to      消息接收者
     * @param message 消息内容
     */
    private void sendNotifyMessage(String to, String message) {
        TalkMessageBean talkMessageBean = new TalkMessageBean();
        talkMessageBean.setTo(to);
        talkMessageBean.setGroupMsg(true);
        talkMessageBean.setMessageType(ConstDef.MSG_TYPE_PRESENTATION);
        talkMessageBean.setContent(message);
        ImUiKit.getInstance().getComponent().proxyRepository().sendCustomMessage(talkMessageBean,
                new Subscriber<TalkMessageBean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("ERROR:Create group and send custom message failed.");
                    }

                    @Override
                    public void onNext(TalkMessageBean talkMessageBean) {
                        LogUtil.d("Create group and send custom message successfully.");
                    }
                });
    }

    /**
     * 你邀请成员列表信息
     *
     * @param master
     * @param memberList
     * @return
     */
    private String buildAddMemberListMessage(AccountBean master, List<AccountBean> memberList) {
        int size = memberList.size();
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < size - 1; i++) {
            AccountBean member = memberList.get(i);
            if (TextUtils.equals(mCurAccount, member.getAccount())) {
                continue;
            }
            builder.append(memberList.get(i).getDisplayName()).append("、");
        }
        builder.append(memberList.get(size - 1).getDisplayName());
        return String.format(CommonTool.getString(R.string.im_uikit_add_member1),
                builder.toString());
    }

    /**
     * 你移除成员列表信息
     *
     * @param master
     * @param memberList
     * @return
     */
    private String buildRemoveMemberListMessage(AccountBean master, List<AccountBean> memberList) {
        int size = memberList.size();
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < size - 1; i++) {
            AccountBean member = memberList.get(i);
            if (TextUtils.equals(mCurAccount, member.getAccount())) {
                continue;
            }
            builder.append(memberList.get(i).getDisplayName()).append("、");
        }
        builder.append(memberList.get(size - 1).getDisplayName());
        return String.format(CommonTool.getString(R.string.im_uikit_remove_member1),
                builder.toString());
    }

    /**
     * 创建群组自定义消息
     *
     * @param master
     * @param memberList
     * @return
     */
    private String buildCreateGroupMessage(AccountBean master, List<AccountBean> memberList) {
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            return buildAddMemberListMessage(master, memberList);
        } else {
            return String.format(CommonTool.getString(R.string.im_uikit_add_member2),
                    master.getDisplayName());
        }
    }

    /**
     * 添加群成员事件
     *
     * @param master     添加
     * @param memberList
     * @return
     */
    private String buildAddMemberMessage(AccountBean master, List<AccountBean> memberList) {
        boolean isCurMaster = TextUtils.equals(mCurAccount, master.getAccount());
        //群主添加，群成员收到消息
        if(isCurMaster) {
            return buildAddMemberListMessage(master, memberList);
        //群成员添加，群主收到消息
        }else {
            String  memberListName = "";
            AccountBean memberBean = memberList.get(0);
            StringBuffer sbMemberName = new StringBuffer();
            int memberListsize = memberList.size();
            if(memberListsize < 0) {
                LogUtil.d("MemberListSiez is Error");
                return "";
            }
            if (memberListsize >= 1) {
                for (int i = 0; i < memberListsize ; i++) {
                    memberBean = memberList.get(i);
                    if(!memberBean.getAccount().equals(master.getAccount())) {
                        memberListName = getMemberListName(master, memberBean);
                        sbMemberName = sbMemberName.append(memberListName).append("、");
                    }
                }
                //sbMemberName.append(memberList.get(memberListsize-1).getDisplayName());
                sbMemberName = sbMemberName.delete(sbMemberName.length()-1,sbMemberName.length());
            }
            LogUtil.d("buildAddMemberMessage == " + sbMemberName);
            return String.format(CommonTool.getString(R.string.im_uikit_add_member3),
                    master.getDisplayName(), sbMemberName.toString());
//            return sbMemberName.toString();
        }
    }

    private String getMemberListName(AccountBean master, AccountBean memberBean) {
        if (TextUtils.equals(mCurAccount, memberBean.getAccount())) {
            //你被 xxx 邀请加入群组
//            return String.format(CommonTool.getString(R.string.im_uikit_add_member2),
//                    master.getDisplayName());
            return CommonTool.getString(R.string.im_uikit_add_member_you);
        } else {
            //xxx 邀请 xxx 加入群组
//            return String.format(CommonTool.getString(R.string.im_uikit_add_member3),
//                    master.getDisplayName(), memberBean.getDisplayName());
            return memberBean.getDisplayName();
        }
    }

    /**
     * 移除群成员事件
     *
     * @param master
     * @param memberList
     * @return
     */
    private String buildRemoveMemberNotifyContent(AccountBean master, List<AccountBean> memberList) {
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            return buildRemoveMemberListMessage(master, memberList);
        } else {
            AccountBean operator = memberList.get(0);
            if (TextUtils.equals(mCurAccount, operator.getAccount())) {
                return String.format(CommonTool.getString(R.string.im_uikit_remove_member2),
                        master.getDisplayName());
            } else {
                return String.format(CommonTool.getString(R.string.im_uikit_remove_member3),
                        operator.getDisplayName(), master.getDisplayName());
            }
        }
    }

    /**
     * 解散群组事件
     *
     * @param master
     * @param accountList
     * @return
     */
    private String buildDissolutionNotifyContent(AccountBean master, List<AccountBean> accountList) {
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            return CommonTool.getString(R.string.im_uikit_dissolution1);
        } else {
            return String.format(CommonTool.getString(R.string.im_uikit_dissolution2), master.getDisplayName());
        }
    }

    /**
     * 退出群组事件
     *
     * @param master
     * @param accountList
     * @return
     */
    private String buildQuitMessage(AccountBean master, List<AccountBean> accountList) {
        if (TextUtils.equals(mCurAccount, master.getAccount())) {
            return CommonTool.getString(R.string.im_uikit_quit_group);
        } else {
            AccountBean opetator = accountList.get(0);
            return String.format(CommonTool.getString(R.string.im_uikit_quit_group2),
                    opetator.getDisplayName());
        }
    }
}
