package com.xdja.im.core.repository.impl.im;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.xdja.google.gson.Gson;
import com.xdja.im.base.di.DiConfig;
import com.xdja.im.base.di.scope.Scoped;
import com.xdja.im.base.executor.ThreadExecutor;
import com.xdja.im.base.executor.UIThread;
import com.xdja.im.common.cache.impl.ConfigCacheImpl;
import com.xdja.im.common.cache.interf.CardCache;
import com.xdja.im.common.cache.interf.ConfigCache;
import com.xdja.im.common.cache.interf.UserCache;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.custom.CustomMsgInfo;
import com.xdja.im.core.model.message.AtInfo;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.model.message.mapper.DataMapper;
import com.xdja.im.core.model.message.mapper.ValueConverter;
import com.xdja.im.core.push.PushControllerImp;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.core.utils.FilePathUtils;
import com.xdja.im.core.utils.IMFileTools;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.file.bean.FileListType;
import com.xdja.im.lib.filemanage.model.HistoryFileCategory;
import com.xdja.im.lib.filemanage.util.ActomaController;
import com.xdja.im.lib.filemanage.util.DateUtils;
import com.xdja.im.lib.filemanage.util.ModelGenerator;
import com.xdja.im.lib.share.model.WebPageInfo;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.file.FileUtils;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.imsdk.ImClient;
import com.xdja.imsdk.callback.CallbackFunction;
import com.xdja.imsdk.callback.IMFileInfoCallback;
import com.xdja.imsdk.callback.IMMessageCallback;
import com.xdja.imsdk.callback.IMSecurityCallback;
import com.xdja.imsdk.callback.IMSessionCallback;
import com.xdja.imsdk.constant.IMSessionType;
import com.xdja.imsdk.constant.ImSdkConfig;
import com.xdja.imsdk.constant.ImSdkConstant;
import com.xdja.imsdk.constant.ImSdkFileConstant;
import com.xdja.imsdk.constant.MsgPackType;
import com.xdja.imsdk.exception.ImSdkException;
import com.xdja.imsdk.model.IMFileInfo;
import com.xdja.imsdk.model.IMMessage;
import com.xdja.imsdk.model.IMSession;
import com.xdja.imsdk.model.InitParam;
import com.xdja.imsdk.model.body.IMFileBody;
import com.xdja.imsdk.model.body.IMMessageBody;
import com.xdja.imsdk.model.body.IMTextBody;
import com.xdja.imsdk.util.ToolUtils;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.inject.Inject;

import rx.Observable;
import rx.Subscriber;
import rx.functions.Func1;
import rx.functions.Func2;

/**
 * @Package: com.xdja.im.core.repository.impl.im
 * @Author: xdjaxa
 * @Creation: 2017-04-11 11:13
 * @Version V1.0
 * @Description:IM业务代理实现
 */
public class IMProxyImpl implements ProxyRepository {

    @Inject
    ThreadExecutor threadExecutor;
    @Inject
    UIThread uiThread;

    private Context context;
    private ImClient imClient;
    private CardCache cardCache;
    private ConfigCache configCache;
    private UserCache userCache;
    private DataMapper mapper;
    private CallbackFunction callbackFunction;
    private IMFileInfoCallback imFileInfoCallback;
    private IMMessageCallback imMessageCallback;
    private IMSessionCallback imSessionCallback;
    private IMSecurityCallback imSecurityCallback;

    @Inject
    public IMProxyImpl(@Scoped(DiConfig.CONTEXT_SCOPE_APP) Context context,
                       ImClient imClient,
                       CardCache cardCache,
                       ConfigCache configCache,
                       UserCache userCache,
                       DataMapper mapper,
                       CallbackFunction callbackFunction,
                       IMFileInfoCallback imFileInfoCallback,
                       IMMessageCallback imMessageCallback,
                       IMSessionCallback imSessionCallback,
                       IMSecurityCallback imSecurityCallback,
                       CommonTool commonTool) {
        this.context = context;
        this.imClient = imClient;
        this.cardCache = cardCache;
        this.configCache = configCache;
        this.userCache = userCache;
        this.mapper = mapper;
        this.callbackFunction = callbackFunction;
        this.imFileInfoCallback = imFileInfoCallback;
        this.imMessageCallback = imMessageCallback;
        this.imSessionCallback = imSessionCallback;
        this.imSecurityCallback = imSecurityCallback;
    }

    @Override
    public Observable<Integer> initIMProxy() {
        return Observable.just("")
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<InitParam>>() {
                    @Override
                    public Observable<InitParam> call(String s) {
                        InitParam initParam = new InitParam();
                        initParam.setAccount(userCache.get().getAccount());
                        initParam.setTicket(userCache.get().getTicket());
                        initParam.setTfcardId(cardCache.get().getCardId());
                        if (configCache.getDevice() != null) {
                            initParam.setdType(configCache.getDevice().getDeviceType());
                        } else {
                            initParam.setdType(ConfigCacheImpl.DEFAULT_DEVICE_TYPE);
                        }

                        initParam.setCallback(callbackFunction);
                        initParam.setSecurityCallback(imSecurityCallback);

                        HashMap<String, String> imProperty = new HashMap<>();
                        //设置配置文件相关信息
                        imProperty.put(ImSdkConfig.K_SERVER, configCache.getServer().getImUrl());
                        if (configCache.getServer() != null && configCache.getServer().getFastDfsUrl() != null) {
                            imProperty.put(ImSdkConfig.K_FILE_ADDR, configCache.getServer().getFastDfsUrl().getAddr());
                            //imProperty.put(ImSdkConfig.K_FILE_PORT, configCache.getServer().getFastDfsUrl().getPort());
                        } else {
                            LogUtil.d("InitFail", "configCache is null");
                        }
                        //设置为Https请求
                        imProperty.put(ImSdkConfig.K_STORE, String.valueOf(R.raw.truststore));
                        imProperty.put(ImSdkConfig.K_PATH, FilePathUtils.getSdkCacheDir());
                        imProperty.put(ImSdkConfig.K_SIZE, ImSdkConfig.V_SIZE);
                        //不进行加密
                        imProperty.put(ImSdkConfig.K_ENCRYPT, ImSdkConfig.V_FALSE);
                        //不支持闪信
                        imProperty.put(ImSdkConfig.K_BOMB, ImSdkConfig.V_FALSE);
                        //不支持消息状态
                        //imProperty.put(ImSdkConfig.K_STATE, ImSdkConfig.V_FALSE);
                        //push相关的配置
                        imProperty.put(ImSdkConfig.K_CLIENT, PushControllerImp.clientId);
                        imProperty.put(ImSdkConfig.K_TOPIC, PushControllerImp.subTopic);
                        if (configCache != null && configCache.getServer() != null){
                            imProperty.put(ImSdkConfig.K_USER_ID, configCache.getServer().getUserId());
                            imProperty.put(ImSdkConfig.K_USER_SECRET, configCache.getServer().getUserSecret());
                        }
                        initParam.setProperties(imProperty);

                        LogUtil.d("InitParam:" + initParam.toString());

                        return Observable.just(initParam);
                    }
                })
                .map(new Func1<InitParam, Integer>() {
                    @Override
                    public Integer call(InitParam param) {

                        //注册回调事件
                        imClient.RegisterIMSessionChangeListener(context, imSessionCallback);
                        imClient.RegisterIMMessageChangeListener(context, imMessageCallback);
                        imClient.RegisterIMFileInfoChangeListener(context, imFileInfoCallback);
                        return imClient.Init(param);
                    }
                })
                .observeOn(uiThread.getScheduler());

    }

    @Override
    public Observable<Integer> releaseIMProxy() {
        return Observable.just(imClient)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<ImClient, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(ImClient imClient) {
                        //反注册回调事件
                        imClient.UnregisterIMSessionChangeListener(context);
                        imClient.UnregisterIMMessageChangeListener(context);
                        imClient.UnregisterIMFileInfoChangeListener(context);
                        return Observable.just(imClient.Release(1));
                    }
                })
                .observeOn(uiThread.getScheduler());
    }

    @Override
    public Observable<Boolean> setProxyConfig(Map<String, String> param) {
        return Observable.just(param)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<Map<String, String>, Boolean>() {
                    @Override
                    public Boolean call(Map<String, String> param) {
                        imClient.SetConfig(param);
                        return true;
                    }
                })
                .observeOn(uiThread.getScheduler());
    }

    @Override
    public Observable<String> getProxyConfig(String key) {
        return Observable.just(key)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<String>>() {
                    @Override
                    public Observable<String> call(String key) {
                        try {
                            String config = imClient.GetConfig(key);
                            return Observable.just(config);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler());
    }

    @Override
    public void getSessionListBeans(final String begin, final int size, final int talkType,
                                    Subscriber<List<TalkSessionBean>> subscriber) {
        Observable.just("")
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<IMSession>>() {
                    @Override
                    public Observable<IMSession> call(String s) {
                        try {
                            //加载本地数据
                            List<IMSession> sessions = imClient.GetIMSessionList(begin, size);
                            return Observable.from(sessions);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .filter(new Func1<IMSession, Boolean>() {
                    @Override
                    public Boolean call(IMSession imSession) {
                        //过滤加载会话数据类型
                        if (imSession != null && talkType != ConstDef.CHAT_TYPE_DEFAULT) {
                            return imSession.getSessionType() == talkType;
                        }
                        return imSession != null;
                    }
                })
                .map(new Func1<IMSession, TalkSessionBean>() {
                    @Override
                    public TalkSessionBean call(final IMSession imSession) {
                        if (imSession == null) {
                            return null;
                        }
                        //数据类型转换
                        return mapper.mapTalkSessionBean(imSession);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .toSortedList()
                .subscribe(subscriber);
    }

    @Override
    public void deleteSession(String sessionFlag, Subscriber<Integer> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String sessionFlag) {

                        ArrayList<String> ids = new ArrayList<>();
                        ids.add(sessionFlag);

                        int code = imClient.DeleteIMSession(ids);
                        return Observable.just(code);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void getMessageList(final String talkId,
                               final int talkType,
                               final long begin,
                               final int size,
                               Subscriber<List<TalkMessageBean>> subscriber) {

        Observable.just("")
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(String s) {
                        try {
                            String talkFlag = ToolUtils.getSessionTag(talkId, talkType);
                            List<IMMessage> messages = imClient.GetIMMessageList(talkFlag, begin, size,false);
                            return Observable.from(messages);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .map(new Func1<IMMessage, TalkMessageBean>() {
                    @Override
                    public TalkMessageBean call(IMMessage message) {
                        if (message == null) {
                            LogUtil.w("message is null");
                            return null;
                        }
                        if (message.isFileIMMessage() && !message.isRecalled()) {
                            IMMessageBody body = message.getMessageBody();
                            if (body instanceof IMFileBody) {
                                IMFileBody fileBody = (IMFileBody) body;
                                if (fileBody.isImage() || fileBody.isPic() || fileBody.isMedia() || fileBody.isVideo()) {
                                    try {
                                        message = imClient.getIMMessageById(message.getIMMessageId());
                                    } catch (ImSdkException e) {
                                        LogUtil.e("", "Get im msg fail");
                                    }
                                }
                            }
                        }
                        return mapper.mapTalkMessageBean(message);
                    }
                })
                .toSortedList()
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void getTalkMessage(long msgId, Subscriber<TalkMessageBean> subscriber) {
        Observable.just(msgId)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<Long, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(Long msgId) {
                        try {
                            IMMessage imMessage = imClient.getIMMessageById(Long.valueOf(msgId));
                            TalkMessageBean talkMessageBean = mapper.mapTalkMessageBean(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            e.printStackTrace();
                        }
                        return Observable.error(new Exception("Get message info failed."));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void deleteMessages(List<Long> msgids, Subscriber<Integer> subscriber) {
        Observable.just(msgids)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<List<Long>, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(List<Long> msgIds) {
                        int code = imClient.DeleteIMMessage(msgIds);
                        if (code != 0) {
                            return Observable.error(new Exception("Delete msg failed, msgId:" +
                                    msgIds.toString() + ",code:" + code));
                        }
                        return Observable.just(code);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void getUnReadMsgCount(String talkId, Subscriber<Integer> subscriber) {
        Observable.just(talkId)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String talkId) {
                        try {
                            Map<String, Integer> map = imClient.GetRemindIMMessageCount(talkId);
                            return Observable.just(map.get(ImSdkConstant.NUM_NORMAL_NEW));
                        } catch (ImSdkException e) {
                            e.printStackTrace();
                        }
//                        return Observable.just(imClient.GetRemindIMMessageCount(talkId));
                        return null;
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public Observable<TalkMessageBean> getNoReadFirstNewMessage(final String flag) {
        return Observable.just(imClient)
                .flatMap(new Func1<ImClient, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(ImClient imClient) {
                        try {
                            IMMessage imMessage = imClient.GetFirstNewMessage(flag);
                            TalkMessageBean talkMessageBean = mapper.mapMessage(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                });
    }

    @Override
    public void clearUnReadMsgCount(String talkId, Subscriber<Integer> subscriber) {
        Observable.just(talkId)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String talkId) {
                        int i = imClient.SetRemind(talkId, ImSdkConstant.REMIND_CLEAR);
                        return Observable.just(i);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void clearAllData(Subscriber<Integer> subscriber) {
        Observable.just("")
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String s) {
                        int i = imClient.ClearAllLocalData();
                        return Observable.just(i);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void clearAllSessionData(String sessionFlag, Subscriber<Integer> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String sessionFlag) {
                        int i = imClient.ClearIMSessionAllIMMessage(sessionFlag);
                        return Observable.just(i);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void clearAllCacheData(Subscriber<Integer> subscriber) {
        Observable.just("")
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(String sessionFlag) {
                        int i = imClient.ClearCache();
                        return Observable.just(i);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void getAllUnReadMsgCount(Subscriber<Integer> subscriber) {
        Observable.just(imClient)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<ImClient, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(ImClient imClient) {

                        return Observable.just(imClient.GetAllRemindIMMessageCount());
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendMessage(TalkMessageBean talkMessageBean,
                            Subscriber<TalkMessageBean> subscriber) {

    }

    @Override
    public void reSendMessage(TalkMessageBean talkMessageBean,
                              Subscriber<Integer> subscriber) {
        //消息重发
        Observable.just(talkMessageBean.getId())
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<Long, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(Long aLong) {
                        return Observable.just(imClient.ResendIMMessage(aLong));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }


    @Override
    public void sendTextMessage(final @NonNull String content,
                                final @NonNull String to,
                                final boolean isShan,
                                final boolean isGroup,
                                Subscriber<TalkMessageBean> subscriber) {
        Observable.just(content)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(String s) {
                        IMMessage imMessage = new IMMessage();
                        imMessage.setCardId(cardCache.get().getCardId().toLowerCase());
                        imMessage.setTimeToLive(isShan ? 9000 : 0);
                        imMessage.setIMMessageTime(System.currentTimeMillis());
                        imMessage.setMessageBody(new IMTextBody(content));
                        imMessage.setFrom(userCache.get().getAccount());
                        imMessage.setTo(to);
                        imMessage.setType(1 + (isGroup ? 4 : 0) + (isShan ? 8 : 0));
                        return Observable.just(imMessage);
                    }
                })
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage imMessage) {
                        try {
                            imMessage = imClient.SendIMMessage(imMessage);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapTalkMessageBean(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendAtTextMessage(final @NonNull String content,
                                  final @NonNull String to,
                                  final boolean isShan,
                                  final boolean isGroup,
                                  final List<String> atAccount,
                                  Subscriber<TalkMessageBean> subscriber) {

        Observable.just(content)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(String s) {
                        IMMessage imMessage = new IMMessage();
                        imMessage.setCardId(cardCache.get().getCardId().toLowerCase());
                        imMessage.setTimeToLive(isShan ? 9000 : 0);
                        imMessage.setIMMessageTime(System.currentTimeMillis());
                        imMessage.setMessageBody(mapper.mapAtBody(new AtInfo(atAccount, content)));
                        imMessage.setFrom(userCache.get().getAccount());
                        imMessage.setTo(to);
                        imMessage.setType(MsgPackType.NORMAL_PG_AT + (isShan ? 8 : 0));
                        return Observable.just(imMessage);
                    }
                })
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage imMessage) {
                        try {
                            imMessage = imClient.SendIMMessage(imMessage);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapMessage(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void recallMessages(final long talkId, final int action,
                               Subscriber<Integer> subscriber) {
        Observable.just(imClient)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<ImClient, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(ImClient imClient) {
                        return Observable.just(imClient.RecallIMMessage(talkId, action));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendNotifyTextMessage(@NonNull final String content,
                                      @NonNull final String to,
                                      final boolean isGroup,
                                      Subscriber<TalkMessageBean> subscriber) {
        Observable
                .create(new Observable.OnSubscribe<IMMessage>() {
                    @Override
                    public void call(Subscriber<? super IMMessage> subscriber) {
                        IMMessage imMessage = new IMMessage();
                        imMessage.setCardId(cardCache.get().getCardId().toLowerCase());
                        imMessage.setIMMessageTime(System.currentTimeMillis());
                        imMessage.setMessageBody(new IMTextBody(content));
                        imMessage.setFrom(userCache.get().getAccount());
                        imMessage.setTo(to);
                        imMessage.setType(isGroup ? MsgPackType.NOTICE_PG_TEXT : MsgPackType.NOTICE_PP_TEXT);
                        subscriber.onNext(imMessage);
                    }
                })
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage imMessage) {
                        try {
                            imMessage = imClient.SendIMMessage(imMessage);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapTalkMessageBean(imMessage);

                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendCustomMessage(TalkMessageBean talkMessageBean, Subscriber<TalkMessageBean> subscriber) {
        Observable.just(talkMessageBean)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<TalkMessageBean, IMMessage>() {
                    @Override
                    public IMMessage call(TalkMessageBean talkMessageBean) {
                        return mapper.mapIMMessage(talkMessageBean);
                    }
                })
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage message) {
                        if (message == null) {
                            return Observable.error(new IllegalArgumentException("message is null."));
                        }
                        try {
                            IMMessage imMessage = imClient.IMMessageListAddCust("", message);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapTalkMessageBean(imMessage);

                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendFileMessage(@NonNull final String to,
                                final boolean isShan,
                                final boolean isGroup,
                                final List<FileInfo> fileInfoList,
                                Subscriber<TalkMessageBean> subscriber) {
        Observable.from(fileInfoList)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<FileInfo, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(FileInfo fileInfo) {
                        //数据类型转换
                        IMMessage imMessage = new IMMessage();
                        imMessage.setCardId(cardCache.get().getCardId().toLowerCase());
                        imMessage.setTimeToLive(isShan ? 9000 : 0);
                        imMessage.setIMMessageTime(System.currentTimeMillis());
                        imMessage.setFrom(userCache.get().getAccount());
                        imMessage.setTo(to);
                        imMessage.setMessageBody(mapper.mapFileBody(fileInfo));
                        imMessage.setType(2 + (isGroup ? 4 : 0) + (isShan ? 8 : 0));
                        return Observable.just(imMessage);
                    }
                })
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage imMessage) {
                        try {
                            imMessage = imClient.SendIMMessage(imMessage);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapTalkMessageBean(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public Observable<List<FileInfo>> getImageFileList(List<ImageFileInfo> pictureList, final int type) {
        return Observable.from(pictureList)
                .concatMap(new Func1<ImageFileInfo, Observable<FileInfo>>() {
                    @Override
                    public Observable<FileInfo> call(ImageFileInfo pictureInfo) {
                        //生成图片缩略图，图片等相关信息
                        String account = userCache.get().getAccount();
                        if (TextUtils.isEmpty(account)) {
                            return Observable.error(new NullPointerException("user account is NULL."));
                        }

                        Observable<FileInfo> fileInfo = null;
                        if (type == ConstDef.FILE_PICTURE) {
                            fileInfo = ModelGenerator.createPictureFileInfo(pictureInfo, account);
                        } else {
                            fileInfo = ModelGenerator.createImageFileInfo(pictureInfo, account);
                        }

                        //Observable<FileInfo> fileInfo = ModelGenerator.createImageFileInfo(pictureInfo, account);
                        if (fileInfo == null) {
                            return Observable.error(new Exception("create thumbnail failed."));
                        }
                        return fileInfo;
                    }
                }).toList();
    }

    @Override
    public void downloadFile(List<FileInfo> fileInfo, Subscriber<Integer> subscriber) {
        Observable.just(fileInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<List<FileInfo>, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(List<FileInfo> fileInfoList) {
                        List<IMFileInfo> files = new ArrayList<>();
                        for (int i = 0; i < fileInfoList.size(); i++) {
                            files.add(mapper.mapIMFileInfo(fileInfoList.get(i)));
                        }
                        int code = imClient.ReceiveFileStart(files);
                        return Observable.just(code);
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void pauseFileSending(FileInfo fileInfo, Subscriber<Integer> subscriber) {
        Observable.just(fileInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<FileInfo, IMFileInfo>() {
                    @Override
                    public IMFileInfo call(FileInfo fileInfo) {
                        if (fileInfo == null) {
                            return null;
                        }
                        return mapper.mapIMFileInfo(fileInfo);
                    }
                })
                .flatMap(new Func1<IMFileInfo, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(IMFileInfo imFileInfo) {
                        return Observable.just(imClient.SendFilePause(imFileInfo));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void resumeFileSend(FileInfo fileInfo, Subscriber<Integer> subscriber) {
        Observable.just(fileInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<FileInfo, IMFileInfo>() {
                    @Override
                    public IMFileInfo call(FileInfo fileInfo) {
                        if (fileInfo == null) {
                            return null;
                        }
                        return mapper.mapIMFileInfo(fileInfo);
                    }
                })
                .flatMap(new Func1<IMFileInfo, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(IMFileInfo file) {
                        return Observable.just(imClient.SendFileResume(file));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void pauseFileReceiving(FileInfo fileInfo, Subscriber<Integer> subscriber) {
        Observable.just(fileInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<FileInfo, IMFileInfo>() {
                    @Override
                    public IMFileInfo call(FileInfo fileInfo) {
                        if (fileInfo == null) {
                            return null;
                        }
                        return mapper.mapIMFileInfo(fileInfo);
                    }
                })
                .flatMap(new Func1<IMFileInfo, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(IMFileInfo file) {
                        return Observable.just(imClient.ReceiveFilePause(file));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void resumeFileReceive(FileInfo fileInfo, Subscriber<Integer> subscriber) {
        Observable.just(fileInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<FileInfo, IMFileInfo>() {
                    @Override
                    public IMFileInfo call(FileInfo fileInfo) {
                        if (fileInfo == null) {
                            return null;
                        }
                        return mapper.mapIMFileInfo(fileInfo);
                    }
                })
                .flatMap(new Func1<IMFileInfo, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(IMFileInfo file) {
                        return Observable.just(imClient.ReceiveFileResume(file));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void getSessionFileList(@NonNull String talkId,
                                   final int begin, final int size,
                                   Subscriber<List<TalkMessageBean>> subscriber) {
        Observable.just(talkId)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<String, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(String talkId) {
                        try {
                            return Observable.from(imClient.GetFileList(talkId, ImSdkFileConstant.FileGetType.CUR_ALL_MEDIA, begin, size));
                        } catch (ImSdkException e) {
                            e.printStackTrace();
                            return Observable.error(e);
                        }
                    }
                })
                .map(new Func1<IMMessage, TalkMessageBean>() {
                    @Override
                    public TalkMessageBean call(IMMessage imMessage) {
                        return mapper.mapTalkMessageBean(imMessage);
                    }
                })
                .filter(new Func1<TalkMessageBean, Boolean>() {
                    @Override
                    public Boolean call(TalkMessageBean talkMessageBean) {
                        //fixed by lll, 不加载缩略图失败的文件
                        FileInfo fileInfo = talkMessageBean.getFileInfo();
                        if (fileInfo == null || (!talkMessageBean.isMine() && fileInfo.getFileState() != ConstDef.DONE)) {
                            return false;
                        }
                        return true;
                    }
                })
                .toList()
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void queryRecentAndLocalFile(final FileListType tabType,
                                        Subscriber<Map<String, List<FileItem>>> subscriber) {
        Observable.just(tabType)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<FileListType, Observable<Map<String, List<FileItem>>>>() {
                    @Override
                    public Observable<Map<String, List<FileItem>>> call(FileListType fileListType) {
                        //获取文件类型，根据不同的类别进行区分
                        Map<String, List<FileItem>> localFileMap = new HashMap<>();
                        try {
                            switch (tabType) {
                                case VIDEO: //音视频
                                    localFileMap.putAll(IMFileTools.queryLocalAudios());
                                    localFileMap.putAll(IMFileTools.queryLocalVideos());
                                    break;
                                case PHOTO: //图片
                                    localFileMap.putAll(IMFileTools.queryLocalImages());
                                    break;
                                case DOCUMENT://文档
                                    localFileMap.putAll(IMFileTools.queryLocalDocuments());
                                    break;
                                case APK:   //应用
                                    localFileMap.putAll(IMFileTools.queryLocalApplication());
                                    break;
                                case OTHER: //其他
                                    localFileMap.putAll(IMFileTools.queryOtherFiles());
                                    break;
                                case RECENT:
                                    localFileMap.putAll(IMFileTools.getLastFileList(imClient.GetFileList("")));
                                    break;
                            }
                            return Observable.just(localFileMap);
                        } catch (Exception e) {
                            return Observable.error(new Throwable("Obtain msg error!"));
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void queryLocalFile(int tabType, Subscriber<Map<String, List<FileItem>>> subscriber) {
        Observable.just(tabType)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<Integer, Observable<Map<String, List<FileItem>>>>() {
                    @Override
                    public Observable<Map<String, List<FileItem>>> call(Integer fileListType) {
                        //获取文件类型，根据不同的类别进行区分
                        Map<String, List<FileItem>> localFileMap = new HashMap<>();
                        try {
                            switch (fileListType) {
                                case ConstDef.FILE_TYPE_VOICE: //音视频
                                    localFileMap.putAll(IMFileTools.queryLocalAudios());
                                    localFileMap.putAll(IMFileTools.queryLocalVideos());
                                    break;
                                case ConstDef.FILE_TYPE_PHOTO: //图片
                                    localFileMap.putAll(IMFileTools.queryLocalImages());
                                    break;
                                case ConstDef.FILE_TYPE_TXT://文档
                                    localFileMap.putAll(IMFileTools.queryLocalDocuments());
                                    break;
                                case ConstDef.FILE_TYPE_APK:   //应用
                                    localFileMap.putAll(IMFileTools.queryLocalApplication());
                                    break;
                                case ConstDef.FILE_TYPE_NORMAL: //其他
                                    localFileMap.putAll(IMFileTools.queryOtherFiles());
                                    break;
                            }
                            return Observable.just(localFileMap);
                        } catch (Exception e) {
                            return Observable.error(new Throwable("Obtain msg error!"));
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public Observable<Map<HistoryFileCategory, List<TalkMessageBean>>> getAllHistoryFileInfoWithTalkId(final String talkId, final int type) {
        return Observable.just(imClient)
                .flatMap(new Func1<ImClient, Observable<Map<HistoryFileCategory, List<TalkMessageBean>>>>() {
                    @Override
                    public Observable<Map<HistoryFileCategory, List<TalkMessageBean>>> call(ImClient imClient) {
                        Map<HistoryFileCategory, List<TalkMessageBean>> resultMaps = new LinkedHashMap<>();

                        try {
                            ImSdkFileConstant.FileGetType getType = ValueConverter.changeObtainFileType(type);
                            List<IMMessage> messages = imClient.GetFileList(talkId, getType, 0, 0);

                            int length = messages.size();
                            //fix bug 12348 by zya ,20170504
                            if (length == 0) {
                                return Observable.just(resultMaps);
                            }//end by zya
                            List<TalkMessageBean> valueLists;
                            for (int i = 0; i < length; i++) {
                                //消息排序方式是由近到远
                                IMMessage message = messages.get(i);
                                TalkMessageBean messageBean = mapper.mapMessage(message);

                                //add by zya 20170103 ,download progress cache
//                                if(message.isFileIMMessage()){
//                                    FileInfo fInfo = messageBean.getFileInfo();
//                                    if(fInfo.getTranslatePercent() == 0 && userCache.containKey(messageBean.getId())){
//                                        fInfo.setTranslatePercent(userCache.getProgress(messageBean.getId()));
//                                    }
//                                }//end by zya

                                //key值获取
                                HistoryFileCategory category = getDataCategory(message.getIMMessageTime());

                                if (resultMaps.containsKey(category)) {
                                    valueLists = resultMaps.get(category);
                                } else {
                                    valueLists = new ArrayList<>();
                                    category.setTime(messageBean.getShowTime());
                                }
                                messageBean.setCategoryId(category.getCategoryId());
                                valueLists.add(messageBean);
                                resultMaps.put(category, valueLists);
                            }
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                        return Observable.just(resultMaps);
                    }
                });
    }

    //add by zya,fix bug 12135 ,20170503
    private HistoryFileCategory getDataCategory(long messageTime) {
        final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd", Locale.getDefault());
        Calendar calendar = Calendar.getInstance(Locale.getDefault());
        calendar.setTimeInMillis(messageTime);
        int msgYear = calendar.get(Calendar.YEAR);
        int msgDay = calendar.get(Calendar.DAY_OF_YEAR);
        calendar.setTimeInMillis(DateUtils.obtainCurrentTime());
        int curYear = calendar.get(Calendar.YEAR);
        int curDay = calendar.get(Calendar.DAY_OF_YEAR);
        int timeDistance = 0;
        if (curYear != msgYear) {
            for (int i = Math.min(curYear, msgYear); i < Math.max(curYear, msgYear); i++) {
                if (i % 4 == 0 && i % 100 != 0 || i % 400 == 0) {
                    timeDistance += 366;
                } else {
                    timeDistance += 365;
                }
            }
            timeDistance = timeDistance - msgDay + curDay;
        } else {
            timeDistance = curDay - msgDay;
        }
        String categoryId = "";
        if (timeDistance == 0) {
            categoryId = ActomaController.getAppConfigContext().getString(R.string.im_uikit_today);
        } else if (timeDistance == 1) {
            categoryId = ActomaController.getAppConfigContext().getString(R.string.im_uikit_yesterday);
        } else if (timeDistance > 1 && timeDistance <= 7) {
            categoryId = ActomaController.getAppConfigContext().getString(R.string.im_uikit_one_week);
        } else if (timeDistance > 7 && timeDistance <= 30) {
            categoryId = ActomaController.getAppConfigContext().getString(R.string.im_uikit_week_ago);
        } else if (timeDistance > 30) {
            categoryId = ActomaController.getAppConfigContext().getString(R.string.im_uikit_month_ago);
        } else {
            categoryId = dateFormat.format(messageTime);
        }
        HistoryFileCategory dataCategory = new HistoryFileCategory();
        dataCategory.setCategoryId(categoryId);
        return dataCategory;
    }

    @Override
    public void changeMessageState(final TalkMessageBean talkMessageBean,
                                   @ConstDef.MsgState int mState,
                                   Subscriber<Integer> subscriber) {

        Observable.just(mState)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<Integer, Integer>() {
                    @Override
                    public Integer call(Integer integer) {
                        return ValueConverter.talkMsgStateConvert(integer);
                    }
                })
                .flatMap(new Func1<Integer, Observable<Integer>>() {
                    @Override
                    public Observable<Integer> call(Integer msgState) {
                        if (talkMessageBean == null) {
                            return Observable.error(new Exception(""));
                        }

                        IMMessage message = mapper.mapIMMessage(talkMessageBean);
                        return Observable.just(imClient.IMMessageStateChange(message, msgState));
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void saveFile(File srcFile, final String savePath, final String fileName, Subscriber<String> subscriber) {
        Observable.just(srcFile)
                .subscribeOn(threadExecutor.getScheduler())
                .map(new Func1<File, String>() {
                    @Override
                    public String call(File srcFile) {
                        //保存图片
                        boolean ret = FileUtils.saveFile(srcFile, savePath, fileName);
                        if (!ret) {
                            Observable.error(new IOException("Save file failed."));
                        }
                        return savePath + fileName;
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void saveCustomIMMessage(CustomMsgInfo msginfo, Subscriber<IMMessage> subscriber) {
        Observable
                .zip(Observable.just(msginfo)
                                .subscribeOn(threadExecutor.getScheduler())
                                .map(new Func1<CustomMsgInfo, String>() {
                                    @Override
                                    public String call(CustomMsgInfo msgInfo) {
                                        int type = msgInfo.getType();
                                        int sessionType = IMSessionType.SESSION_SINGLE;
                                        if (type == ConstDef.MEETING_P2P || type == ConstDef.CALLBACK_P2P || type == ConstDef.VIDEO_CHAT) {
                                            sessionType = IMSessionType.SESSION_SINGLE;
                                        } else if (type == ConstDef.MEETING_P2G || type == ConstDef.CALLBACK_P2G) {
                                            sessionType = IMSessionType.SESSION_GROUP;
                                        }
                                        String sessiongTag = "";
                                        if (userCache != null) {
                                            if (userCache.isMine(msgInfo.getFromAtNo())) {
                                                sessiongTag = ToolUtils.getSessionTag(msgInfo.getToAtNo(), sessionType);
                                            } else {
                                                if (sessionType == IMSessionType.SESSION_GROUP) {
                                                    sessiongTag = ToolUtils.getSessionTag(msgInfo.getToAtNo(), sessionType);
                                                } else {
                                                    sessiongTag = ToolUtils.getSessionTag(msgInfo.getFromAtNo(), sessionType);
                                                }
                                            }
                                        }

                                        return sessiongTag;
                                    }
                                }),
                        Observable.just(msginfo)
                                .subscribeOn(threadExecutor.getScheduler())
                                .map(new Func1<CustomMsgInfo, IMMessage>() {
                                    @Override
                                    public IMMessage call(CustomMsgInfo msgInfo) {
                                        IMMessage message = new IMMessage();
                                        Gson gson = new Gson();
                                        String info = gson.toJson(msgInfo);
                                        IMTextBody body = new IMTextBody(info);
                                        message.setFrom(msgInfo.getFromAtNo());
                                        message.setTo(msgInfo.getToAtNo());
                                        int type = msgInfo.getType();
                                        if (type == ConstDef.MEETING_P2P || type == ConstDef.CALLBACK_P2P || type == ConstDef.VIDEO_CHAT) {
                                            message.setType(MsgPackType.CUSTOM_PP_TEXT);
                                        } else if (type == ConstDef.MEETING_P2G || type == ConstDef.CALLBACK_P2G) {
                                            message.setType(MsgPackType.CUSTOM_PG_TEXT);
                                        }
                                        if (!userCache.get().getAccount().equals(msgInfo.getFromAtNo()) && (msgInfo.getState() == ConstDef.SENDER_CANCEL
                                                || msgInfo.getState() == ConstDef.NO_RESPONSE)){
                                            message.setState(1);
                                        } else {
                                            message.setState(0);
                                        }
                                        message.setMessageBody(body);
                                        return message;
                                    }
                                }),
                        new Func2<String, IMMessage, IMMessage>() {
                            @Override
                            public IMMessage call(String session, IMMessage imMessage) {

                                try {
                                    return imClient.IMMessageListAddCust(session, imMessage);
                                } catch (ImSdkException e) {
                                    e.printStackTrace();
                                }
                                return null;
                            }
                        }).observeOn(uiThread.getScheduler())
                .subscribe(subscriber);
    }

    @Override
    public void sendWebMessage(@NonNull final String to,
                               final boolean isShan,
                               final boolean isGroup,
                               final WebPageInfo webPageInfo,
                               Subscriber<TalkMessageBean> subscriber) {
        Observable.just(webPageInfo)
                .subscribeOn(threadExecutor.getScheduler())
                .flatMap(new Func1<WebPageInfo, Observable<IMMessage>>() {
                    @Override
                    public Observable<IMMessage> call(WebPageInfo webPageInfo) {
                        IMMessage imMessage = new IMMessage();
                        imMessage.setCardId(cardCache.get().getCardId().toLowerCase());
                        imMessage.setTimeToLive(isShan ? 9000 : 0);
                        imMessage.setIMMessageTime(System.currentTimeMillis());
                        imMessage.setFrom(userCache.get().getAccount());
                        imMessage.setTo(to);
                        imMessage.setMessageBody(mapper.mapWebBody(webPageInfo));
                        imMessage.setType(64 + (isGroup ? 4 : 0) + (isShan ? 8 : 0));
                        return Observable.just(imMessage);
                    }
                })
                .flatMap(new Func1<IMMessage, Observable<TalkMessageBean>>() {
                    @Override
                    public Observable<TalkMessageBean> call(IMMessage imMessage) {
                        try {
                            imMessage = imClient.SendIMMessage(imMessage);
                            TalkMessageBean talkMessageBean
                                    = mapper.mapTalkMessageBean(imMessage);
                            return Observable.just(talkMessageBean);
                        } catch (ImSdkException e) {
                            return Observable.error(e);
                        }
                    }
                })
                .observeOn(uiThread.getScheduler())
                .subscribe(subscriber);

    }
}
