package com.xdja.im.lib.audio.player;

import android.content.res.AssetFileDescriptor;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.text.TextUtils;

import com.xdja.im.lib.audio.manager.VoicePlayState;
import com.xdja.im.lib.filemanage.util.ActomaController;
import com.xdja.im.lib.filemanage.util.UniversalUtil;
import com.xdja.im.lib.video.utils.AudioFunctions;
import com.xdja.im.uikit.action.media.MediaManger;
import com.xdja.im.uikit.utils.IoUtils;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

/**
 * @Package: com.xdja.im.lib.audio.player
 * @Author: xdjaxa
 * @Creation: 2017-04-24 11:58
 * @Version V1.0
 * @Description: 语音播放管理类
 */
public class AudioPlayerManager {

    /**
     * 媒体播放器
     */
    private MediaPlayer mMediaPlayer;
    /**
     * 当前播放媒体文件路径
     */
    private String mCurPlayPath;
    /**
     * 播放相关参数信息
     */
    private Object[] mArgs;

    private static class SingletonInstance {
        private static final AudioPlayerManager INSTANCE = new AudioPlayerManager();
    }

    public static AudioPlayerManager getInstance() {
        return SingletonInstance.INSTANCE;
    }

    private AudioPlayerManager() {
    }

    /**
     * 播放对应的音频文件
     *
     * @param filePath 文件路径
     */
    public void startToPlay(String filePath, Object... args) {

        synchronized (this){
            LogUtil.d("Start play voice " + filePath);
            if (!checkNullForPath(filePath)) {
                return;
            }
//            if (mMediaPlayer == null) {
//                mMediaPlayer = new MediaPlayer();
////                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
//                mMediaPlayer.setOnErrorListener(errorListener);
//                mMediaPlayer.setOnCompletionListener(completionListener);
//            } else {
//                if (mMediaPlayer.isPlaying()) {
//                    stopToPlay(VoicePlayState.STOP, args);
//                }
//                mMediaPlayer.reset();
//            }

            if (mMediaPlayer != null) {
                try {
                    if (mMediaPlayer.isPlaying()) {
                        //[S]modify by lll@xdja.com fixed bug 6859 2016/12/9
                        stopToPlay(VoicePlayState.STOP,mArgs);
                        //[E]modify by lll@xdja.com fixed bug 6859 2016/12/9
                    } else {
                        mMediaPlayer.release();
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                mMediaPlayer = null;
            }
            mMediaPlayer = new MediaPlayer();

            ///播放模式设置
            if (MediaManger.getInstance().isWakeAcquire()) {
                MediaManger.getInstance().setReceiverMode(true);
            }
            if (MediaManger.getInstance().isReceiverMode()) {
                MediaManger.getInstance().setReceiverModeOn();
                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
            } else {
                MediaManger.getInstance().setReceiverModeOff();
                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
            }
            //播放模式设置结束
            mMediaPlayer.setOnCompletionListener(completionListener);

            mMediaPlayer.setOnErrorListener(errorListener);
            if (TextUtils.equals(mCurPlayPath, filePath)) {
                stopToPlay(VoicePlayState.STOP, args);
                AudioFunctions.muteAudioFocus(false);
                return;
            }
            AudioFunctions.muteAudioFocus(true);
            FileInputStream fis = null;
            try {
                File file = new File(filePath);
                fis = new FileInputStream(file);
                mMediaPlayer.setDataSource(fis.getFD());
                mMediaPlayer.prepare();
                mMediaPlayer.start();
                mCurPlayPath = filePath;
                this.mArgs = args;
                AudioListenerManager.getInstance().playStateChange(
                        mCurPlayPath, VoicePlayState.PLAYING, args);
            } catch (Exception e) {
                LogUtil.e("ERROR:Play voice failed. " + e.getMessage());
                stopToPlay(VoicePlayState.ERROR, args);
            } finally {
                IoUtils.closeIO(fis);
            }
        }
    }


    /**
     * 重新开始播放
     */
    public void startToRePlay() {
        int duration;
        if (mMediaPlayer != null && mMediaPlayer.isPlaying()) {
            duration = mMediaPlayer.getCurrentPosition();
            //只有在播放时，才进行重播
            startToPlay(mCurPlayPath, duration , mArgs);
        }
    }


    /**
     * 开始重新播放
     * @param filePath
     * @param duration
     */
    public void startToPlay(String filePath, final int duration , Object... args) {
        // 在开始播放之前，先发送广播关闭上一个播放动画
            // 如果正在播放，则先停止播放
            if (mMediaPlayer != null) {
                try {
                    if (mMediaPlayer.isPlaying()) {
                        //[S]modify by lll@xdja.com fixed bug 6859 2016/12/9
                        if (duration > 0) { //重播（语音闪信不应该销毁）
                            stopToPlay(VoicePlayState.ERROR,args);
                        } else {
                            stopToPlay(VoicePlayState.STOP,args);
                        }
                        //[E]modify by lll@xdja.com fixed bug 6859 2016/12/9
                    } else {
                        mMediaPlayer.release();
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                mMediaPlayer = null;
            }
            mMediaPlayer = new MediaPlayer();

        ///播放模式设置
        if (MediaManger.getInstance().isWakeAcquire()) {
            MediaManger.getInstance().setReceiverMode(true);
        }
        if (MediaManger.getInstance().isReceiverMode()) {
            MediaManger.getInstance().setReceiverModeOn();
            mMediaPlayer.setAudioStreamType(AudioManager.STREAM_VOICE_CALL);
        } else {
            MediaManger.getInstance().setReceiverModeOff();
            mMediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
        }
            //播放模式设置结束
            mMediaPlayer.setOnCompletionListener(completionListener);

            mMediaPlayer.setOnErrorListener(errorListener);

            //如果正在播放当前消息，停止播放
            if (TextUtils.equals(mCurPlayPath, filePath)) {
                stopPlay();
                AudioFunctions.muteAudioFocus(false);
                MediaManger.getInstance().restoreAudioMode();
                return;
            }
            AudioFunctions.muteAudioFocus(true);
            FileInputStream fileInputStream = null;
            try {
                File file = new File(filePath);
                fileInputStream = new FileInputStream(file);
                mMediaPlayer.setDataSource(fileInputStream.getFD());
                mMediaPlayer.prepare();
                if (duration > 0) {
                    mMediaPlayer.seekTo(duration);
                }
                mMediaPlayer.start();
                mCurPlayPath = filePath;
                mArgs = args;
                AudioListenerManager.getInstance().playStateChange(
                        mCurPlayPath, VoicePlayState.PLAYING, args);
            } catch (Exception e) {
                e.printStackTrace();
                stopToPlay(VoicePlayState.ERROR,args);
                MediaManger.getInstance().restoreAudioMode();
            } finally {
                if (fileInputStream != null) {
                    IoUtils.closeIO(fileInputStream);
                }
            }
    }

    /**
     * 对文件信息进行必要的校验
     *
     * @param filePath
     * @return
     */
    private boolean checkNullForPath(String filePath) {
        if (filePath == null || TextUtils.isEmpty(filePath)) {
            return false;
        }

        File file = new File(filePath);
        return file.exists();
    }

    /**
     * 停止播放
     *
     * @param state 播放状态,因为某种原因停止播放
     */
    private synchronized void stopToPlay(VoicePlayState state, Object... args) {
        LogUtil.d("Stop play voice " + mCurPlayPath);
        if (mMediaPlayer != null) {
            try {
                if (mMediaPlayer.isPlaying()) {
                    mMediaPlayer.stop();
                }
                mMediaPlayer.release();
            } catch (IllegalStateException e) {
                e.printStackTrace();
            }
            mMediaPlayer = null;
            AudioFunctions.muteAudioFocus(false);
        }

        mCurPlayPath = "";
        this.mArgs = null;
        AudioListenerManager.getInstance().playStateChange(
                mCurPlayPath,
                state,
                args);
    }

    public void stopPlay() {
        stopToPlay(VoicePlayState.STOP, mArgs);
    }

    /**
     * 释放MediaPlayer,调用方式在每个页面销毁的时候
     * 尽量释放掉不用的MediaPlayer
     */
    public void releasePlayer() {
        stopToPlay(VoicePlayState.STOP, mArgs);
        if (mMediaPlayer != null) {
            mMediaPlayer.release();
            mMediaPlayer = null;
        }
    }

    public String getCurrentPlayPath() {
        return mCurPlayPath;
    }

    public Object[] getArgs() {
        return mArgs;
    }

    /**
     * 判断当前是否在播放
     *
     * @return
     */
    public boolean isPlaying() {
        return mMediaPlayer != null
                && mMediaPlayer.isPlaying()
                && !TextUtils.isEmpty(mCurPlayPath)
                && mArgs != null && mArgs.length > 0;
    }

    /**
     * 播放出错的监听事件
     */
    private MediaPlayer.OnErrorListener errorListener = new MediaPlayer.OnErrorListener() {
        @Override
        public boolean onError(MediaPlayer mp, int what, int extra) {
            stopToPlay(VoicePlayState.ERROR, mArgs);
            return false;
        }
    };

    /**
     * 播放完成的监听
     */
    private MediaPlayer.OnCompletionListener completionListener = new MediaPlayer.OnCompletionListener() {
        @Override
        public void onCompletion(MediaPlayer mp) {
            stopToPlay(VoicePlayState.COMPLETION, mArgs);
            MediaManger.getInstance().restoreAudioMode();
        }
    };
}
