package com.xdja.im.lib.video.widget;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.Handler;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.SurfaceView;
import android.view.View;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.xdja.im.lib.video.palyer.VideoPlayState;
import com.xdja.im.lib.video.palyer.VideoPlayerHelper;
import com.xdja.im.lib.video.palyer.onStateChangedListener;
import com.xdja.im.uikit.R;

import java.lang.ref.WeakReference;

/**
 * @Package: com.xdja.im.lib.video.widget
 * @Author: xdjaxa
 * @Creation: 2017-05-16 10:42
 * @Version V1.0
 * @Description:重写视频播放界面
 */
public class VideoPlayView extends RelativeLayout {

    /**
     * 播放界面
     */
    private SurfaceView mSurfaceView;
    /**
     * 底部控件
     */
    private LinearLayout mControllerLayout;
    /**
     * 播放暂停按钮
     */
    private CheckBox mPlayChx;
    /**
     * 播放视频时长
     */
    private TextView mPlayTimeTv;
    /**
     * 视频时长控件
     */
    private TextView mDurationTv;
    /**
     * 播放进度
     */
    private ProgressBar mPlayPBar;
    /**
     * 视频播放帮助类
     */
    private VideoPlayerHelper mHelper;
    /**
     * 视频播放回调
     */
    private VideoPlayView.IVideoPlayCallback mPlayCallback;
    /**
     * 进度通知Handler
     */
    private ProgressHandle mHandler;
    /**
     * 时长
     */
    private long mDuration = 0;

    public VideoPlayView(Context context) {
        super(context);
        init();
    }

    public VideoPlayView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public VideoPlayView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    public void setVideoPlayCallback(VideoPlayView.IVideoPlayCallback playCallback) {
        this.mPlayCallback = playCallback;
    }

    private void init() {
        initView();
        setListener();
        mHandler = new ProgressHandle(new WeakReference<>(getContext()));
    }

    private  static class ProgressHandle extends Handler {
        WeakReference<Context> mReference;

        public ProgressHandle(WeakReference<Context> reference) {
            this.mReference = reference;
        }
    }

    private void initView() {
        View.inflate(getContext(), R.layout.im_uikit_layout_video_player, this);
        mSurfaceView = (SurfaceView) findViewById(R.id.view_surface);
        mControllerLayout = (LinearLayout) findViewById(R.id.layout_controller);
        mPlayChx = (CheckBox) findViewById(R.id.chx_play_pause);
        mPlayTimeTv = (TextView) findViewById(R.id.tv_play_time);
        mDurationTv = (TextView) findViewById(R.id.tv_video_duration);
        mPlayPBar = (ProgressBar) findViewById(R.id.pbar_progress);

        mHelper = new VideoPlayerHelper(mSurfaceView.getHolder());

        mPlayTimeTv.setText(String.format(getResources().getString(R.string.im_uikit_video_duration), 0, 0));
        mDurationTv.setText(String.format(getResources().getString(R.string.im_uikit_video_duration),
                mDuration / 60,
                mDuration % 60));
    }

    private void setListener() {
        mHelper.setOnStateChangedListener(new onStateChangedListener() {
            @Override
            public void onStateChanged(VideoPlayState state) {
                if (state == VideoPlayState.COMPLETION) {
                    removeAllCallbacks();
                    playCompleted();
                } else if (state == VideoPlayState.ERROR ||
                        state == VideoPlayState.EXCEPTION) {
                    removeAllCallbacks();
                }
            }
        });
        mSurfaceView.setOnTouchListener(new OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (mControllerLayout.getVisibility() == GONE) {
                    mControllerLayout.setVisibility(VISIBLE);
                    mHandler.postDelayed(controllerRunnable, 2 * 1000);
                }
                return false;
            }
        });
        mPlayChx.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
                if (isChecked) {
                    pause();
                } else {
                    resume();
                }
            }
        });
    }

    private void removeAllCallbacks() {
        mHandler.removeCallbacks(controllerRunnable);
        mHandler.removeCallbacks(progressRunnable);
    }

    private Runnable controllerRunnable = new Runnable() {
        @Override
        public void run() {
            mControllerLayout.setVisibility(GONE);
        }
    };

    private Runnable progressRunnable = new Runnable() {
        @Override
        public void run() {
            if (mDuration == 0) {
                return;
            }
            long curPos = (long) (mHelper.getCurrentPosition() * 1.0f / 1000);
            if (curPos < 0) curPos = 0;
            mPlayTimeTv.setText(String.format(getResources().getString(R.string.im_uikit_video_duration),
                    curPos / 60,
                    curPos % 60));

            int progress = (int) (curPos * 100.0f / mDuration);
            mPlayPBar.setProgress(progress);
            mHandler.postDelayed(progressRunnable, 500);
        }
    };

    /**
     * 播放完成是回调
     */
    private void playCompleted() {
        mPlayPBar.setProgress(100);
        mPlayTimeTv.setText(String.format(getResources().getString(R.string.im_uikit_video_duration),
                mDuration / 60,
                mDuration % 60));
        mPlayChx.setChecked(false);
        if (mPlayCallback != null) {
            mPlayCallback.completeListener();
        }

    }

    public void setDuration(long duration) {
        mDuration = duration;
        mDurationTv.setText(String.format(getResources().getString(R.string.im_uikit_video_duration),
                mDuration / 60,
                mDuration % 60));
    }

    public VideoPlayState getVideoPlayState() {
        return mHelper.getPlayState();
    }

    public void play(String filePath) {
        mControllerLayout.setVisibility(VISIBLE);
        mPlayPBar.setProgress(0);
        mHandler.postDelayed(progressRunnable, 500);

        mHelper.play(filePath, false);
    }

    public void pause() {
        mHelper.pause();
        removeAllCallbacks();
    }

    public void resume() {
        mHelper.resume();
        mHandler.postDelayed(progressRunnable, 500);
    }

    public void stop() {
        mHelper.stop();
        removeAllCallbacks();
    }

    public void destroy() {
        mHelper.destroy();
        removeAllCallbacks();
    }

    public void reset() {
        mHelper.reset();
        removeAllCallbacks();
    }

    public boolean isPlaying() {
        return mHelper.isPlaying();
    }

    /**
     * 项目名称：短视频             <br>
     * 接口描述  ：短视频播发回调接口     <br>
     * 创建时间：2017/1/28        <br>
     * 修改记录：                 <br>
     *
     * @author jyg@xdja.com   <br>
     */
    public interface IVideoPlayCallback {

        /**
         * 视频播放完成回调
         */
        void prepareListener();

        /**
         * 播放视频进度回调
         *
         * @param pro 当前进度
         */
        void progressListener(int pro);

        /**
         * 视频播放完成回调
         */
        void completeListener();

        /**
         * 视频播放暂停回调
         */
        void pauseListener(int currentPos);

        /**
         * 视频播放恢复回调
         */
        void resumeListener(int currentPos);

        /**
         * surfaceView准备完毕
         */
        void surfacePrepareCallback();

        /**
         * 视频播放恢复回调
         */
        void surfaceDestroyListener(Bitmap currentPosBitmap);

    }
}
