package com.xdja.im.uikit.ui.view;

import android.support.annotation.NonNull;
import android.view.View;
import android.widget.Button;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.action.options.ImOption;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.dialog.BaseListDialog;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;

/**
 * @Package: com.xdja.im.uikit.ui
 * @Author: xdjaxa
 * @Creation: 2017-04-20 14:38
 * @Version V1.0
 * @Description:聊天栏目消息发送基类
 */
public class BaseSendItem extends BaseChatItem {

    /**
     * 内容布局
     */
    protected RelativeLayout mContentLayout;
    /**
     * 消息状态
     */
    private TextView mStateTv;
    /**
     * 失败重发按钮
     */
    private Button mResendBtn;
    /**
     * 发送进度
     */
    private ProgressBar mSendingPBar;

    @Override
    protected void injectView() {
        super.injectView();

        mStateTv = (TextView) findViewById(R.id.tv_state);
        mResendBtn = (Button) findViewById(R.id.btn_resend);
        mSendingPBar = (ProgressBar) findViewById(R.id.pbar_sending);
        mContentLayout = (RelativeLayout) findViewById(R.id.layout_message_container);

        //不显示消息状态
        //mStateTv.setVisibility(View.GONE);

        mContentLayout.setOnLongClickListener(new View.OnLongClickListener() {
            @Override
            public boolean onLongClick(View v) {
                return showPopDialog();
            }
        });
        mResendBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mResendBtn.setVisibility(View.GONE);
                mSendingPBar.setVisibility(View.VISIBLE);
                getPresenter().reSendMessage(curPosition);
            }
        });
        mPopDialog.setOnItemClickListener(new BaseListDialog.onItemClickListener() {
            @Override
            public void onItemClick(View view, ImOption option, int itemPosition) {
                dismissPopDialog();
                switch (option.getOptionType()) {
                    case Constant.ITEM_CHAT_DELETE:  //删除消息
                        getPresenter().deleteItem(itemPosition);
                        break;
                    case Constant.ITEM_CHAT_COPY:    //复制消息
                        getPresenter().copyMessage(itemPosition);
                        break;
                    case Constant.ITEM_CHAT_RECALL:  //撤回消息
                        getPresenter().recallMessage(itemPosition);
                        break;
                    case Constant.ITEM_CHAT_RESEND:  //消息重发
                        getPresenter().reSendMessage(itemPosition);
                        break;
                    case Constant.ITEM_CHAT_FORWARD: //消息转发
                        getPresenter().forwardMessage(itemPosition);
                        break;
                }
            }
        });
    }

    @Override
    public void bindDataSource(int position, final @NonNull TalkMessageBean dataSource) {
        super.bindDataSource(position, dataSource);

        if (dataSource == null) {
            LogUtil.e("Action:set message state, reason: dataSource is null.");
            return;
        }


        //设置消息重发状态
        if (dataSource.getMessageState() == ConstDef.STATE_SEND_FAILED) {
            mResendBtn.setVisibility(View.VISIBLE);
            mSendingPBar.setVisibility(View.GONE);
        }

        //消息状态
        @ConstDef.MsgState
        int messageState = dataSource.getMessageState();
        mStateTv.setText(ConstDef.mapMsgState(messageState));

        //群组暂不支持消息状态
        if (dataSource.isGroupMsg()) {
            mStateTv.setVisibility(View.GONE);
        }

        //发送进度
        if (messageState == ConstDef.STATE_SENDING) {
            setProgressBarVisible(true);
        } else {
            setProgressBarVisible(false);
        }

        //设置消息重发状态
        if (messageState == ConstDef.STATE_SEND_FAILED) {
            mResendBtn.setVisibility(View.VISIBLE);
            mSendingPBar.setVisibility(View.GONE);

            if (dataSource.getFailCode() == Constant.FAIL_FRIEND) {
                mStateTv.setVisibility(View.VISIBLE);
                mStateTv.setText(dataSource.isGroupMsg() ?
                        R.string.im_uikit_not_member :
                        R.string.im_uikit_not_friend);
            }
        } else {
            mResendBtn.setVisibility(View.GONE);
        }

        //设置闪信背景
        setSendLayoutBackground(dataSource.isBomb());
    }

    /**
     * 发送进度条是否可见
     *
     * @param bVisible
     */
    private void setProgressBarVisible(boolean bVisible) {
        if (bVisible) {
            mSendingPBar.setVisibility(View.VISIBLE);
        } else {
            mSendingPBar.setVisibility(View.GONE);
        }
    }

    /**
     * 设置发送方闪信的背景和字体颜色
     *
     * @param isLimit
     */
    private void setSendLayoutBackground(boolean isLimit) {
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_PHOTO ||
                dataSource.getMessageType() == ConstDef.MSG_TYPE_VIDEO) {
            return;
        }
        if (mContentLayout != null) {
            if (isLimit) {
                mContentLayout.setBackgroundResource(R.drawable.im_uikit_bg_send_shan_selector);
                setTxtContentColor(getLimitTextColor());
            } else {
                mContentLayout.setBackgroundResource(R.drawable.im_uikit_bg_send_selector);
                setTxtContentColor(getNormalTextColor());
            }
        }
    }

    protected void setTxtContentColor(int color) {

    }

    public int getNormalTextColor() {
        return getResources().getColor(R.color.im_uikit_view_normal_text_item_color);
    }

    public int getLimitTextColor() {
        return getResources().getColor(R.color.im_uikit_view_limit_text_item_color);
    }

    /**
     * 显示选择框
     */
    protected boolean showPopDialog() {
        List<ImOption> data = new ArrayList<>();
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_TEXT) {
            data.add(mImOptions.get(0));
            data.add(mImOptions.get(1));
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        } else {
            data.add(mImOptions.get(0));
        }
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_AT_TEXT) {
            data.add(mImOptions.get(1));
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        }
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_VIDEO) {
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        }
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_PHOTO) {
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        }
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_FILE) {
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        }
        if (dataSource.getMessageType() == ConstDef.MSG_TYPE_WEB) {
            if (CommonTool.getHasForward() != 1)
                data.add(mImOptions.get(4));
        }
        if (dataSource.getMessageState() == ConstDef.STATE_SEND_FAILED) {
            data.add(mImOptions.get(3));
        } else if (dataSource.getMessageState() != ConstDef.STATE_SENDING) {
            data.add(mImOptions.get(2));
        }

        mPopDialog.addData(data);
        mPopDialog.show(curPosition);
        return true;
    }

    /**
     * 隐藏选择框
     */
    protected void dismissPopDialog() {
        if (mPopDialog != null) {
            mPopDialog.dismiss();
        }
    }

    protected void setMessageVisible(@ConstDef.Visibility int visibility) {
        mStateTv.setVisibility(View.VISIBLE);
    }
}
