package com.xdja.im.uikit.utils.image;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.style.ImageSpan;

import com.xdja.im.core.config.ConstDef;
import com.xdja.im.uikit.utils.ImageCache;

import java.io.IOException;
import java.io.InputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Package: com.xdja.im.uikit.utils.image
 * @Author: xdjaxa
 * @Creation: 2017-04-11 14:06
 * @Version V1.0
 * @Description:
 */
public class BitmapUtils {

    private static final int itemh = 360;
    private static final int itemw = 360;

    /**
     * 根据图片名称获取对应的bitmap
     *
     * @param fileName 名称
     * @param resource 资源实例
     * @return
     * @throws NoSuchFieldException
     * @throws NumberFormatException
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     */
    public static Bitmap getBitmapWithName(String fileName, Resources resource)
            throws NoSuchFieldException,
            IllegalArgumentException, IllegalAccessException {
        InputStream inputStream = null;
        try {
            inputStream = resource.getAssets().open(fileName);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return BitmapFactory.decodeStream(inputStream);
    }

    /**
     * 缩小bitmap
     *
     * @param bitmap
     * @return
     */
    public static Bitmap small(Bitmap bitmap, float scale) {
        Matrix matrix = new Matrix();
        matrix.postScale(scale, scale); // 长和宽放大缩小的比例
        Bitmap resizeBmp = Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(),
                bitmap.getHeight(), matrix, true);
        //memory leak
        if (resizeBmp != bitmap) {
            //bitmap.recycle();
            BitmapUtils.recycleBmp(bitmap);
        }
        return resizeBmp;
    }

    /*
    * 获取压缩后的图片
    */
    public static Bitmap getZoomedDrawable(String filePath, int zoom) {

        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(filePath, options);
        int mWidth = options.outWidth;
        int mHeight = options.outHeight;
        int s = 1;
        while ((mWidth / s > itemw * 2 * zoom)
                || (mHeight / s > itemh * 2 * zoom)) {
            s *= 2;
        }

        options = new BitmapFactory.Options();
        options.inSampleSize = s;
        options.inPreferredConfig = Bitmap.Config.ARGB_8888;
        options.inJustDecodeBounds = false;
        Bitmap bm = BitmapFactory.decodeFile(filePath, options);

        if (bm != null) {
            int h = bm.getHeight();
            int w = bm.getWidth();

            float ft = ((float) w / (float) h);
            float fs = ((float) itemw / (float) itemh);

            int neww = ft >= fs ? itemw * zoom : (int) (itemh * zoom * ft);
            int newh = ft >= fs ? (int) (itemw * zoom / ft) : itemh * zoom;

            float scaleWidth = ((float) neww) / w;
            float scaleHeight = ((float) newh) / h;

            Matrix matrix = new Matrix();
            matrix.postScale(scaleWidth, scaleHeight);
            bm = Bitmap.createBitmap(bm, 0, 0, w, h, matrix, true);
            // System.gc();
            return bm;
        }
        return null;
    }


    /*
    * 获取压缩后的图片
    */
    public static Bitmap getZoomedBitMap(Bitmap bitmap, int zoom) {

        if (bitmap != null) {
            int h = bitmap.getHeight();
            int w = bitmap.getWidth();

            float ft = ((float) w / (float) h);
            float fs = ((float) itemw / (float) itemh);

            int neww = ft >= fs ? itemw * zoom : (int) (itemh * zoom * ft);
            int newh = ft >= fs ? (int) (itemw * zoom / ft) : itemh * zoom;

            float scaleWidth = ((float) neww) / w;
            float scaleHeight = ((float) newh) / h;

            Matrix matrix = new Matrix();
            matrix.postScale(scaleWidth, scaleHeight);
            bitmap = Bitmap.createBitmap(bitmap, 0, 0, w, h, matrix, true);
            // System.gc();
            return bitmap;
        }
        return null;
    }


    public static void recycleBmp(Bitmap bitmap) {
        if (null != bitmap && !bitmap.isRecycled()) {
            bitmap.recycle();
            bitmap = null;
        }
    }

    public static SpannableString formatSpanContent(CharSequence sourceContent, Context context, float smallScall) {
        if (sourceContent == null) {
            return null;
        }
        SpannableString spannableString = null;
        try {
            spannableString = new SpannableString(sourceContent);
            String emoStr;
            Pattern pattern = Pattern.compile("(\\[emoji_[0-9]{3}\\])");
            Matcher matcher = pattern.matcher(sourceContent);
            while (matcher.find()) {
                emoStr = matcher.group(1);
                emoStr = emoStr.substring(1, emoStr.length() - 1);
                ImageCache imageCache = ImageCache.getInstance();
                Bitmap bitmap = null;
                if (smallScall== 0.7f){
                    if (imageCache.smallCashGet(emoStr + "small") != null) {
                        bitmap = imageCache.smallCashGet(emoStr + "small");
                    }else {
                        bitmap = BitmapUtils.getBitmapWithName(emoStr,
                                context.getResources());
                        bitmap = BitmapUtils.small(bitmap, smallScall);
                        imageCache.smallCashPut(emoStr + "small", bitmap);
                    }

                } else if (smallScall== 1.1f) {
                    if (imageCache.smallCashGet(emoStr + "large") != null) {
                        bitmap = imageCache.smallCashGet(emoStr + "large");
                    } else {
                        bitmap = BitmapUtils.getBitmapWithName(emoStr,
                                context.getResources());
                        bitmap = BitmapUtils.small(bitmap, smallScall);
                        imageCache.smallCashPut(emoStr + "large", bitmap);
                    }
                }

                // 缩小表情图片
                // 用ImageSpan指定图片替代文字
                ImageSpan span = new ImageSpan(context, bitmap);
                // 其实写入EditView中的是这个字段“[fac”，表情图片会替代这个字段显示
                spannableString.setSpan(span, matcher.start(), matcher.end(),
                        Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return spannableString;
    }


    public static Bitmap imageCrop(Bitmap source){
        if (source == null) {
            return null;
        }
        int w = source.getWidth();
        int h = source.getHeight();
        int wh = w > h ? h : w;
        int retX = w > h ? (w - h) / 2 : 0;
        int retY = w > h ? 0 : (h - w) / 2;
        Bitmap bmp = Bitmap.createBitmap(source, retX, retY,wh,wh,null,false);
        if (source != null && !source.equals(bmp) && !source.isRecycled()) {
            source.recycle();
            source = null;
        }
        return bmp;
    }

    /**
     * 获取一张图片的高宽比
     */
    public static float getPictureSacle(String path) {
        try {
            BitmapFactory.Options options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            Bitmap bitmap = BitmapFactory.decodeFile(path, options);
            return (float)options.outHeight/(float)options.outWidth;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0f;

    }

}
