package com.xdja.im.core.repository.impl.datasource;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.xdja.im.common.database.helper.IMSettingHelper;
import com.xdja.im.common.database.helper.MXSettingHelper;
import com.xdja.im.common.persistent.PreferencesUtil;
import com.xdja.im.core.model.param.SessionParam;
import com.xdja.im.core.model.param.SettingParam;
import com.xdja.im.core.repository.interf.datasource.DiskDataStore;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.entity.KeyValuePair;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.sys.NetworkUtil;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.core.repository.impl.datasource
 * @Author: xdjaxa
 * @Creation: 2017-04-11 14:02
 * @Version V1.0
 * @Description:
 */
public class DiskDataStoreImpl implements DiskDataStore {

    private PreferencesUtil preferencesUtil;

    @Inject
    public DiskDataStoreImpl(PreferencesUtil preferencesUtil) {
        this.preferencesUtil = preferencesUtil;
    }

    @Override
    public void setSessionTop(String sessionFlag, final boolean isTop, Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String sessionFlag) {
                        boolean ret = MXSettingHelper.setSessionTop(sessionFlag, isTop);
                        if (!ret) {
                            return Observable.error(new Exception("Save session error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void setSessionDisturb(String sessionFlag, final boolean isDisturb, Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String sessionFlag) {
                        boolean ret = MXSettingHelper.setSessionDisturb(sessionFlag, isDisturb);
                        if (!ret) {
                            return Observable.error(new Exception("Save session error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void setReceiveMode(String sessionFlag, final boolean isOpen, Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String sessionFlag) {
                        boolean ret = MXSettingHelper.setReceiveMode(sessionFlag, isOpen);
                        if (!ret) {
                            return Observable.error(new Exception("Save session error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void saveDraftContent(String sessionFlag, final String content, final long draftTime,
                                 Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String sessionFlag) {
                        boolean ret = MXSettingHelper.insertDraft(sessionFlag, content, draftTime);
                        if (!ret) {
                            return Observable.error(new Exception("Save draft error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void clearDraftContent(String sessionFlag, final long draftTime, Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String sessionFlag) {
                        boolean ret = MXSettingHelper.clearDraft(sessionFlag, draftTime);
                        if (!ret) {
                            return Observable.error(new Exception("Clear draft error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void getDraftContent(String sessionFlag, Subscriber<String> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<String>>() {
                    @Override
                    public Observable<String> call(String sessionFlag) {
                        String content = MXSettingHelper.getDraftContent(sessionFlag);
                        return Observable.just(content);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void loadSessionParam(String sessionFlag, Subscriber<SessionParam> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<SessionParam>>() {
                    @Override
                    public Observable<SessionParam> call(String sessionFlag) {
                        SessionParam param = MXSettingHelper.getSessionParam(sessionFlag);
                        if (param == null) {
                            return Observable.error(new Exception("Load session param is null."));
                        }
                        return Observable.just(param);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void loadAllSessionParams(Subscriber<List<SessionParam>> subscriber) {
        Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<List<SessionParam>>>() {
                    @Override
                    public Observable<List<SessionParam>> call(String s) {
                        List<SessionParam> params = MXSettingHelper.getAllSessionParams();
                        return Observable.just(params);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public Observable<List<SessionParam>> saveSettingTopAndNodisturb2Local(final List<SessionParam> configs) {
        return querySessionSettingsAtLocal().map(
                new Func1<List<SessionParam>, List<SessionParam>>() {
                    @Override
                    public List<SessionParam> call(List<SessionParam> localSConfigs) {

                        if (!NetworkUtil.isNetworkConnected(ImUiKit.getInstance().getContext())) {
                            return localSConfigs;
                        }
                        //当从云端获取的数据为空时，查询本地数据，将置顶，免打扰全部设置为false
                        if (configs.size() == 0) {
                            for (SessionParam config : localSConfigs) {
                                config.setDisturb(false);
                                config.setTop(false);
                            }
                            return localSConfigs;
                        } else {
                            if (localSConfigs.size() > 0) {
                                boolean isMatch = false;
                                SessionParam config = null;
                                SessionParam itemConfig;
                                List<SessionParam> addChange = new ArrayList<>();
                                List<SessionParam> addClodChange = new ArrayList<>();
                                if (configs != null) {// TODO: 2017/2/16 确认此处需要不
                                    for (int i = 0; i < configs.size(); i++) {
                                        itemConfig = configs.get(i);
                                        for (int j = 0; j < localSConfigs.size(); j++) {
                                            config = localSConfigs.get(j);
                                            if (config.equals(itemConfig)) {
                                                isMatch = true;
                                                break;
                                            }
                                        }
                                        if (isMatch) {
                                            config.setTop(itemConfig.isTop());
                                            config.setDisturb(itemConfig.isDisturb());
                                            addChange.add(config);
                                            localSConfigs.remove(config);
                                            addClodChange.add(itemConfig);
                                        }
                                        isMatch = false;
                                    }
                                }
                                configs.removeAll(addClodChange);
                                if (configs.size() > 0) {
                                    for (SessionParam clodConfig : configs) {
                                        SessionParam sessionConfig = new SessionParam();
                                        sessionConfig.setDisturb(clodConfig.isDisturb());
                                        sessionConfig.setTop(clodConfig.isTop());
                                        sessionConfig.setSessionFlag(clodConfig.getSessionFlag());
                                        addChange.add(sessionConfig);
                                    }
                                }

                                if (localSConfigs != null) {
                                    for (SessionParam localConfig : localSConfigs) {
                                        localConfig.setDisturb(false);
                                        localConfig.setTop(false);
                                    }
                                }
                                localSConfigs.addAll(addChange);

                                return localSConfigs;
                            } else {
                                return configs;
                            }

                        }

                    }
                }

        ).flatMap(
                new Func1<List<SessionParam>, Observable<List<SessionParam>>>() {
                    @Override
                    public Observable<List<SessionParam>> call(List<SessionParam> configs) {
                        return Observable.from(configs).flatMap(
                                new Func1<SessionParam, Observable<SessionParam>>() {
                                    @Override
                                    public Observable<SessionParam> call(final SessionParam sessionConfig) {
//                                        操作数据库更新到本地
                                        MXSettingHelper.updateSessionParam(sessionConfig);
                                        return Observable.just(sessionConfig);
                                    }
                                }
                        ).toList();
                    }
                }
        );//end
    }


    public Observable<List<SessionParam>> querySessionSettingsAtLocal() {
        return Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<List<SessionParam>>>() {
                    @Override
                    public Observable<List<SessionParam>> call(String s) {
                        List<SessionParam> params = MXSettingHelper.getAllSessionParams();
                        return Observable.just(params);
                    }
                });
    }

    @Override
    public void deleteSessionParam(final String sessionFlag, Subscriber<Boolean> subscriber) {
        Observable.just(sessionFlag)
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String s) {
                        /*boolean ret = MXSettingHelper.deleteOnlyDraft(sessionFlag);
                        if (!ret) {
                            return Observable.error(new Exception("Save session error."));
                        }*/

                        //产品需求：删除会话需要保留置顶和免打扰
                        MXSettingHelper.deleteOnlyDraft(sessionFlag);
                        return Observable.just(true);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public void deleteAllSessionParams(Subscriber<Boolean> subscriber) {
        Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(String s) {
                        boolean ret = MXSettingHelper.deleteAllSessionParams();
                        if (!ret) {
                            return Observable.error(new Exception("Save session error."));
                        }
                        return Observable.just(ret);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }

    @Override
    public Observable<Boolean> saveIntKeyValuePairs(@NonNull KeyValuePair<String, Integer> keyValuePair) {
        if (keyValuePair == null || TextUtils.isEmpty(keyValuePair.getKey())) {
            return Observable.error(new Throwable(ImUiKit.getInstance().getContext().getString(R.string.im_uikit_cache_data_null)));
        }
        return Observable.just(keyValuePair)
                .map(
                        new Func1<KeyValuePair<String, Integer>, Boolean>() {
                            @Override
                            public Boolean call(@NonNull KeyValuePair<String, Integer> keyValuePair) {
                                LogUtil.i("", "存储数据到本地 --- Key : " + keyValuePair.getKey());
                                LogUtil.i("", "\t\t\tValue : " + keyValuePair.getValue());
                                return preferencesUtil
                                        .setPreferenceIntValue(
                                                keyValuePair.getKey(), keyValuePair.getValue()
                                        );
                            }
                        }
                );
    }

    @Override
    public Observable<Boolean> queryIntegerSharePref(@Nullable String key) {
        if (TextUtils.isEmpty(key)) {
            return Observable.error(new Throwable(ImUiKit.getInstance().getContext().getString(R.string.im_uikit_query_key_null)));
        }
        return Observable.just(key)
                .map(
                        new Func1<String, Boolean>() {
                            @Override
                            public Boolean call(String s) {
                                int value = preferencesUtil.gPrefIntValue(s);
                                LogUtil.i("",
                                        "从SharePreference中查询到缓存数据，Key : "
                                                + s + " ,Value : " + value
                                );
                                return value > 0;
                            }
                        }
                ).flatMap(new Func1<Boolean, Observable<Boolean>>() {
                    @Override
                    public Observable<Boolean> call(Boolean aBoolean) {
                        return Observable.just(aBoolean);
                    }
                });
    }

    //***************************************************************
    //***********************以下接口保存IM设置信息**********************
    //***************************************************************

    @Override
    public void loadSettingParams(Subscriber<SettingParam> subscriber) {
        Observable.just("")
                .subscribeOn(Schedulers.newThread())
                .flatMap(new Func1<String, Observable<SettingParam>>() {
                    @Override
                    public Observable<SettingParam> call(String s) {
                        SettingParam settingParam = IMSettingHelper.getSettingParam();
                        if (settingParam == null) {
                            return Observable.error(new Exception("Save session error."));
                        }
                        return Observable.just(settingParam);
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(subscriber);
    }
}
