package com.xdja.im.core.service;

import android.app.NotificationChannel;
import android.app.Service;
import android.content.Intent;
import android.os.Build;
import android.os.IBinder;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.support.v4.app.NotificationCompat;
import com.squareup.otto.Subscribe;
import com.xdja.im.base.di.component.DaggerUserComponent;
import com.xdja.im.base.di.component.RepositoryComponent;
import com.xdja.im.base.di.component.UserComponent;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.database.helper.MXSettingHelper;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.event.IMProxySessionEvent;
import com.xdja.im.core.model.event.IMProxyUpdateSession;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.proxy.IMModuleProxyImpl;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.push.bean.MxPushBean;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.core.repository.interf.datasource.CloudDataStore;
import com.xdja.im.core.repository.interf.datasource.DiskDataStore;
import com.xdja.im.core.utils.DataCache;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import javax.inject.Inject;
import dagger.Lazy;
import rx.Subscriber;

/**
 * @Package: com.xdja.im.core.service
 * @Author: xdjaxa
 * @Creation: 2017-06-26 10:10
 * @Version V1.0
 * @Description:
 */
public class KitService extends Service {

    @Inject
    Lazy<ProxyRepository> proxyRepository;

    @Inject
    Lazy<CloudDataStore> cloudDataStore;

    /**
     * 数据存储
     */
    @Inject
    Lazy<DiskDataStore> diskDataStore;

    private AtomicInteger initIsDone;

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public void onCreate() {
        super.onCreate();

        LogUtil.d("KitServer start...");
        UserComponent mComponent = null;
        RepositoryComponent component = ImUiKit.getInstance().getComponent();
        if (component != null) {
            mComponent = DaggerUserComponent.builder()
                    .repositoryComponent(component)
                    .build();
        }
        if (mComponent == null) {
            LogUtil.e("ERROR: Component instance is null.");
            stopSelf();
            return;
        }
        mComponent.inject(this);
        BusProvider.getInstance().register(this);
        initIsDone = new AtomicInteger(0);

        initProxy();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (getInitState() == 0 && ConstDef.IM_PUSH_STATE.equals(intent.getAction())) {
            //防止PushClientId上报失败
            initMxProxy();
        }
        // start add by lyj 8.0前台服务启动后，5s内需startForground，否则ANR 2020。05.25
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            String channelId = "ImUiKit";
            if (!TextUtils.isEmpty(ImUiKit.getInstance().getChannelId())) {
                channelId = ImUiKit.getInstance().getChannelId();
            }
            String channelName = "ImUiKit";
            if (!TextUtils.isEmpty(ImUiKit.getInstance().getChannelName())) {
                channelName = ImUiKit.getInstance().getChannelName();
            }
            int server_id = 1;
            NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, android.app.NotificationManager.IMPORTANCE_HIGH);
            android.app.NotificationManager notificationManager = (android.app.NotificationManager) getSystemService(NOTIFICATION_SERVICE);
            notificationManager.createNotificationChannel(notificationChannel);
            startForeground(server_id, new NotificationCompat.Builder(this, channelId).build());
        }
        // end
        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        LogUtil.d("KitServer destroy...");

        try {
            BusProvider.getInstance().unregister(this);
        } catch (Exception e) {
            e.printStackTrace();
        }

        releaseProxy();
        DataCache.getInstance().clearAll();
        stopForeground(true);
    }

    public void initProxy() {
        initImProxy();
        initMxProxy();
    }

    /**
     * IM相关服务初始化
     */
    private void initImProxy() {

        LogUtil.d("Start init im proxy.");

        if (proxyRepository == null || proxyRepository.get() == null) {
            LogUtil.e("ERROR: Init im proxy failed. mInstance is null.");
            return;
        }

        proxyRepository.get().initIMProxy().subscribe(new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
                LogUtil.d("initProxy onCompleted");
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("initProxy onError:" + e.getMessage());
            }

            @Override
            public void onNext(Integer integer) {
                if (integer == 0) {
                    LogUtil.d("Init imProxy successfully.");
                } else {
                    LogUtil.d("Init imProxy failed, failCode:" + integer);
                }
            }
        });
    }

    /**
     * Mx数据初始化
     */
    private void initMxProxy() {

        LogUtil.d("Start init mx proxy.");
        setInitIsDone(1);

        if (cloudDataStore == null) {
            LogUtil.e("ERROR: Init mx proxy failed. mInstance is null.");
            return;
        }

        cloudDataStore.get().reportPushClientId(getPackageName(),
                new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        LogUtil.e("Report push client failed." + e.getMessage());
                        setInitIsDone(0);
                    }

                    @Override
                    public void onNext(Boolean b) {
                        LogUtil.d("Report push client successfully.");
                        setInitIsDone(2);
                    }
                });
    }

    public void releaseProxy() {

        LogUtil.d("Start release im proxy.");
        setInitIsDone(0);
        if (proxyRepository == null) {
            LogUtil.e("ERROR: Release im proxy failed. mInstance is null.");
            return;
        }

        proxyRepository.get().releaseIMProxy()
                .subscribe(new Subscriber<Integer>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {

                    }

                    @Override
                    public void onNext(Integer integer) {
                        LogUtil.d("Release im proxy " + integer);
                    }
                });
    }

    /**
     * 获取所有未读消息数
     */
    private void getAllUnReadMessageCount() {
        proxyRepository.get().getAllUnReadMsgCount(new Subscriber<Integer>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("Get unread message count failed. " + e.getMessage());
            }

            @Override
            public void onNext(Integer integer) {
                if (integer > 0) {
                    IMUiKitProxyImpl.getInstance().updateUnReadMessageCount(integer);
                }
            }
        });
    }

    /**
     * 获取当前初始化状态
     *
     * @return 返回结果
     */
    private int getInitState() {
        if (initIsDone != null) {
            return initIsDone.get();
        }
        return 0;
    }

    /**
     * 设置当前初始化状态
     *
     * @param state
     */
    private void setInitIsDone(int state) {
        if (initIsDone != null) {
            initIsDone.set(state);
        }
    }

    //*****************************************************************
    // *                    以下为通知事件处理
    //*****************************************************************
    //IMSDK初始化完成事件
    @Subscribe
    public void onInitFinished(IMProxyEvent.OnInitFinishedEvent event) {
        LogUtil.d("EVENT: onInitFinished");

        //获取所有未读消息数
        getAllUnReadMessageCount();

        IMProxySessionEvent.RefreshSessionListEvent refreshMessageListEvent
                = new IMProxySessionEvent.RefreshSessionListEvent();
        BusProvider.getInstance().post(refreshMessageListEvent);

        IMUiKitProxyImpl.getInstance().onInitFinished();
    }

    //IMSDK初始化失败事件
    @Subscribe
    public void onInitFailed(IMProxyEvent.OnInitFailedEvent event) {

        LogUtil.d("EVENT: onInitFailed");
        LogUtil.e("ERROR: init sdk failed.");
        IMUiKitProxyImpl.getInstance().onInitFailed();
    }

    //新消息接收事件
    @Subscribe
    public void onNewMsgComeEvent(IMProxyMessageEvent.NewMsgComeEvent event) {

        LogUtil.d("EVENT: NewMsgComeEvent");

        //数据信息
        TalkSessionBean sessionBean = event.getListBean();
        List<TalkMessageBean> msgLists = event.getBeans();
        boolean isDisturb = MXSettingHelper.isSessionDisturb(sessionBean.getTalkFlag());
        //第三方通知接口
        boolean isResult = IMUiKitProxyImpl.getInstance().notify(
                sessionBean,
                msgLists.size(),
                isDisturb);
        //调用内部默认接口
        if (!isResult) {
            IMModuleProxyImpl.getInstance().notify(sessionBean, msgLists.size(), isDisturb,msgLists);
        }
    }

    /**
     * 收到置顶或者免打扰的配置信息改变的消息，更新数据库，通知界面刷新
     * @param event
     */
    @Subscribe
    public void onSessionConfigUpDate(IMProxyUpdateSession.UpDateConfigInformation event) {
        if (event == null) return;
        MxPushBean mxPushBean = event.getContent();
        String id = mxPushBean.getMessage();
        String type = mxPushBean.getType();
        if (type.equals(ConstDef.CON_UPDATETOP)) {
            setLocalSessionTop(id, true);
        } else if (type.equals(ConstDef.CON_DELETETOP)) {
            setLocalSessionTop(id, false);
        } else if (type.equals(ConstDef.CON_UPDATENODISTURB)) {
            id = id.replace(":", "_");
            setLocalDisturb(id,true);
        } else if (type.equals(ConstDef.CON_DELETENODISTURB)) {
            id = id.replace(":", "_");
            setLocalDisturb(id,false);
        }
    }

    private void setLocalSessionTop(final String flag, final boolean setting) {

        diskDataStore.get().setSessionTop(flag, setting, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: set session top error." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                if (aBoolean) {
                    IMUiKitProxyImpl.getInstance().upDateConfigUi(ConstDef.MX_TOP,setting,flag);
                    IMProxyUpdateSession.UpDateSessionConfig upDateSessionConfig = new IMProxyUpdateSession.UpDateSessionConfig();
                    upDateSessionConfig.setTalkFlag(flag);
                    upDateSessionConfig.setSetting(setting);
                    upDateSessionConfig.setMessage(ConstDef.MX_TOP);
                    upDateSessionConfig.setSave(false);
                    BusProvider.getInstance().post(upDateSessionConfig);

                }
            }
        });
    }

    public void setLocalDisturb(final String flag, final boolean setting) {
        diskDataStore.get().setSessionDisturb(flag, setting, new Subscriber<Boolean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: set session disturb error." + e.getMessage());
            }

            @Override
            public void onNext(Boolean aBoolean) {
                if (aBoolean) {

                    IMUiKitProxyImpl.getInstance().upDateConfigUi(ConstDef.MX_DISTRUB,setting,flag);
                    IMProxyUpdateSession.UpDateSessionConfig upDateSessionConfig = new IMProxyUpdateSession.UpDateSessionConfig();
                    upDateSessionConfig.setTalkFlag(flag);
                    upDateSessionConfig.setSetting(setting);
                    upDateSessionConfig.setMessage(ConstDef.MX_DISTRUB);
                    upDateSessionConfig.setSave(false);
                    BusProvider.getInstance().post(upDateSessionConfig);
                }
            }
        });
    }
}
