package com.xdja.im.lib.filemanage.ui;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v4.app.Fragment;
import android.view.View;

import com.bumptech.glide.Glide;
import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.core.model.event.IMProxyEvent;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.filemanage.adapter.LocalFileListAdapter;
import com.xdja.im.lib.filemanage.commend.IFileExplorerCommand;
import com.xdja.im.lib.filemanage.commend.IFileExplorerVu;
import com.xdja.im.lib.filemanage.fragment.LastFilePagePresenter;
import com.xdja.im.lib.filemanage.fragment.LocalFileListPresenter;
import com.xdja.im.lib.filemanage.presenter.IMActivityPresenter;
import com.xdja.im.lib.filemanage.util.FileInfoCollection;
import com.xdja.im.lib.filemanage.util.NetworkUtils;
import com.xdja.im.lib.filemanage.util.XToast;
import com.xdja.im.lib.filemanage.view.ViewFileExplorer;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.List;

import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * <p>Author: leiliangliang</p>
 * <p>Date: 2016/11/29 10:40</p>
 * <p>Package: com.xdja.imp.presenter.activity</p>
 * <p>Description: IM文件浏览器</p>
 */
public class FileExplorerPresenter extends IMActivityPresenter<IFileExplorerCommand, IFileExplorerVu>
        implements IFileExplorerCommand {

    /**
     * 最大可发送文件大小
     */
    //private static final long MAX_FILE_SIZE = 30 * 1024 * 1024;
    private static final long MAX_FILE_SIZE = ImUiKit.getInstance().getComponent().configCache().getDevice().getFileSize() * 1024 * 1024;

    //事件总线
    BusProvider busProvider;

    private LocalFileListAdapter mAdapter;

    @Override
    protected void onBindView(Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);

        if (null == userComponent) {
            LogUtil.e("FileExplorerPresenter", "useCaseComponent is null");
            return;
        }
        //初始化注入
        userComponent.inject(this);

        //初始化事件总线
        busProvider = BusProvider.getInstance();
        busProvider.register(this);

        List<Fragment> fragments = new ArrayList<>();
        //最近聊天文件fragment
        fragments.add(new LastFilePagePresenter());
        //本地文件fragment
        fragments.add(new LocalFileListPresenter());

        mAdapter = buildAdapter(fragments);

        getVu().setFragmentAdapter(mAdapter);

        getVu().setCurrentSelectedFileSize(0);
    }

    @NonNull
    @Override
    protected Class<? extends IFileExplorerVu> getVuClass() {
        return ViewFileExplorer.class;
    }

    @NonNull
    @Override
    protected IFileExplorerCommand getCommand() {
        return this;
    }


    private LocalFileListAdapter buildAdapter(final List<Fragment> fragments) {
        return new LocalFileListAdapter(getSupportFragmentManager()) {

            @Override
            public int getCount() {
                return fragments.size();
            }

            @Override
            public Fragment getItem(int position) {
                return fragments.get(position);
            }
        };
    }

    @Override
    public void onLastFileTabClick(View view) {
        getVu().setViewPagerCurrentItem(0);
    }

    @Override
    public void onLocalFileTabClick(View view) {
        getVu().setViewPagerCurrentItem(1);
    }

    @Override
    public void onSendBtnClick(View view) {
        if (NetworkUtils.obtainCurrentNetworkState(this) == NetworkUtils.MOBILE_NETWORK) {
            checkNetStatue();
        } else {
            sendSelectedFiles();
        }
    }

    /**
     * 发送文件之前，监测网络状态
     */
    private void checkNetStatue() {
        checkNetType(new OnNetCheckCallback() {
            @Override
            public void onResult(boolean isCanceled) {
                if (isCanceled) {
                    LogUtil.d("Has canceled send files.");
                    return;
                }
                sendSelectedFiles();
            }

            @Override
            protected String getContent() {
                return String.format(getString(R.string.im_uikit_mobile_data_hint_contnet),
                        getString(R.string.im_uikit_send));
            }
        });
    }

    private void sendSelectedFiles(){
        getVu().setProgressBarVisibility(View.VISIBLE);
        //发送文件之前进行文件大小校验
        Observable.from(FileInfoCollection.getInstance().getAllSelectFiles())
                .flatMap(new Func1<FileItem, Observable<FileItem>>() {
                    @Override
                    public Observable<FileItem> call(FileItem localFileInfo) {
                        long fileSize = localFileInfo.getFileSize();
                        if (fileSize > MAX_FILE_SIZE || fileSize < 0) {
                            return Observable.error(new IllegalArgumentException("file is too larger!"));
                        }
                        return Observable.just(localFileInfo);
                    }
                })
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<FileItem>() {
                    @Override
                    public void onCompleted() {
                        getVu().setProgressBarVisibility(View.GONE);
                    }

                    @SuppressLint("StringFormatInvalid")
                    @Override
                    public void onError(Throwable e) {
                        getVu().setProgressBarVisibility(View.GONE);
                        //new XToast(FileExplorerPresenter.this).display(R.string.im_uikit_file_is_too_larger);
                        new XToast(FileExplorerPresenter.this).display(String.format(getString(R.string.im_uikit_file_is_too_larger)
                                ,(ImUiKit.getInstance().getComponent().configCache().getDevice().getFileSize()+"MB")));
                    }

                    @Override
                    public void onNext(FileItem localFileInfo) {
                        getVu().setProgressBarVisibility(View.GONE);
                        setResult();
                    }
        });

    }

    /**
     * 返回发送文件
     */
    private void setResult() {
        //发送文件
        Intent intent = new Intent();
        //数据绑定
        Bundle bundle = new Bundle();
        ArrayList<FileItem> bundleList = new ArrayList<>();
        bundleList.addAll(FileInfoCollection.getInstance().getAllSelectFiles());
        bundle.putParcelableArrayList(Constant.TAG_FILE_SELECT, bundleList);
        //添加数据到Intent
        intent.putExtras(bundle);
        setResult(Constant.RESULT_CODE_FILE, intent);
        finish();
    }

    /**
     * 选择文件相关提示信息设置更新
     *
     * @param event
     */
    @Subscribe
    public void onRefreashFileSelectStatus(IMProxyEvent.FileSelectedEvent event) {
        //更新已选文件大小
        getVu().setCurrentSelectedFileSize(FileInfoCollection.getInstance().getSelectedFileSize());
        //更新已选文件个数
        getVu().setCurrentSelectedFileCount(FileInfoCollection.getInstance().getSelectedFileCount());
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        if (busProvider != null) {
            busProvider.unregister(this);
        }

        Glide.get(this).clearMemory();
        //ImageLoader.clearCacheWithCheck();
        //清除缓存相关
        FileInfoCollection.getInstance().clearCache();
    }
}
