package com.xdja.im.lib.historyfile.util;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RawRes;
import android.text.TextUtils;

import com.xdja.im.common.cache.interf.UserCache;
import com.xdja.im.uikit.ImUiKit;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.imsdk.constant.ImSdkFileConstant;
import com.xdja.imsdk.util.HmacSHA1Util;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;

import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;

/**
 * Created by ALH on 2018/5/4.
 * 用于控制文件上传下载的控制器
 */

public final class FilesController {
    /**
     * 文件名最大长度
     */
    private static final int MAX_FILE_NAME_LENGTH = 128;

    /**
     * 网络连接超时时间
     */
    public static final int TIME_OUT_UNIT = 30 * 1000;


    public static final String HEAD_TICKET = "ticket";

    private FilesController() {

    }

    private static class FilesControllerHolder {
        private static FilesController sInstance = new FilesController();
    }

    public static synchronized final FilesController getInstance() {
        return FilesControllerHolder.sInstance;
    }

    /**
     * 获取文件后缀名
     *
     * @param fileName 文件名
     * @return 文件后缀名，null - 如果文件不包含后缀名
     */
    @Nullable
    public static String getSuffix(String fileName) {
        int index = fileName.lastIndexOf('.');
        if (index == -1) {
            return null;
        }
        return fileName.substring(index);
    }

    /**
     * 该方法和SignInterceptor2拦截器的功能一样.
     * 提供给其他有需要的模块调用,如果只是想上传文件,可以直接调用uploadFile
     *
     * @param type   签名类型
     * @param fileId 文件标识
     * @return 签名结果（0：签名，1：过期时间戳，2：用户ID）
     * @throws JSONException
     * @throws IOException
     */
    public String[] requestSign(UserCache userCache, String type, String fileId) throws JSONException, IOException {
        String url = userCache.get().getFastDfsAddr();
        if (!url.endsWith("/")) url = url + File.separator;
        Request request = new Request.Builder()
                .url(url + OkHttpConstants.URL_TIME)
                .build();
        // Modified by guting on 2018/5/17: 返回解析之后的签名信息
        OkHttpClient okHttpClient = getOkHttpClients(userCache);
        Response response = okHttpClient.newCall(request)
                .execute();
        String[] signResults = null;
        if (response.isSuccessful()) {
            signResults = new String[3];
            try {
                long timestramp = 0;
                String result = response.body().string();
                LogUtil.d("result:  " + result);
                JSONObject jsonObject = new JSONObject(result);
                if (jsonObject.has("timestamp")) {
                    timestramp = jsonObject.getLong("timestamp");
                }
                timestramp = timestramp + 20 * 60 * 1000;
                String data = "";
                if (type.equals(ImSdkFileConstant.SIGN_TYPE_UPLOAD)) {//userid
                    data = userCache.get().getFastUserId() + timestramp;
                } else if (type.equals(ImSdkFileConstant.SIGN_TYPE_INFO)) {//  /info/+fileId
                    data = "/info/" + fileId + timestramp;
                } else if (type.equals(ImSdkFileConstant.SIGN_TYPE_DOWNLOAD)) {// fileId
                    data = fileId + timestramp;
                } else if (type.equals(ImSdkFileConstant.SIGN_TYPE_DELETE)) {// /delete/+fileId
                    data = "/delete/" + fileId + timestramp;
                }
                String sign = HmacSHA1Util.hamcsha1(data, userCache.get().getFastUserSecret());
                signResults[0] = sign;
                signResults[1] = String.valueOf(timestramp);
                signResults[2] = userCache.get().getFastUserId();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return signResults;
    }


    private OkHttpClient getOkHttpClients(final UserCache userCache) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();

        if (userCache.get() != null && !TextUtils.isEmpty(userCache.get().getFileKeystorePsd()) && userCache.get().getFileKeystoreId() > 0) {
            builder.sslSocketFactory(makeSSLSocketFactory(userCache));
        }

        builder.connectTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .readTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .writeTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .hostnameVerifier(new HostnameVerifier() {
                    @Override
                    public boolean verify(String hostname, SSLSession session) {
                        return true;
                    }
                })
                .addInterceptor(
                        new Interceptor() {
                            @Override
                            public Response intercept(Chain chain) throws IOException {

                                Request request = chain.request();
                                Request.Builder builder = request.newBuilder();
                                builder.header(HEAD_TICKET, userCache.get().getTicket());
                                return chain.proceed(builder.build());
                            }
                        }
                )
                .addInterceptor(new HttpLoggingInterceptor(new HttpLoggingInterceptor.Logger() {
                    @Override
                    public void log(String message) {
                        LogUtil.d(message);
                    }
                }).setLevel(HttpLoggingInterceptor.Level.BODY));
        return builder.build();
    }


    /**
     * 获取SSLSocketFactory
     *
     * @return
     */
    private SSLSocketFactory makeSSLSocketFactory(final UserCache userCache) {
        if (userCache == null || userCache.get() == null || userCache.get().getFileKeystoreId() == 0) {
            return null;
        }
        SSLContext sslContext = getSSLContext(
                userCache, readKeyStore(userCache, userCache.get().getFileKeystoreId(), ImUiKit.getInstance().getContext()));
        if (sslContext == null) {
            return null;
        }
        return sslContext.getSocketFactory();
    }

    /**
     * 读取本地证书
     *
     * @param res 证书资源ID
     * @return 获取到的证书
     */
    @Nullable
    private KeyStore readKeyStore(final UserCache userCache, @RawRes int res, @NonNull Context context) {
        InputStream inputStream = null;
        KeyStore keyStore = null;
        try {
            if (TextUtils.isEmpty(userCache.get().getFileKeystorePsd())) {
                return null;
            }
            keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            inputStream = context.getResources().openRawResource(res);
            keyStore.load(inputStream, userCache.get().getFileKeystorePsd().toCharArray());
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return keyStore;
    }

    /**
     * 获取SSL连接上下文
     *
     * @param keyStore
     * @return
     */
    @Nullable
    private SSLContext getSSLContext(UserCache userCache, @Nullable KeyStore keyStore) {
        if (keyStore == null) {
            return null;
        }
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("TLS");

            TrustManagerFactory trustManagerFactory =
                    TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(keyStore);

            KeyManagerFactory keyManagerFactory =
                    KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keyStore, userCache.get().getFileKeystorePsd().toCharArray());

            sslContext.init(keyManagerFactory.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());
        } catch (NoSuchAlgorithmException
                | KeyStoreException
                | KeyManagementException
                | UnrecoverableKeyException e) {
            e.printStackTrace();
        }
        return sslContext;
    }
}
