package com.xdja.im.uikit;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.xdja.im.base.frame.view.AdapterBaseView;
import com.xdja.im.core.callback.AccountCallback;
import com.xdja.im.core.callback.ClearSessionDataCallback;
import com.xdja.im.core.callback.CommonCallback;
import com.xdja.im.core.callback.CustomComponentCallBack;
import com.xdja.im.core.callback.NotificationCallback;
import com.xdja.im.core.callback.NotificationChannelCallback;
import com.xdja.im.core.callback.OnAudioRecordCallback;
import com.xdja.im.core.callback.OnChatAt2GroupCallBack;
import com.xdja.im.core.callback.OnChatMeetCallBack;
import com.xdja.im.core.callback.OnChatSettingCallback;
import com.xdja.im.core.callback.OnChatUserInfoCallback;
import com.xdja.im.core.callback.OnEncryptListener;
import com.xdja.im.core.callback.OnImInitStateCallback;
import com.xdja.im.core.callback.OnOpenChatPageCallback;
import com.xdja.im.core.callback.OnShareOverCallback;
import com.xdja.im.core.callback.OnUserCanAccessToOptionCallBack;
import com.xdja.im.core.callback.SessionDeleteCallBack;
import com.xdja.im.core.callback.ShareSelectCallback;
import com.xdja.im.core.callback.UnReadMessageCallback;
import com.xdja.im.core.model.account.AccountAction;
import com.xdja.im.core.model.account.ActionDesc;
import com.xdja.im.core.model.account.GroupAction;
import com.xdja.im.core.model.message.MeetInfo;
import com.xdja.im.core.model.message.TalkCustomBean;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkNotifyBean;
import com.xdja.im.core.model.param.SessionParam;
import com.xdja.im.lib.share.model.ShareBean;
import com.xdja.im.uikit.ui.presenter.ChatListAdapterPresenter;

import java.util.List;

/**
 * @Package: com.xdja.im.uikit
 * @Author: xdjaxa
 * @Creation: 2017-04-17 15:25
 * @Version V1.0
 * @Description: UIKit对外提供接口类
 */
public interface IClient {

    /**
     * uikit初始化
     *
     * @param context 上下文
     * @param options 初始化参数
     */
//    void init(Context context, @Nullable UiKitOptions options);

    /**
     * sdk初始化
     *
     * @param context 全局上下文环境
     * @param params  初始化信息，非空 {@link UiKitParams}
     * @param options sdk配置信息{@link UiKitOptions},
     *                如果为空或者未设置，则使用默认配置项
     */
    void init(Context context, @NonNull UiKitParams params, @Nullable UiKitOptions options);

    /**
     * imsdk退出调用，释放相关资源信息
     *
     * @param flag 说明<br>
     *             flag = 1: 应用关闭，释放所有资源,不再接收消息 <br>
     *             flag = 0: 应用退出前台，ImSdkService后台运行，继续接收消息
     */
    void release(int flag);

    /**
     * 获取sdk配置信息
     *
     * @return 当前im模块的配置信息
     */
    UiKitOptions getOptions();

    /**
     * 发送一条文本通知消息到聊天界面
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param bean        消息实体
     */
    void sendNotifyTextMessage(String sessionId, int sessionType, TalkNotifyBean bean);

    /**
     * 发送分享联系人信息
     */
    void sendShareContact(int type, List<ShareBean> list);

    /**
     * 发送自定义消息到聊天界面
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param bean        消息实体
     */
    void sendCustomTextMessage(String sessionId, int sessionType, TalkCustomBean bean);

    /**
     * 发送自定义消息到聊天界面
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param jsonContent 消息实体
     */
    void sendCustomMessage(String sessionId, int sessionType, String jsonContent);

    /**
     * 清除聊天会话中所有消息
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     */
    void clearSessionData(String sessionId, int sessionType);

    void clearSessionData(String sessionId, int sessionType, ClearSessionDataCallback callback);

    /**
     * 清空消息列表
     */
    void clearAllSessionData();

    /**
     * 清除缓存数据
     */
    void clearAllCacheData();

    /**
     * 删除三方自定义消息（消息助手）
     *
     * @param bean 消息实体
     */
    void deleteThirdMessage(TalkMessageBean bean);

    /**
     * 是否打开听筒模式
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param isOpen      {@code true} 打开听筒 {@code false} 关闭听筒
     */
    void setReceiveMode(String sessionId, int sessionType, boolean isOpen);

    /**
     * 会话免打扰模式<br/>
     * 聊天设置界面调用
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param isDisturb   {code{true}}：设置免打扰； {code{false}}：取消免打扰
     */
    void setNoDisturbMode(String sessionId, int sessionType, boolean isDisturb);

    /**
     * 会话置顶功能设置
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @param isTop       {code{true}}：设置置顶； {code{false}}：取消置顶
     */
    void setSessionTop(String sessionId, int sessionType, boolean isTop);

    /**
     * 获取回话的配置信息功能
     *
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     * @return
     */
    SessionParam getSessionParam(String sessionId, int sessionType);

    /**
     * 联系人变更通知接口
     *
     * @param account 联系人账号
     * @param action  联系人变更动作{@link AccountAction}
     */
    void notifyAccountChanged(String account, AccountAction action);

    /**
     * 群组变更通知接口
     *
     * @param groupId    群组ID
     * @param action     联系人变更动作{@link GroupAction}
     * @param actionDesc 描述信息{@link ActionDesc}
     */
    void notifyGroupChanged(String groupId, GroupAction action, ActionDesc actionDesc);

    /**
     * 渠道id回调
     * @param callback
     */
    void setNotificationChannelCallback(NotificationChannelCallback callback);

    /**
     * 注册联系人回调接口，用于读取联系人信息
     *
     * @param callback 联系人回调接口
     */
    void registerAccountCallback(AccountCallback callback);

    /**
     * 此接口用于添加自定义聊天消息
     * 页面展示类（Item）需继承 CustomBaseChatItem
     * 数据类型为json格式存储于TalkMessageBean对象的content字段，此字段的json值需可以解析为CustomItemDataProvider类
     */
    void registerAndAddCustomChatListItem(int type, Class<? extends AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean>> customItem);

    /**
     * 注册通知消息回调接口，用于提示新消息通知
     *
     * @param callback 通知回调接口
     */
    void registerNotificationCallback(NotificationCallback callback);

    /**
     * 注册更新未读消息回调接口，提示未读消息数量
     *
     * @param callback 未读消息数量更新通知回调
     */
    void registerUnReadMessageCallback(UnReadMessageCallback callback);

    /**
     * 注册自定义组件点击回调
     *
     * @param callBack 返回选择的组件信息
     */
    void registerCustomComponentCallBack(CustomComponentCallBack callBack);

    /**
     * 注册启动会话设置界面回调
     *
     * @param callback
     */
    void registerOpenChatSettingCallback(OnChatSettingCallback callback);

    /**
     * 打开用户信息界面
     *
     * @param callback
     */
    void registerOpenChatUserInfoCallback(OnChatUserInfoCallback callback);

    /**
     * 打开聊天页面回调
     * 用于传参控制底部菜单显隐
     *
     * @param callback 回调值中id隐藏
     */
    void registerOnOpenChatPageCallback(OnOpenChatPageCallback callback);

    /**
     * 注册需要通过上层调用的通用接口
     *
     * @param callback 未读消息数量更新通知回调
     */
    void registerCommonCallback(CommonCallback callback);

    /**
     * 加入会议回掉
     */
    void registerAddChtMeeting(OnChatMeetCallBack callBack);

    /**
     * 分享操作回掉
     */
    void registerShareSelectCallback(ShareSelectCallback callback);

    /**
     * 分享完成操作回掉
     */
    void registerOnShareOverCallback(OnShareOverCallback callback);

    /**
     * 打开聊天文件
     *
     * @param ifRecall    是否又返回值  ifRecall 现在传false 此参数为预留参数，为后续准备
     * @param sessionId   如果是群组，则是群ID；如果是单聊，则是会话ID
     * @param sessionType 0：群组 1：单聊
     */
    void openHistoryChatFile(Activity activity, String sessionId, int sessionType, boolean ifRecall);

    /**
     * 自定义加密回调
     *
     * @param onEncryptListener
     */
    void registerEncryptListener(OnEncryptListener onEncryptListener);

    /**
     * 通知会议事件
     *
     * @param meetInfo 是否显示提示信息等
     */
    void notifyMeet(MeetInfo meetInfo);

    /**
     * 首次登录数量初始化
     */
    void registerInitUnReadMsgCallBack();

    /**
     * @打开群成员界面
     */
    void registerAtOpenGroupCallBack(OnChatAt2GroupCallBack OnChatAt2GroupCallBack);


    /**
     * 删除会话列表会话回调
     *
     * @param sessionDeleteCallBack
     */
    void registerSessionDeleteCallBack(SessionDeleteCallBack sessionDeleteCallBack);

    /**
     * @param groupId
     * @param account
     * @param nickName
     */
    void registerAtMembsItemClickCallBack(String groupId, String account, String nickName);

    /**
     * @param activity activity
     * @param content  第三方转发内容
     */
    void forwardThirdContent(Activity activity, String content);

    /**
     * 清除账户信息缓存，一般用于更新用户信息后调用，保证imuikit中用户信息的及时更新
     *
     * @param account at+账号
     */
    void removeAccountCache(String account);

    /**
     * 用户是否可以进入底部菜单
     *
     * @param callBack 回调接口
     */
    void registerUserAccessToOption(OnUserCanAccessToOptionCallBack callBack);

    /**
     * 注册回调im初始化状态
     *
     * @param callback
     */
    void registerOnImInitStateCallback(OnImInitStateCallback callback);



    /**
     * 注册回调录音按钮监听
     *
     * @param callback
     */
    void registerOnAudioRecordCallback(OnAudioRecordCallback callback);

}
